package org.jboss.cache.loader;

import org.jboss.cache.TreeCache;
import org.jboss.cache.Fqn;
import org.jboss.cache.Node;
import org.jboss.cache.Modification;
import org.jboss.logging.Logger;

import java.util.*;

/**
 * CacheLoader implementation which delegates to another TreeCache. This allows to stack caches on top of each
 * other, allowing for hierarchical cache levels. For example, first level cache delegates to a second level cache,
 * which delegates to a persistent cache. Sample code:
 * <pre>
 * TreeCache firstLevel=new TreeCache();
 * TreeCache secondLevel=new TreeCache();
 * DelegatingCacheLoader l=new DelegatingCacheLoader(secondLevel);
 * l.setCache(firstLevel);
 * firstLevel.setCacheLoader(l);
 * secondLevel.start();
 * firstLevel.start();
 * </pre>
 * @author Bela Ban
 * @version $Id: DelegatingCacheLoader.java,v 1.1.2.2 2004/12/30 17:08:27 starksm Exp $
 */
public class DelegatingCacheLoader implements CacheLoader {
   TreeCache delegate=null;
   Logger    log=Logger.getLogger(getClass());
     /** HashMap<Object,List<Modification>>. List of open transactions. Note that this is purely transient, as
    * we don't use a log, recovery is not available */
   HashMap   transactions=new HashMap();

   public DelegatingCacheLoader(TreeCache delegate) {
      this.delegate=delegate;
   }

   public void setConfig(Properties props) {
      throw new UnsupportedOperationException();
   }

   public void setCache(TreeCache c) {
      ;
   }

   public Set getChildrenNames(Fqn fqn) throws Exception {
      Set retval=delegate.getChildrenNames(fqn);
      return retval == null? null : (retval.size() == 0? null : retval);
   }

   public Object get(Fqn name, Object key) throws Exception {
      return delegate.get(name, key);
   }

   public Map get(Fqn name) throws Exception {
      Map attributes=null;
      Node n=delegate.get(name);
      if(n == null || (attributes=n.getData()) == null || attributes.size() == 0)
         return null;
      else
         return new HashMap(attributes);
   }

   public boolean exists(Fqn name) throws Exception {
      return delegate.exists(name);
   }

   public Object put(Fqn name, Object key, Object value) throws Exception {
      return delegate.put(name, key, value);
   }

   public void put(Fqn name, Map attributes) throws Exception {
      delegate.put(name, attributes);
   }


   public void put(Fqn fqn, Map attributes, boolean erase) throws Exception {
      if(erase)
         removeData(fqn);
      put(fqn, attributes);
   }

   public void put(List modifications) throws Exception {
      if(modifications == null) return;
      for(Iterator it=modifications.iterator(); it.hasNext();) {
         Modification m=(Modification)it.next();
         switch(m.getType()) {
            case Modification.PUT_DATA:
               put(m.getFqn(), m.getData());
               break;
            case Modification.PUT_DATA_ERASE:
               put(m.getFqn(), m.getData(), true);
               break;
            case Modification.PUT_KEY_VALUE:
               put(m.getFqn(), m.getKey(), m.getValue());
               break;
            case Modification.REMOVE_DATA:
               removeData(m.getFqn());
               break;
            case Modification.REMOVE_KEY_VALUE:
               remove(m.getFqn(), m.getKey());
               break;
            case Modification.REMOVE_NODE:
               remove(m.getFqn());
               break;
            default:
               log.error("modification type " + m.getType() + " not known");
               break;
         }
      }
   }

   public Object remove(Fqn name, Object key) throws Exception {
      return delegate.remove(name, key);
   }

   public void remove(Fqn name) throws Exception {
      delegate.remove(name);
   }

   public void removeData(Fqn name) throws Exception {
      delegate.removeData(name);
   }

   public void prepare(Object tx, List modifications, boolean one_phase) throws Exception {
      if(one_phase)
         put(modifications);
      else
         transactions.put(tx, modifications);
   }

   public void commit(Object tx) throws Exception {
      List modifications=(List)transactions.get(tx);
      if(modifications == null)
         throw new Exception("transaction " + tx + " not found in transaction table");
      put(modifications);
   }

   public void rollback(Object tx) {
      transactions.remove(tx);
   }

   public byte[] loadEntireState() throws Exception {
      return delegate.getMessageListener().getState();
   }

   public void storeEntireState(byte[] state) throws Exception {
      delegate.getMessageListener().setState(state);
   }


   public void create() throws Exception {
   }

   public void start() throws Exception {
   }

   public void stop() {
   }

   public void destroy() {
   }
}
