// $Id: Cache.java,v 1.1 2003/10/30 01:21:19 bwang00 Exp $

/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.cache;


import javax.transaction.xa.XAResource;
import java.util.Collection;
import java.util.Map;
import java.util.Set;


/**
 * Cache similar to java.util.Map. Entries are key/value pairs. Operations
 * include adding key/value pairs, removal, removing all entries, getting a
 * value associated with a key and so on.<br/> A cache typically has a bounded
 * size; entries will be removed according to an eviction policy (e.g. oldest-
 * first) when we need to make space for more entries. Cache entries can also be
 * transient, persistent (stored in a DB), or replicated to other caches,
 * running in different processes.
 * <br/> A cache is configured through {@link #configure(Object)}, which
 * determines for example the mode for the default entries (e.g. transient,
 * persistent, replicated) and the properties for cache eviction (e.g. oldest-
 * first, LRU).
 * <br/>
 * The current cache is designed with replication in mind. There is a cluster of
 * caches, and changes to <em>one</em> cache will be propagated to <em>all</em>
 * caches in the cluster. The way an update is to be replicated can be chosen
 * per method, and the following update modes exist (in order of cost):
 * <ol>
 * <li>Asynchronous (non-blocking) updates. An update is multicast to all
 * caches in the cluster. Returns immediately.
 * <li>Synchronous (blocking) updates (optionally with a timeout). Same as
 * above, but we wait for all responses, or a timeout, whichever occurs first.
 * <li>Synchronous (blocking) updates with locking. Same as above, but
 * aditionally we can acquire locks on the map to ensure that updates are
 * transactionally serialized. In this mode, we can start a transaction,
 * associate it with the current thread (thread-local storage) and multicast
 * multiple updates and then commit or abort the transaction.
 * </ol>
 * The methods of java.util.Map which modify the map (put(), putAll(), remove()
 * and clear()) have been overridden: they behave according to the default
 * policy (defined by {@link #configure(Object)}). E.g. when the default policy
 * disables replication, then put() will not generate any replication
 * multicast. If replication is enabled, each put(), clear(), putAll() and
 * remove() will generate a replication multicast.<br/>
 * Regardless of the default policy, there are put(), clear(), remove() and
 * putAll() methods, which have additional parameters, and can be used to
 * override the default behavior. E.g. when the default policy disables
 * replication, {@link #put(Object,Object,Options)} will override it and
 * replicate the entry. Also, when the default replication mode is asynchronous,
 * the above method can be called to be synchronously executed.
 * <br/>
 * Note that when an entry is to be replicated, its key and value (unless null)
 * have to be serializable. If this is not the case, the entry will be marked as
 * transient (with an error message in the log). This applies to all methods.
 * <br/>
 * Example (error handling omitted):
 * <pre>
 * Options opts;
 * Cache    cache; // get reference to cache
 *
 * // turn replication on, use asynchronous replication by default, cache group is "demo"
 * cache.configure("file:/home/bela/async-cache.xml");
 *
 * // 1. Add an entry. Will be replicated asynchronously to all caches in the cluster
 * //    (call returns immediately)
 * cache.put("name", "Bela Ban");
 *
 * // 2. Add a transient entry. Although replication is on by default,
 * //    this entry won't be replicated
 * cache.put("age", new Integer(37), new Options(Options.TRANSIENT));
 *
 * // 3. Now it will get replicated
 * cache.setTransience("age", false); // transient = false
 *
 * // 4. Add an entry synchronously (default is asynchronous). Wait for all responses
 * //    or 2 seconds (whichever is first) until the call returns.
 * opts=new Options(Options.REPL_SYNC, 2000);
 * cache.put("city", "San Jose, CA", opts);
 *
 * // 5. Use a transaction and locking to synchronously add multiple items
 *
 * // Start a transaction (implies locking to be used)
 * cache.begin(Xid.READ_COMMITTED);
 *
 * // Add an entry asynchronously (no locking). This item will not be associated
 * // with the current transaction
 * cache.put("item1", new Integer(1));
 *
 * // Add an entry with locking. Wait 7 seconds max for the call to return,
 * // wait 5 secs max to acquire the lock, hold it for 10 secs max (until
 * // it is released automatically) and don't commit the transaction
 * opts=new Options(Options.REPL_SYNC, 7000, 5000, 10000, false);
 * cache.put("item2", opts);
 *
 * // Remove an entry from the cache. Wait 1 sec for all responses,
 * // try to fetch the lock (if not yet held) for 5 seconds, hold it for
 * // a maximum of 10 secs and don't commit the transaction
 * opts=new Options(Options.REPL_SYNC, 1000, 5000, 10000, false);
 * cache. remove("name", opts);
 *
 * // Now commit the transaction. This will add "item2" (*NOT* "item1",
 * // which has already been added) and remove "name");
 * cache.commit();
 *
 * </pre>
 * todo Check whether we should make transactions explicit; ie. instead of
 * associating them with the current thread, we should explicitly specify Xids
 * in begin(), commit() and rollback().
 * @author  <a href="mailto:bela@jboss.org">Bela Ban</a>.
 * @version $Revision: 1.1 $
 *
 * <p><b>Revisions:</b>
 * <p>Dec 27 2002 Bela Ban: first implementation
 */
public interface Cache extends XAResource {

    /**
     * Adds a cache listener. The same listener will not be registered twice.
     * @param l The CacheListener
     */
    void addListener(CacheListener l);

    /**
     * Removes a cache listener
     * @param l The CacheListener
     */
    void removeListener(CacheListener l);

    /**
     * Sets the default properties for this cache implementation // todo: find a better type (URL ?)
     */
    void configure(Object properties);


    /**
     * Returns the number of entries in this map.
     * @return the number of entries in this map.
     */
    int size();

    /**
     * Returns <tt>true</tt> if this map contains no key-value mappings.
     * @return <tt>true</tt> if this map contains no key-value mappings.
     */
    boolean isEmpty();

    /**
     * Returns <tt>true</tt> if this map contains a mapping for the specified
     * key
     * @param key key whose presence in this map is to be tested.
     * @return <tt>true</tt> if this map contains a mapping for the specified key.
     */
    boolean containsKey(Object key);


    /**
     * Returns the value to which this map maps the specified key.  Returns
     * <tt>null</tt> if the map contains no mapping for this key.  A return
     * value of <tt>null</tt> does not <i>necessarily</i> indicate that the
     * map contains no mapping for the key; it's also possible that the map
     * explicitly maps the key to <tt>null</tt>.  The <tt>containsKey</tt>
     * operation may be used to distinguish these two cases.
     * @param key key whose associated value is to be returned.
     * @return the value to which this map maps the specified key, or
     *	       <tt>null</tt> if the map contains no mapping for this key.
     * @see #containsKey(Object)
     */
    Object get(Object key);


    /**
     * Returns a set view of the keys contained in this map. The set is
     * <em>not</em> backed by the map, so changes to the map are <em>not</em>
     * reflected in the set, and vice-versa
     * @return a set view of the keys contained in this map.
     */
    Set keySet();

    /**
     * Returns a collection view of the values contained in this map.  The
     * collection is <em>not</em>backed by the map, so changes to the map
     * are <em>not</em>reflected in the collection, and vice-versa.
     * @return a collection view of the values contained in this map.
     */
    Collection values();

    /**
     * Returns a set view of the mappings contained in this map.  Each element
     * in the returned set is a {@link Map.Entry}.  The set is <em>not</em>
     * backed by the map, so changes to the map are <em>not</em> reflected
     * in the set, and vice-versa.
     * @return a set view of the mappings contained in this map.
     */
    Set entrySet();


    /**
     * Adds an entry to the cache according to the default policy.
     * @param key The key to be set. Can be null.
     * @param value The value to be set. Can be null.
     * @return Object The previous value associated with the given key, or null if none was associated
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    Object put(Object key, Object value) throws CacheException, TimeoutException, LockingException;

    /**
     * Adds an entry to the cache, overriding the default policy.
     * @param key The key to be set. Can be null.
     * @param value The value to be set. Can be null.
     * @param update_options Further options that can be set per update. These
     * will override the default cache options
     * @param update_options Update-specific options which override the
     * cache defaults for this update
     * @return Object The previous value associated with the given key, or null if none was associated
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    Object put(Object key, Object value, Options update_options) throws CacheException, TimeoutException, LockingException;


    /**
     * Adds all values from a map to this map. Existing entries will be
     * overwritten. Updates will be handled according to the default policy.
     * @param m The map
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    void putAll(Map m) throws CacheException, TimeoutException, LockingException;

    /**
     * Adds all values from a map to this map. Existing entries will be
     * overwritten. Updates will be handled according to the default policy.
     * @param m The map
     * @param update_options Update-specific options which override the
     * cache defaults for this update
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    void putAll(Map m, Options update_options) throws CacheException, TimeoutException, LockingException;


    /**
     * Removes an entry from the cache according to the default policy. If the key cannot be found, this
     * is a null operation.
     * @param key The key to the entry to be removed.
     * @return Object The previous value associated with the given key, or null
     * if none was associated
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    Object remove(Object key) throws CacheException, TimeoutException, LockingException;

    /**
     * Removes an entry from the cache according to the default policy. If the key cannot be found, this
     * is a null operation.
     * @param key The key to the entry to be removed.
     * @param update_options Update-specific options which override the
     * cache defaults for this update
     * @return Object The previous value associated with the given key, or null
     * if none was associated
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    Object remove(Object key, Options update_options) throws CacheException, TimeoutException, LockingException;


    /**
     * Removes all entries from the map. Replication takes place acording to the
     * default policy.
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    void clear() throws CacheException, TimeoutException, LockingException;


    /**
     * Removes all entries from the map. Replication takes place acording to the
     * default policy.
     * @param update_options Update-specific options which override the
     * cache defaults for this update
     * @exception LockingException Thrown when one or more of the members failed
     * acquiring the lock within <code>lock_acquisition_timeout</code>
     * milliseconds
     * @exception TimeoutException Thrown when one or more of the members didn't
     * send a response. LockingExceptions take precedence over
     * TimeoutExceptions, e.g. if we have both locking and timeout exceptions, a
     * LockingException will be thrown.
     */
    void clear(Options update_options) throws CacheException, TimeoutException, LockingException;


    /**
     * Changes an entry from transient to non-transient and vice versa. If an
     * entry has been transient up to now, and becomes non-transient, it's
     * value is replicated to all caches in the cluster. From then on, all
     * changes will be replicated. If an entry has been non-transient up to now,
     * and becomes transient, all further modifications to that entry will not
     * be replicated.<br/>
     * If the key or value of the entry is not serializable, this method is a
     * null operation (an error will be logged).
     * @param key The key of the entry
     * @param transient_entry Whether or not to make the entry transient. Note
     * that setting an entry to the mode it has been in so far is a null
     * operation.
     * @throws KeyNotExistsException Thrown if <code>key</code> is not present
     * in the cache
     */
    void setTransience(Object key, boolean transient_entry) throws KeyNotExistsException;

    /**
     * Returns the mode of an entry (whether it is transient or not).
     * @param key The key denoting the object
     * @return boolean Whether or not the entry is transient
     * @throws KeyNotExistsException Thrown if <code>key</code> is not present
     * in the cache
     */
    boolean getTransience(Object key) throws KeyNotExistsException;
}
