/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.hibernate.cache;

import net.sf.hibernate.cache.Cache;
import net.sf.hibernate.cache.CacheException;

import org.jboss.cache.TreeCache;
import org.jboss.cache.Fqn;

/**
 * A Hibernate Cache implementation wraps the passed TreeCache
 * (presumably gotten from JNDI).
 *
 * @author <a href="mailto:steve@hibernate.org">Steve Ebersole</a>
 * @version <tt>$Revision: 1.2.2.1 $</tt>
 */
public class DeployedTreeCache
      implements Cache
{
   private static final String ITEM = "item";

   private final TreeCache cache;
   private final String regionName;

   public DeployedTreeCache(String regionName, TreeCache cache)
   {
      this.regionName = '/' + regionName.replace('.', '/');
      this.cache = cache;
   }

   public Object get(Object key) throws CacheException
   {
      try
      {
         return cache.get( constructEntityFqn(key), ITEM );
      }
      catch (Exception e)
      {
         throw new CacheException(e);
      }
   }

   public void put(Object key, Object value) throws CacheException
   {
      try
      {
         cache.put( constructEntityFqn(key), ITEM, value );
      }
      catch (Exception e)
      {
         throw new CacheException(e);
      }
   }

   public void remove(Object key) throws CacheException
   {
      try
      {
         cache.remove( constructEntityFqn(key) );
      }
      catch (Exception e)
      {
         throw new CacheException(e);
      }
   }

   public void clear() throws CacheException
   {
      try
      {
         cache.remove( constructRegionFqn() );
      }
      catch (Exception e)
      {
         throw new CacheException(e);
      }
   }

   public void destroy() throws CacheException
   {
      clear();
   }

   public void lock(Object key) throws CacheException
   {
      throw new UnsupportedOperationException("TreeCache is a fully transactional cache" + regionName);
   }

   public void unlock(Object key) throws CacheException
   {
      throw new UnsupportedOperationException("TreeCache is a fully transactional cache: " + regionName);
   }

   public long nextTimestamp()
   {
      return System.currentTimeMillis() / 100;
   }

   public int getTimeout()
   {
      return 600; //60 seconds
   }

   public String getRegionName()
   {
      return regionName;
   }

   public long getSizeInMemory()
   {
      return -1;
   }

   public long getElementCountInMemory()
   {
      return -1;
   }

   public long getElementCountOnDisk()
   {
      return -1;
   }

   private Fqn constructRegionFqn()
   {
      return new Fqn(regionName);
   }

   private Fqn constructEntityFqn(Object entityKey)
   {
      return new Fqn( new Object[] {regionName, entityKey} );
   }
}
