/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.axis.attachments;

import org.jboss.logging.Logger;

import javax.activation.DataSource;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * This class allows small attachments to be cached in memory.
 *
 * @author Thomas.Diesler@jboss.org
 */
public class ManagedMemoryDataSource implements DataSource
{
   private static Logger log = Logger.getLogger(ManagedMemoryDataSource.class);

   /** Field contentType */
   protected String contentType = "application/octet-stream";    // Is the default.

   // The memory cache
   private byte[] contentData;

   // The file cache
   private File cacheFile;

   // Memory is allocated in these size chunks.

   /** Field READ_CHUNK_SZ */
   public static final int READ_CHUNK_SZ = 32 * 1024;

   /** Field READ_CHUNK_SZ */
   public static final int MAX_MEMORY_SZ = 64 * 1024;

   /** Create a new in memory data source
    *
    * @param inputStream is the source input stream that is used to create this data source..
    * @param contentType the mime type for this data stream
    */
   public ManagedMemoryDataSource(InputStream inputStream, String contentType) throws IOException
   {
      if (contentType != null)
         this.contentType = contentType;

      ByteArrayOutputStream baos = new ByteArrayOutputStream(READ_CHUNK_SZ);
      OutputStream os = baos;

      byte[] bytes = new byte[READ_CHUNK_SZ];
      int read = inputStream.read(bytes);
      while (read > 0)
      {
         os.write(bytes, 0, read);

         if (baos != null && baos.size() > MAX_MEMORY_SZ)
         {
            // Create a cache file, which is cleared in finalize
            cacheFile = File.createTempFile("attachment", ".dat");
            cacheFile.deleteOnExit();

            log.debug("Created cache file: " + cacheFile.toURL());
            os = new FileOutputStream(cacheFile);
            os.write(baos.toByteArray());
            baos = null;
         }

         read = inputStream.read(bytes);
      }
      os.flush();
      os.close();

      if (baos != null)
      {
         log.debug("Using in memory cache: " + baos.size());
         this.contentData = baos.toByteArray();
      }
   }

   public String getContentType()
   {
      return contentType;
   }

   public InputStream getInputStream() throws IOException
   {
      if (contentData != null)
         return new ByteArrayInputStream(contentData);
      else if (cacheFile != null)
         return new FileInputStream(cacheFile);
      else
         throw new IllegalStateException("No cache available");
   }

   public String getName()
   {
      return null;
   }

   public OutputStream getOutputStream() throws IOException
   {
      return null;
   }

   /** Delete the cache file if one exists */
   protected void finalize() throws Throwable
   {
      super.finalize();

      if (cacheFile != null)
         cacheFile.delete();
   }
}
