/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.axis.attachments;

import javax.activation.DataSource;
import javax.xml.transform.stream.StreamSource;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.net.URL;

public class SourceDataSource implements DataSource
{
   public static final String CONTENT_TYPE = "text/xml";

   private final String name;
   private final String contentType;
   private byte[] data;
   private ByteArrayInputStream is;
   private ByteArrayOutputStream os;

   public SourceDataSource(String name, StreamSource data)
   {
      this(name, CONTENT_TYPE, data);
   } // ctor

   public SourceDataSource(String name, String contentType, StreamSource data)
   {
      this.name = name;
      this.contentType = contentType == null ? CONTENT_TYPE : contentType;
      os = new ByteArrayOutputStream();
      try
      {
         if (data != null)
         {
            // Try the Reader first
            Reader reader = data.getReader();
            if (reader != null)
            {
               reader = new BufferedReader(reader);
               int ch;
               while ((ch = reader.read()) != -1)
               {
                  os.write(ch);
               }
            }
            else
            {
               // Try the InputStream
               InputStream is = data.getInputStream();
               if (is == null)
               {
                  // Try the URL
                  String id = data.getSystemId();
                  if (id != null)
                  {
                     URL url = new URL(id);
                     is = url.openStream();
                  }
               }
               if (is != null)
               {
                  is = new BufferedInputStream(is);
                  // If reading from a socket or URL, we could get a partial read.
                  byte[] bytes = null;
                  int avail;
                  while ((avail = is.available()) > 0)
                  {
                     if (bytes == null || avail > bytes.length)
                        bytes = new byte[avail];
                     is.read(bytes, 0, avail);
                     os.write(bytes, 0, avail);
                  }
               }
            }
         }
      }
      catch (Exception e)
      {
         // Is this sufficient?
      }
   } // ctor

   public String getName()
   {
      return name;
   } // getName

   public String getContentType()
   {
      return contentType;
   } // getContentType

   public InputStream getInputStream() throws IOException
   {
      if (os.size() != 0)
      {
         data = os.toByteArray();
         os.reset();
      }
      return new ByteArrayInputStream(data == null ? new byte[0] : data);
   } // getInputStream

   public OutputStream getOutputStream() throws IOException
   {
      if (os.size() != 0)
      {
         data = os.toByteArray();
         os.reset();
      }
      return os;
   } // getOutputStream
} // class SourceDataSource
