/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.axis.encoding.ser;

import org.apache.axis.Constants;
import org.apache.axis.encoding.SerializationContext;
import org.apache.axis.encoding.SimpleValueSerializer;
import org.apache.axis.wsdl.fromJava.Types;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import javax.xml.namespace.QName;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

/**
 * Serializer for Dates.
 *
 * @author Sam Ruby <rubys@us.ibm.com>
 *         Modified by @author Rich scheuerle <scheu@us.ibm.com>
 * @see <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">XML Schema 3.2.16</a>
 */
public class DateSerializer extends SimpleSerializer
{

   private static SimpleDateFormat zuluDate =
           new SimpleDateFormat("yyyy-MM-dd");

   private static SimpleDateFormat zuluDateTime =
           new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");

   private static Calendar calendar = Calendar.getInstance();

   protected QName defaultXmlType = Constants.XSD_DATE;

   public DateSerializer(Class javaType, QName xmlType)
   {
      super(javaType, xmlType);
   }
               
   private String convertToZoneString(long offset) {
      if (offset == 0) return "Z";
      
      StringBuffer buffer = new StringBuffer();
      DecimalFormat hourFormat = new DecimalFormat("'+'00;-00");
      DecimalFormat minuteFormat = new DecimalFormat("00");

      int minutes = (int) offset / (1000 * 60);
      int hours = minutes / 60;
      
      minutes -= (hours * 60);

      return hourFormat.format(hours) + ":" + minuteFormat.format(minutes);
   }
   
   private SimpleDateFormat getDateFormatForType(QName type) {
   
      if (type.equals(Constants.XSD_DATE))
      {
         return zuluDate;
      }
      
      if (type.equals(Constants.XSD_DATETIME))
      {
         return zuluDateTime;
      }

      return null;
   }

   /**
    * Serialize a Date.
    */
   public void serialize(QName name, Attributes attributes,
                         Object value, SerializationContext context)
           throws IOException
   {
      context.startElement(name, attributes);
      context.writeString(getValueAsString(value, context));
      context.endElement();
   }

   public String getValueAsString(Object value, SerializationContext context)
   {
      StringBuffer buf = new StringBuffer();
      SimpleDateFormat zulu;

      zulu = getDateFormatForType(super.xmlType);

      if (zulu == null)
      {
         zulu = getDateFormatForType(defaultXmlType);
      }

      if (zulu == null) 
      {
         // Invlaid default, lets assume XSD_DATE
         zulu = zuluDate;
      }

      if (value instanceof Calendar)
      {
         Calendar calendarValue = (Calendar) value;
         
         if (calendarValue.get(Calendar.ERA) == GregorianCalendar.BC)
         {
            buf.append("-");
         }

         synchronized (zulu)
         {
            zulu.setCalendar(calendarValue);
            buf.append(zulu.format(calendarValue.getTime()));
         }

         /* Due to the limitations of xsd:datetime, we must lose DST information */
         buf.append(convertToZoneString(calendarValue.get(Calendar.ZONE_OFFSET)
                                      + calendarValue.get(Calendar.DST_OFFSET)));
      } 
      else
      {
         synchronized (calendar)
         {
            calendar.setTime((Date)value);
            if (calendar.get(Calendar.ERA) == GregorianCalendar.BC)
            {
               buf.append("-");
               calendar.set(Calendar.ERA, GregorianCalendar.AD);
               value = calendar.getTime();
            }
            synchronized (zulu)
            {
               zulu.setCalendar(calendar); 
               zulu.setTimeZone(TimeZone.getTimeZone("GMT"));
               buf.append(zulu.format((Date)value));
            }
            buf.append("Z");
      
         }
      }
      return buf.toString();
   }

   public String getMechanismType()
   {
      return Constants.AXIS_SAX;
   }

   /**
    * Return XML schema for the specified type, suitable for insertion into
    * the &lt;types&gt; element of a WSDL document, or underneath an
    * &lt;element&gt; or &lt;attribute&gt; declaration.
    *
    * @param javaType the Java Class we're writing out schema for
    * @param types    the Java2WSDL Types object which holds the context
    *                 for the WSDL being generated.
    * @return a type element containing a schema simpleType/complexType
    * @see org.apache.axis.wsdl.fromJava.Types
    */
   public Element writeSchema(Class javaType, Types types) throws Exception
   {
      return null;
   }
}
