/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.axis.encoding.ser;

import org.apache.axis.encoding.DeserializationContext;
import org.apache.axis.encoding.DeserializerImpl;
import org.jboss.logging.Logger;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import javax.xml.namespace.QName;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

/**
 * Deserializer for xsd:list
 *
 * @author Thomas Diesler (thomas.diesler@jboss.org)
 */
public class ListDeserializer extends DeserializerImpl
{
   private static Logger log = Logger.getLogger(ListDeserializer.class.getName());

   private List valueList = new ArrayList();

   private Class javaType;
   private QName xmlType;

   public ListDeserializer(Class javaType, QName xmlType)
   {
      this.javaType = javaType;
      this.xmlType = xmlType;
   }

   /**
    * This method is invoked after startElement when the element requires
    * deserialization (i.e. the element is not an href and the value is not nil.)
    * <p/>
    * Simply creates
    *
    * @param namespace  is the namespace of the element
    * @param localName  is the name of the element
    * @param prefix     is the prefix of the element
    * @param attributes are the attributes on the element...used to get the type
    * @param context    is the DeserializationContext
    */
   public void onStartElement(String namespace, String localName,
                              String prefix, Attributes attributes,
                              DeserializationContext context)
           throws SAXException
   {

      if (log.isDebugEnabled())
      {
         log.debug("Enter: ListDeserializer::startElement()");
      }

      if (context.isNil(attributes))
      {
         return;
      }

      if (log.isDebugEnabled())
      {
         log.debug("Exit: ListDeserializer::startElement()");
      }
   }

   /**
    * onEndElement is called by endElement.  It is not called
    * if the element has an href.
    *
    * @param namespace is the namespace of the child element
    * @param localName is the local name of the child element
    * @param context   is the deserialization context
    */
   public void onEndElement(String namespace, String localName, DeserializationContext context) throws SAXException
   {

      if (log.isDebugEnabled())
      {
         log.debug("Enter: ListDeserializer::endElement()");
      }

      Class componentType = javaType.getComponentType();

      Constructor ctor = null;
      try
      {
         // Get component type string constructor
         ctor = componentType.getConstructor(new Class[]{String.class});
      }
      catch (NoSuchMethodException e)
      {
         throw new IllegalStateException("Cannot obtain string constructor for: " + componentType);
      }

      // Build the return object array
      Object[] objArr = (Object[])Array.newInstance(componentType, valueList.size());

      // Fill the object array
      String strValue = null;
      try
      {
         for (int i = 0; i < valueList.size(); i++)
         {
            strValue = (String)valueList.get(i);
            Object obj = ctor.newInstance(new Object[]{strValue});
            objArr[i] = obj;
         }

         value = objArr;
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException("Cannot construct object with: " + strValue);
      }


      if (log.isDebugEnabled())
      {
         log.debug("Exit: ListDeserializer::endElement()");
      }
   }

   /**
    * Parse the content and construct an Object array from it.
    */
   public void characters(char[] p1, int p2, int p3) throws SAXException
   {
      super.characters(p1, p2, p3);

      String strContent = new String(p1, p2, p3);

      StringTokenizer st = new StringTokenizer(strContent);
      while (st.hasMoreTokens())
      {
         String nextToken = st.nextToken();
         Object item = getValueAsObject(nextToken);
         valueList.add(item);
      }
   }

   /**
    * Get the Object for this token
    */
   private Object getValueAsObject(String token)
   {
      return token;
   }
}
