/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 * 
 * Created on Feb 2, 2004
 */
package org.jboss.net.axis.security.handler;

import java.security.KeyStore;
import java.util.Map;

import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.axis.AxisFault;
import org.apache.axis.MessageContext;
import org.apache.log4j.Logger;
import org.apache.ws.axis.security.WSDoAllConstants;
import org.apache.ws.axis.security.WSDoAllSender;
import org.apache.ws.security.components.crypto.Crypto;
import org.jboss.net.axis.security.JBossCryptoFactory;
import org.jboss.net.axis.security.SecurityConstants;
import org.jboss.security.SecurityDomain;

/**
 * <dl>
 * <dt><b>Title: </b><dd>Web Service Security Response Handler</dd>
 * <p>
 * <dt><b>Description: </b><dd>This handler is responsible for signing and/or encrypting the response message.</dd>
 * <p>
 * <dt><b>Copyright: </b><dd>Copyright (c) 2004</dd>
 * <dt><b>Company: </b><dd>Green River Computing Services</dd>
 * </dl>
 * @author <a href="mailto:jasone@greenrivercomputing.com">Jason Essington</a>
 * @version $Revision: 1.3.4.1 $
 */
public class WSSResponseHandler extends WSDoAllSender
{
   protected Logger log = Logger.getLogger(this.getClass());

   SecurityDomain domain = null;

   public void invoke(MessageContext mc) throws AxisFault
   {
      /*
       * We stored off the alias of the user who signed the request (to be used to encrypt the response), 
       * but just recently werner added a feature to wss4j to track the senders certificate, so we'll first
       * check to see if wss4j should track the cert or if we should.
       */
      if (!WSDoAllConstants.USE_REQ_SIG_CERT.equals(getOption(WSDoAllConstants.ENCRYPTION_USER)))
      {
	      // retrieve the alias used to sign the request. This guy will be used to encrypt the response (encryption user)
	      String actor = (String) getOption(WSDoAllConstants.ACTOR);
	      String alias = null;
	      Map signers = (Map) mc.getProperty(SecurityConstants.MC_REQ_SIGNERS);
	      if (signers != null)
	      {
	         alias = (String) signers.get(actor);
	      }
	      // now put our ENCRYPTION_USER where the handler expects to find him.
	      // if there was no signature for this actor in the request, then this ENCRYPTION_USER will be set to null
	      // causing an AxisFault to be thrown later.
	      mc.setProperty(WSDoAllConstants.ENCRYPTION_USER, alias);
      }
      
      super.invoke(mc);
   }

   protected Crypto loadSignatureCrypto() throws AxisFault
   {
      if (log.isDebugEnabled())
         log.debug("Loading the Signature Crypto Class");
      if (domain == null)
         getSecurityDomain();
      // no need to test for a null domain as it is handled by getSecurityDomain
      
      KeyStore truststore = domain.getTrustStore();
      if (truststore == null)
         throw new AxisFault("WSSReceiverHandler: No truststore available.");
      String cryptoClass;
      if ((cryptoClass = (String) getOption(SecurityConstants.HANDLER_CRYPTO_CLASS)) == null)
         throw new AxisFault("WSSReceiverHandler: No Crypto implementation was defined.");
      return JBossCryptoFactory.getInstance(cryptoClass, truststore);
   }

   protected Crypto loadEncryptionCrypto() throws AxisFault
   {
      if (log.isDebugEnabled())
         log.debug("Loading the Decryption Crypto Class");
      if (domain == null)
         getSecurityDomain();
      // npe is handled in getSecurityDomain
      KeyStore keystore = domain.getKeyStore();
      if (keystore == null)
         throw new AxisFault("WSSReceiverHandler: No keystore available.");
      String cryptoClass;
      if ((cryptoClass = (String) getOption(SecurityConstants.HANDLER_CRYPTO_CLASS)) == null)
         throw new AxisFault("WSSReceiverHandler: No Crypto implementation was defined.");
      return JBossCryptoFactory.getInstance(cryptoClass, keystore);

   }

   private void getSecurityDomain() throws AxisFault
   {
      String sd;
      if ((sd = (String) getOption(SecurityConstants.HANDLER_SEC_DOMAIN)) == null)
         sd = "java:/jaas/other"; // this is as good a default as any I suppose.
      if (log.isDebugEnabled())
         log.debug("WSSReceiveHandler, securityDomain=" + sd);
      try
      {
         Object tempDomain = new InitialContext().lookup(sd);
         if (tempDomain != null && tempDomain instanceof SecurityDomain)
            domain = (SecurityDomain) tempDomain;
         else
         {
            // oops, we will not be able to get our keystore in the login module.
            log.fatal("The SecurityManager named " + sd + " is not a SecurityDomain");
            throw new AxisFault("WSSReceiverHandler: No security domain is available.");
         }
      }
      catch (NamingException e)
      {
         throw new AxisFault("Unable to find the securityDomain named: " + sd, e);
      }
   }
}
