/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
package org.jboss.mx.remoting;

import java.io.Serializable;
import java.util.*;


/**
 * NotificationQueue is an object that holds one or more NotificationEntry objects. This
 * object is created and passed from the server to the client during invocation so that the
 * client and re-deliver Notifications to client-side NotificationListeners.
 * 
 * @author <a href="mailto:jhaynie@vocalocity.net">Jeff Haynie</a>
 * @version $Revision: 1.1.2.3 $
 */
public class NotificationQueue implements Serializable
{
    private final String sessionId;
    private final Set notifications=new HashSet();
    static final long serialVersionUID = 1L;

    /**
     * create an empty queue
     *
     * @param sessionId
     */
    public NotificationQueue (String sessionId)
    {
        this.sessionId  = sessionId;
    }

    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (!(o instanceof NotificationQueue)) return false;

        final NotificationQueue notificationQueue = (NotificationQueue) o;

        if (notifications != null ? !notifications.equals(notificationQueue.notifications) : notificationQueue.notifications != null) return false;
        if (sessionId != null ? !sessionId.equals(notificationQueue.sessionId) : notificationQueue.sessionId != null) return false;

        return true;
    }

    public int hashCode()
    {
        int result;
        result = (sessionId != null ? sessionId.hashCode() : 0);
        result = 29 * result + (notifications != null ? notifications.hashCode() : 0);
        return result;
    }

    public String toString ()
    {
        return "NotificationQueue [sessionId:"+sessionId+",count:"+notifications.size()+",notifications:"+notifications+"]";
    }

    /**
     * clear the queue
     */
    public void clear ()
    {
        synchronized(notifications)
        {
            notifications.clear();
        }
    }

    /**
     * add an entry to the queue
     *
     * @param notification
     */
    void add (NotificationEntry notification)
    {
        synchronized(notifications)
        {
            notifications.add(notification);
        }
    }

    /**
     * return the session ID associated with the queue
     *
     * @return
     */
    public String getSessionID ()
    {
        return sessionId;
    }

    /**
     * return true if there are no entries, false if there are 1..n entries
     * @return
     */
    public boolean isEmpty ()
    {
        synchronized(notifications)
        {
            return notifications.isEmpty();
        }
    }

    /**
     * return an Iterator of NotificationEntry objects
     *
     * @return
     */
    public Iterator iterator ()
    {
        synchronized (notifications)
        {
            return notifications.iterator();
        }
    }
}
