/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package javax.management.modelmbean;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;
import java.io.StreamCorruptedException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.management.Descriptor;
import javax.management.MBeanException;
import javax.management.RuntimeOperationsException;

import org.jboss.logging.Logger;
import org.jboss.mx.modelmbean.ModelMBeanConstants;
import org.jboss.mx.util.Serialization;

/**
 * Support class for creating descriptors.
 *
 * @see javax.management.Descriptor
 *
 * @author  <a href="mailto:juha@jboss.org">Juha Lindfors</a>.
 * @author  <a href="mailto:adrian.brock@happeningtimes.com">Adrian Brock</a>.
 * @version $Revision: 1.6.4.6 $ 
 *
 * <p><b>Revisions:</b>
 *
 * <p><b>20020320 Juha Lindfors:</b>
 * <ul>
 * <li>toString() implementation</li>
 * </ul> 
 *
 * <p><b>20020525 Juha Lindfors:</b>
 * <ul>
 * <li>public currClass field removed to match JMX 1.1 MR </li>
 * </ul>
 *
 * <p><b>20020715 Adrian Brock:</b>
 * <ul>
 * <li> Serialization
 * </ul>
 */
public class DescriptorSupport
   implements Descriptor, Cloneable, Serializable
{

   // TODO: the spec doesn't define equality for descriptors
   //       we should override equals to match descriptor field, value pairs
   //       this does not appear to be the case with the 1.0 RI though

   private static Logger log = Logger.getLogger(DescriptorSupport.class);
   
   // Attributes ----------------------------------------------------
   
   /**
    * Map for the descriptor field -> value.
    */
   private Map fieldMap;

   // Static --------------------------------------------------------

   private static final int DEFAULT_SIZE = 20;

   private static final String[] PERSIST_POLICIES = new String[]
   {
      ModelMBeanConstants.PM_ON_UPDATE, 
      ModelMBeanConstants.PM_NO_MORE_OFTEN_THAN, 
      ModelMBeanConstants.PM_NEVER, 
      ModelMBeanConstants.PM_ON_TIMER
   };

   private static final long serialVersionUID;
   private static final ObjectStreamField[] serialPersistentFields;

   static
   {
      switch (Serialization.version)
      {
      case Serialization.V1R0:
         serialVersionUID = 8071560848919417985L;
         break;
      default:
         serialVersionUID = -6292969195866300415L;
      }
      serialPersistentFields = new ObjectStreamField[]
      {
         new ObjectStreamField("descriptor", HashMap.class)
      };
   }

   
   // Constructors --------------------------------------------------
   /**
    * Default constructor.
    */
   public DescriptorSupport()
   {
      fieldMap = Collections.synchronizedMap(new HashMap(DEFAULT_SIZE));
   }

   /**
    * Creates descriptor instance with a given initial size.
    *
    * @param   initialSize initial size of the descriptor
    * @throws  MBeanException this exception is never thrown but is declared here
    *          for Sun RI API compatibility
    * @throws  RuntimeOperationsException if the <tt>initialSize</tt> is zero or negative. The target
    *          exception wrapped by this exception is an instace of <tt>IllegalArgumentException</tt> class.
    */
   public DescriptorSupport(int initialSize) throws MBeanException
   { 
      if (initialSize <= 0)
         // required by RI javadoc
         throw new RuntimeOperationsException(new IllegalArgumentException("initialSize <= 0"));
         
      fieldMap = Collections.synchronizedMap(new HashMap(initialSize));
   }

   /**
    * Copy constructor.
    *
    * @param   descriptor the descriptor to be copied
    * @throws  RuntimeOperationsException if descriptor is null. The target exception wrapped by this
    *          exception is an instance of <tt>IllegalArgumentException</tt> class.
    */
   public DescriptorSupport(DescriptorSupport descriptor)
   {
      if (descriptor != null)
      {
         String[] fieldNames = descriptor.getFieldNames();
         fieldMap = Collections.synchronizedMap(new HashMap(fieldNames.length));
         this.setFields(fieldNames, descriptor.getFieldValues(fieldNames));
      }
      else
         fieldMap = Collections.synchronizedMap(new HashMap(DEFAULT_SIZE));
   }

   /**
    * Creates descriptor instance with given field names and values.if both field names and field
    * values array contain empty arrays, an empty descriptor is created.
    * None of the name entries in the field names array can be a <tt>null</tt> reference.
    * Field values may contain <tt>null</tt> references.
    *
    * @param   fieldNames  Contains names for the descriptor fields. This array cannot contain
    *                      <tt>null</tt> references. If both <tt>fieldNames</tt> and <tt>fieldValues</tt>
    *                      arguments contain <tt>null</tt> or empty array references then an empty descriptor
    *                      is created. The size of the <tt>fieldNames</tt> array must match the size of
    *                      the <tt>fieldValues</tt> array.
    * @param   fieldValues Contains values for the descriptor fields. Null references are allowed.
    *
    * @throws RuntimeOperationsException if array sizes don't match
    */
   public DescriptorSupport(String[] fieldNames, Object[] fieldValues) 
      throws RuntimeOperationsException
   {
      fieldMap = Collections.synchronizedMap(new HashMap(DEFAULT_SIZE));
      setFields(fieldNames, fieldValues);
   }

   public DescriptorSupport(String[] fields)
   {
      if (fields == null)
      {
         fieldMap = Collections.synchronizedMap(new HashMap(DEFAULT_SIZE));
         return;
      }

      int j = 0;
      for (int i = 0; i < fields.length; ++i)
         if (fields[i] != null && fields[i].length() != 0)
            ++j;

      fieldMap = Collections.synchronizedMap(new HashMap(j));
      String[] names = new String[j];
      String[] values = new String[j];

      j = 0;
      for (int i = 0; i < fields.length; ++i)
      {
         if (fields[i] == null || fields[i].length() == 0)
            continue;

         try
         {
            int index = fields[i].indexOf('=');
            if (index == -1)
               throw new IllegalArgumentException("Invalid field " + fields[i]);

            names[j] = fields[i].substring(0, index);
            if (index == fields[i].length()-1)
               values[j] = null;
            else
               values[j] = fields[i].substring(index + 1, fields[i].length());
            ++j;
         }
         catch (RuntimeException e)
         {
            throw new RuntimeOperationsException(e, "Error in field " + i);
         }
      }

      setFields(names, values);
   }

   /**
    * @todo implement this
    */
   public DescriptorSupport(String xmlString)
      throws MBeanException, RuntimeOperationsException, XMLParseException
   {
      throw new RuntimeException("NYI");
   }

   // Public --------------------------------------------------------
   public Object getFieldValue(String inFieldName)
   {
      try
      {
         checkFieldName(inFieldName);
         return fieldMap.get(inFieldName.toLowerCase());
      }
      catch (RuntimeException e)
      {
         throw new RuntimeOperationsException(e, e.toString());
      }
   }

   public void setField(String inFieldName, Object fieldValue)
   {
      try
      {
         checkFieldName(inFieldName);
         validateField(inFieldName, fieldValue);
         fieldMap.put(inFieldName.toLowerCase(), fieldValue);
      }
      catch (RuntimeException e)
      {
         throw new RuntimeOperationsException(e, e.toString());
      }
   }

   public String[] getFields()
   {
      String[] fieldStrings = new String[fieldMap.size()];
      Iterator it = fieldMap.keySet().iterator();
      synchronized (fieldMap)
      {
         for (int i = 0; i < fieldMap.size(); ++i)
         {
            String key = (String)it.next();
            fieldStrings[i] = key + "=" + fieldMap.get(key.toLowerCase());
         }
      }

      return fieldStrings;
   }

   public String[] getFieldNames()
   {
      return (String[])fieldMap.keySet().toArray(new String[0]);
   }

   public Object[] getFieldValues(String[] fieldNames)
   {
      Object[] values = null;
      if (fieldNames == null)
      {
         values = new Object[fieldMap.size()];  
         Iterator it = fieldMap.values().iterator();
         synchronized (fieldMap)
         {
            for (int i = 0; i < fieldMap.size(); ++i)
               values[i] = it.next();
         }
      }
      else
      {
         values = new Object[fieldNames.length];
         for (int i = 0; i < fieldNames.length; ++i)
         {
            if (fieldNames[i] == null || fieldNames[i].equals(""))
               values[i] = null;
            else
               values[i] = fieldMap.get(fieldNames[i].toLowerCase());
         }
      }
         
      return values;
   }

   public void setFields(String[] fieldNames, Object[] fieldValues)
   {
      try
      {
         if (fieldNames == null || fieldValues == null)
         throw new IllegalArgumentException("fieldNames or fieldValues was null.");
         
         if (fieldNames.length == 0 || fieldValues.length == 0)
            return;
         
         if (fieldNames.length != fieldValues.length)
            throw new IllegalArgumentException("fieldNames and fieldValues array size must match.");
         
         for (int i = 0; i < fieldNames.length; ++i)
         {
            String name = fieldNames[i];
            checkFieldName(name);
            validateField(name, fieldValues[i]);
            fieldMap.put(name.toLowerCase(), fieldValues[i]);
         }
      }
      catch (RuntimeException e)
      {
         throw new RuntimeOperationsException(e, e.toString());
      }
   }

   public synchronized Object clone()
   {
      try
      {
         DescriptorSupport clone = (DescriptorSupport)super.clone();
      
         clone.fieldMap = Collections.synchronizedMap(new HashMap(this.fieldMap));

         return clone;
      }
      catch (CloneNotSupportedException e)
      {
         // Descriptor interface won't allow me to throw CNSE
         throw new RuntimeOperationsException(new RuntimeException(e.getMessage()), e.toString());
      }
   }

   public void removeField(String fieldName)
   {
      if (fieldName == null || fieldName.equals(""))
         return;

      fieldMap.remove(fieldName.toLowerCase());
   }

   public boolean isValid() 
      throws RuntimeOperationsException
   {
      try
      {
         validateString(ModelMBeanConstants.NAME, true);
         validateString(ModelMBeanConstants.DESCRIPTOR_TYPE, true);

         synchronized (fieldMap)
         {
            for (Iterator i = fieldMap.entrySet().iterator(); i.hasNext(); )
            {
               Map.Entry entry = (Map.Entry) i.next();
               String name = (String) entry.getKey();
               Object value = entry.getValue();
               validateField(name, value);
            }
         }
      }
      catch (IllegalArgumentException e)
      {
         log.error("Invalid descriptor", e);
         return false;
      }
      catch (RuntimeException e)
      {
         throw new RuntimeOperationsException(e, e.toString());
      }

      return true;
   }

   /**
    * @todo implement this
    */
   public String toXMLString()
      throws RuntimeOperationsException
   {
      throw new RuntimeException("NYI");
   }

   // Object overrides ----------------------------------------------

   public String toString()
   {
      String[] names  = getFieldNames();
      Object[] values = getFieldValues(names);
      
      if (names.length == 0)
         return "<empty descriptor>";
         
      StringBuffer sbuf = new StringBuffer(500);
      
      for (int i = 0; i < values.length; ++i)
      {
         sbuf.append(names[i]);
         sbuf.append("=");
         sbuf.append(values[i]);
         sbuf.append(",");
      }
      
      sbuf.deleteCharAt(sbuf.length() - 1);
      
      return sbuf.toString();
   }

   // Private -----------------------------------------------------

   private void checkFieldName(String inFieldName)
   {
      if (inFieldName == null || inFieldName.equals(""))
         throw new IllegalArgumentException("null or empty field name");
   }

   private void validateField(String inFieldName, Object value)
   {
      String fieldName = inFieldName.toLowerCase();
      if (fieldName.equalsIgnoreCase(ModelMBeanConstants.NAME))
         validateString(inFieldName, value, true);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.DESCRIPTOR_TYPE))
         validateString(inFieldName, value, true);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.ROLE))
         validateString(inFieldName, value, false);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.GET_METHOD))
         validateString(inFieldName, value, false);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.SET_METHOD))
         validateString(inFieldName, value, false);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.PERSIST_PERIOD))
         validateNumeric(inFieldName, value);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.CURRENCY_TIME_LIMIT))
         validateNumeric(inFieldName, value);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.LAST_UPDATED_TIME_STAMP))
         validateNumeric(inFieldName, value);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.LAST_RETURNED_TIME_STAMP))
         validateNumeric(inFieldName, value);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.LOG))
         validateBoolean(inFieldName, value);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.VISIBILITY))
         validateNumeric(inFieldName, value, 1, 4);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.SEVERITY))
         validateNumeric(inFieldName, value, 1, 6);
      else if (fieldName.equalsIgnoreCase(ModelMBeanConstants.PERSIST_POLICY))
         validateString(inFieldName, value, false, PERSIST_POLICIES);
   }

   private void validateString(String fieldName, boolean mandatory)
   {
      validateString(fieldName, fieldMap.get(fieldName.toLowerCase()), mandatory);
   }

   private void validateString(String fieldName, Object value, boolean mandatory)
   {
      if (value == null && mandatory == true)
         throw new IllegalArgumentException("Expected a value for mandatory field '" + fieldName + "'");
      else if (value == null)
         throw new IllegalArgumentException("Expected a value for field '" + fieldName + "'");
      if ((value instanceof String) == false)
         throw new IllegalArgumentException("Expected a String for field '" + fieldName + "'");
      String string = (String) value;
      if (string.length() == 0)
         throw new IllegalArgumentException("Empty value for field '" + fieldName + "'");
   }

   private void validateString(String fieldName, Object value, boolean mandatory, String[] validValues)
   {
      validateString(fieldName, value, mandatory);
      String string = ((String) value).toLowerCase();
      for (int i = 0; i < validValues.length; ++i)
      {
         if (validValues[i].equals(string))
            return;
      }
      throw new IllegalArgumentException("Invalid value " + value + " for field '" + fieldName +
                                         "' expected one of " + Arrays.asList(validValues));
   }

   private void validateBoolean(String fieldName, Object value)
   {
      if (value == null)
         throw new IllegalArgumentException("Expected a value for field '" + fieldName + "'");
      if (value instanceof String)
      {
         String string = ((String) value).toLowerCase();
         if (string.equals("t") || string.equals("f") || string.equals("true") || string.equals("false"))
            return;
      }
      else if (value instanceof Boolean)
         return;
      throw new IllegalArgumentException("Invalid value " + value + " for field '" + fieldName + "'");
   }

   private long validateNumeric(String fieldName, Object value)
   {
      if (value == null)
         throw new IllegalArgumentException("Expected a value for field '" + fieldName + "'");

      Long number = null;
      if (value instanceof String)
         number = new Long((String) value);
      else if (value instanceof Number)
         number = new Long(((Number) value).longValue());
      if (number != null && number.longValue() >= -1)
         return number.longValue();

      throw new IllegalArgumentException("Invalid value " + value + " for field '" + fieldName + "'");
   }

   private void validateNumeric(String fieldName, Object value, int min, int max)
   {
      long result = validateNumeric(fieldName, value);
      if (result >= min && result <= max)
         return;
      throw new IllegalArgumentException("Invalid value " + value + " for field '" + fieldName + "'");
   }

   private void readObject(ObjectInputStream ois)
      throws IOException, ClassNotFoundException
   {
      ObjectInputStream.GetField getField = ois.readFields();
      HashMap descriptor = (HashMap) getField.get("descriptor", null);
      if (descriptor == null)
         throw new StreamCorruptedException("Null descriptor?");
      fieldMap = Collections.synchronizedMap(descriptor);
   }
   
   private void writeObject(ObjectOutputStream oos)
      throws IOException
   {
      ObjectOutputStream.PutField putField = oos.putFields();
      /* Since non-Serializable values can be put into the descriptor
         just remove them when writing out the serialized form
      */
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      ObjectOutputStream tstOOS = new ObjectOutputStream(baos);
      HashMap serialMap = new HashMap();
      Iterator it = fieldMap.entrySet().iterator();
      while (it.hasNext())
      {
         Map.Entry entry = (Map.Entry) it.next();
         Object value = entry.getValue();
         if( value instanceof Serializable )
         {
            // Validate that the object's references are serializable
            try
            {
               baos.reset();
               tstOOS.writeObject(value);
               serialMap.put(entry.getKey(), value);
            }
            catch(Exception ignore)
            {
            }
         }
      }
      baos.close();
      tstOOS.close();

      putField.put("descriptor", serialMap);
      oos.writeFields();
   }
}
