/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting;

import org.jboss.logging.Logger;
import org.jboss.remoting.invocation.InternalInvocation;

import java.util.ArrayList;
import java.util.List;


/**
 * Responsible for all callbacks in remoting at invoker level (on the server side).
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class ServerInvokerCallbackHandler implements InvokerCallbackHandler
{
    private InvocationRequest invocation;
    private Client callBackClient;
    private ArrayList callbacks = new ArrayList();
    private String sessionId;

    private static final Logger log = Logger.getLogger(ServerInvokerCallbackHandler.class);


    public ServerInvokerCallbackHandler(InvocationRequest invocation) throws Exception
    {
        if(invocation == null)
        {
            throw new Exception("Can not construct ServerInvokerCallbackHandler with null InvocationRequest.");
        }
        this.invocation = invocation;
        init(invocation);
    }

    private void init(InvocationRequest invocation) throws Exception
    {
        if(invocation.getLocator() != null)
        {
            callBackClient = new Client(invocation.getLocator(), invocation.getSubsystem());
            callBackClient.connect();
        }
        sessionId = invocation.getSessionId();

        if(log.isDebugEnabled())
        {
            log.debug("Session id for callback handler is " + sessionId);
        }
    }


    /**
     * Returns an id that can be used to identify this particular
     * callback handler, which should be representative of the
     * client invoker it will make callbacks to.  Currently, this
     * is the session id associated with the invocation request.
     * @return
     */
    public static String getId(InvocationRequest invocation)
    {
        String sessionId = invocation.getSessionId();
        return sessionId;
    }

    /**
     * Returns an id that can be used to identify this particular
     * callback handler, which should be representative of the
     * client invoker it will make callbacks to.
     * @return
     */
    public String getId()
    {
        return getId(invocation);
    }

    public List getCallbacks()
    {
        List callbackList = null;
        synchronized(callbacks)
        {
            callbackList = (List)callbacks.clone();
            callbacks.clear();
        }
        return callbackList;
    }

    /**
     * Will take the callback message and send back to client.
     * If client locator is null, will store them till client polls to get them.
     *
     * @param invocation
     * @throws Throwable
     */
    public void handleCallback(InvocationRequest invocation)
            throws HandleCallbackException
    {
        if(callBackClient == null)
        {
            synchronized(callbacks)
            {
                if(log.isDebugEnabled())
                {
                    log.debug("sync callback.  adding to callback list");
                }
                //TODO: Need to make this a fixed size to do not run out of memory -TME
                callbacks.add(invocation);
            }
        }
        else
        {
            try
            {
            // make sure connected
            if(!callBackClient.isConnected())
            {
                    callBackClient.connect();
            }
            if(callBackClient.isConnected())
            {
                if(log.isDebugEnabled())
                {
                    log.debug("async callback.  Calling client now.");
                }
                // sending internal invocation so server invoker we are sending to
                // will know how pass onto it's client callback handler
                InternalInvocation internalInvocation = new InternalInvocation(InternalInvocation.HANDLECALLBACK,
                                                                                new Object[]{invocation});
                callBackClient.setSessionId(sessionId);
                callBackClient.invoke(internalInvocation,
                                      invocation.getRequestPayload());
            }
            else
            {
                log.error("Can not handle callback since can not connect to client invoker.");
                throw new HandleCallbackException("Can not handle callback since can not connect to client invoker.");
            }
            }
            catch(Throwable ex)
            {
                log.debug("Error dispatching callback to handler.", ex);
                throw new HandleCallbackException("Error dispatching callback to handler.", ex);
            }
        }
    }

    /**
     * Returns the id for this handler
     * @return
     */
    public String toString()
    {
        return getClass().getName() + " - id: " + getId();
    }

    /**
     * This method is required to be called upon removing a callback listener
     * so can clean up resources used by the handler.  In particular, should
     * call disconnect on internal Client.
     */
    public void destroy()
    {
        if(callBackClient != null)
        {
            callBackClient.disconnect();
            callBackClient = null;
        }
    }
}
