/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
package org.jboss.remoting.transport.socket;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.util.ArrayList;
import java.util.List;

import org.jboss.logging.Logger;
import org.jboss.remoting.ConnectionFailedException;
/**
 * SocketClientPool is a simple pool of client Socket connections back to the remote
 * Socket Server.
 *
 * @author <a href="mailto:jhaynie@vocalocity.net">Jeff Haynie</a>
 * @version $Revision: 1.5.4.1 $
 */
public class SocketClientPool
{
    private static final Logger log = Logger.getLogger(SocketClientPool.class);

    private final InetAddress addr;
    private final int port;
    private final List available = new ArrayList();

    SocketClientPool (InetAddress addr, int port)
    {
        this.addr = addr;
        this.port = port;
    }
    public void connect () throws IOException
    {
        if (log.isDebugEnabled())
        {
            log.debug("connect");
        }

        // just check out and back it which will establish the first
        // connection client
        makeAvailable(getNextAvailable());
    }
    public int getAvailableCount ()
    {
        synchronized(available)
        {
            return available.size();
        }
    }
    public void disconnect ()
    {
        if (log.isDebugEnabled())
        {
            log.debug("disconnect");
        }
        synchronized(available)
        {
            int size = available.size();
            if(size > 0)
            {
                for(int x = size - 1; x >= 0; x--)
                {
                    Socket s = (Socket) available.get(x);
                    try
                    {
                        s.close();
                    }
                    catch(Exception ex)
                    {
                    }
                    available.remove(s);
                    s = null;
                }
            }
        }
    }
    /**
     * return the next available Socket object for a given invocation session.
     * will automatically allocate a new Socket if none are available.
     *
     * @return
     * @throws IOException
     */
    public Socket getNextAvailable () throws IOException
    {
        synchronized(available)
        {
            if (available.isEmpty()==false)
            {
                return (Socket)available.remove(0);
            }
        }
        // not available, make one on demand
        try
        {
            if (log.isDebugEnabled())
            {
                log.debug("connecting to client: "+addr+":"+port);
            }
            Socket s = new Socket(addr, port)
            {
                public void close() throws IOException
                {
                    try
                    {
                        super.close();
                    }
                    finally
                    {
                        try
                        {
                            if (available!=null)
                            {
                                synchronized(available)
                                {
                                    available.remove(this);
                                }
                            }
                        }
                        catch (Throwable ignore) { }
                    }
                }
            };
            s.setKeepAlive(true);
            s.setSoTimeout(0);
            return s;
        }
        catch (Exception ex)
        {
            if (log.isDebugEnabled())
            {
                log.debug("connection to client: "+addr+":"+port+" failed",ex);
            }
           throw new ConnectionFailedException(ex.getMessage());
        }
    }
    /**
     * return an available Socket that has been used to the pool for some other
     * execution thread to use
     *
     * @param socket
     */
    public void makeAvailable (Socket socket)
    {
        synchronized(available)
        {
            available.add(socket);
        }
    }

    public String toString() {
        return "org.jboss.remoting.transport.socket.SocketClientPool{" +
                "addr=" + addr +
                ", port=" + port +
                ", available=" + (available == null ? null : "size:" + available.size() + available) +
                "}";
    }

}
