/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package test;

import org.jboss.logging.Logger;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.InvokerRegistry;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.mock.MockClientInvoker;
import org.jboss.remoting.transport.mock.MockServerInvoker;
import org.jboss.remoting.invocation.NameBasedInvocation;
import org.jboss.remoting.transport.mock.MockTest;
import org.w3c.dom.Document;

import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.rmi.server.UID;
import java.util.Random;

/**
 * This is the actual concrete test for the invoker client.
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class InvokerClientTest extends AbstractInvokerTest
{
    private static final Logger log = Logger.getLogger(InvokerClientTest.class);

    private String sessionId = new UID().toString();

    private Client client;

    private static final String NAME = "InvokerClientTest.class";

    public InvokerClientTest(String name)
    {
        super(NAME);
    }

    public InvokerClientTest(int numberOfInstances)
    {
        super(NAME, numberOfInstances);
    }

    public InvokerClientTest(String transport, int port)
    {
        super(NAME, transport, port);
    }

    public InvokerClientTest(String transport, int port, int numberOfInstances)
    {
        super(NAME, transport, port, numberOfInstances);
    }

    public void init()
    {
        try
        {
            InvokerLocator locator = new InvokerLocator(transport + "://localhost:" + port);
            client = new Client(locator, "mock");
            client.connect();
        }
        catch(Exception e)
        {
            log.error(e.getMessage(), e);
        }
    }

    //TODO: Same method in InvokerServerTest so could move up into AbstractInvokerTest
    private InvokerLocator initServer(int port) throws Exception
    {
        if(port < 0)
        {
            port = Math.abs(new Random().nextInt(2000));
        }
        log.debug("port = " + port);

        InvokerRegistry.registerInvoker("mock", MockClientInvoker.class, MockServerInvoker.class);
        Connector connector = new Connector();
        InvokerLocator locator = new InvokerLocator(transport + "://localhost:" + port);
        StringBuffer buf = new StringBuffer();
        buf.append("<?xml version=\"1.0\"?>\n");
        buf.append("<handlers>\n");
        buf.append("  <handler subsystem=\"mock\">org.jboss.remoting.transport.mock.MockServerInvocationHandler</handler>\n");
        buf.append("</handlers>\n");
        Document xml = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new ByteArrayInputStream(buf.toString().getBytes()));
        connector.setInvokerLocator(locator.getLocatorURI());
        connector.setConfiguration(xml.getDocumentElement());
        //connector.create();
        connector.start();
        return locator;
    }


    public void runInvokers() throws Throwable
    {
        startup(getNumberOfInstances());
        try
        {
            testPullCallback();
            testLocalPushCallback();
        }
        finally
        {
            shutdown();
        }
    }

    /**
     * Test simple invocation and adding of listener with push callback (meaning server
     * will send callback message when it gets it) to a local callback server
     * @throws Throwable
     */
    public void testLocalPushCallback() throws Throwable
    {
        try
        {
            log.debug("running testLocalPushCallback()");

            sessionId = new UID().toString();
            InvokerLocator locator = initServer(-1);
            init();

            sessionId = client.getSessionId();
            MockInvokerCallbackHandler handler = new MockInvokerCallbackHandler(sessionId);

            log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

            // simple invoke, should return bar
            Object ret = makeInvocation("foo", "bar");
            assertTrue("Result of testLocalPushCallback() invocation of foo.", "bar".equals(ret));
            client.addListener(handler, locator);
            // invoke which should cause callback
            ret = makeInvocation("test", "test");
            // allow time for callback
            Thread.sleep(3000);
            log.debug("done sleeping.");
            int callbacksPerformed = handler.isCallbackReceived();
            log.debug("callbacksPerformed after adding listener is " + callbacksPerformed);
            assertTrue("Result of testLocalPushCallback() failed since did not get callback.",
                       (callbacksPerformed == 1));
            // Can now call direct on client
            client.removeListener(handler);
            // shouldn't get callback now since removed listener
            ret = makeInvocation("test", "test");
            // allow time for callback
            Thread.sleep(2000);
            log.debug("done sleeping.");
            callbacksPerformed = handler.isCallbackReceived();
            log.debug("callbackPerformed after removing listener is " + callbacksPerformed);
            assertTrue("Result of testLocalPushCallback() failed since did get callback " +
                       "but have been removed as listener.",
                       (callbacksPerformed == 1));
        }
        finally
        {
            if(client != null)
            {
                client.disconnect();
            }
        }
    }

    /**
     * Test simple invocation and adding of listener with push callback (meaning server
     * will send callback message when it gets it) to a remote callback server
     * @throws Throwable
     */
    public void testRemotePushCallback() throws Throwable
    {
        try
        {
            log.debug("running testRemotePushCallback()");

            sessionId = new UID().toString();
            initServer(-1);
            init();
            InvokerLocator locator = client.getInvoker().getLocator();
            sessionId = client.getSessionId();
            MockInvokerCallbackHandler handler = new MockInvokerCallbackHandler(sessionId);

            log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

            // simple invoke, should return bar
            Object ret = makeInvocation("foo", "bar");
            assertTrue("Result of testRemotePushCallback() invocation of foo.", "bar".equals(ret));
            client.addListener(handler, locator);
            // invoke which should cause callback
            ret = makeInvocation("test", "test");
            // allow time for callback
            Thread.sleep(3000);
            log.debug("done sleeping.");
            // TODO: No way to currently check the remote callback handler
            // to see if it got callback -TME
            /*
            int callbacksPerformed = handler.isCallbackReceived();
            log.debug("callbacksPerformed after adding listener is " + callbacksPerformed);
            assertTrue("Result of testRemotePushCallback() failed since did not get callback.",
                       (callbacksPerformed == 1));
            */
            // Can now call direct on client
            client.removeListener(handler);
            // shouldn't get callback now since removed listener
            ret = makeInvocation("test", "test");
            // allow time for callback
            Thread.sleep(2000);
            log.debug("done sleeping.");
            /*
            callbacksPerformed = handler.isCallbackReceived();
            log.debug("callbackPerformed after removing listener is " + callbacksPerformed);
            assertTrue("Result of testRemotePushCallback() failed since did get callback " +
                       "but have been removed as listener.",
                       (callbacksPerformed == 1));
            */
        }
        finally
        {
            if(client != null)
            {
                client.disconnect();
            }
        }
    }

    /**
     * Tests simple invocation and pull callbacks.  Meaning will add a listener and
     * will then have to get the callbacks from the server.
     * @throws Throwable
     */
    public void testPullCallback() throws Throwable
    {
        try
        {
            log.debug("running testPullCallback()");

            init();
            // should be null by default, since don't have connector started, but setting anyway
            //client.setClientLocator(null);

            MockInvokerCallbackHandler handler = new MockInvokerCallbackHandler(sessionId);

            // simple invoke, should return bar
            Object ret = makeInvocation("bar", "foo");
            assertTrue("Result of runPullCallbackTest() invocation of bar.", "foo".equals(ret));
            client.addListener(handler);
            // invoke which should cause callback on server side
            ret = makeInvocation("test", "test");
            // allow time for callback
            Thread.sleep(2000);
            ret = client.getCallbacks();
            log.debug("getCallbacks returned " + ret);
            log.debug("should have something.");
            assertTrue("Result of runPullCallbackTest() getCallbacks() after add listener.",
                       ret != null);
            // can now call directly on client
            //ret = makeInvocation("removeListener", null);
            client.removeListener(handler);
            ret = makeInvocation("getCallbacks", null);
            log.debug("getCallbacks returned " + ret);
            log.debug("should have been empty.");
            assertTrue("Result of runPullCallbackTest() getCallbacks() after remove listener.",
                       ret == null);
        }
        finally
        {
            if(client != null)
            {
                client.disconnect();
                client = null;
            }
        }
    }

    /**
     * Tests complex invocation to get object containing array of complex objects.
     * @throws Throwable
     */
    public void testArrayReturn() throws Throwable
    {
        try
        {
            init();

            // simple invoke, should return bar
            Object ret = makeInvocation("testComplexReturn", null);
            ComplexReturn complexRet = (ComplexReturn)ret;
            MockTest[] mockTests = complexRet.getMockTests();
            assertTrue("ComplexReturn's array should contain 2 items",
                       2 == mockTests.length);
            for(int x = 0; x < mockTests.length; x++)
            {
                System.err.println(mockTests[x]);
                MockTest test = mockTests[x];
                assertNotNull("MockTest should not be null", test);
            }

//            assertTrue("Result of runPullCallbackTest() invocation of bar.",
//                       "foo".equals(ret));
        }
        finally
        {
            if(client != null)
            {
                client.disconnect();
                client = null;
            }
        }
    }

    private Object makeInvocation(String method, String param) throws Throwable
    {
        Object ret = client.invoke(new NameBasedInvocation(method,
                                                           new Object[]{param},
                                                           new String[]{String.class.getName()}),
                                   null);

        return ret;
    }

    public static void main(String[] args)
    {
        InvokerClientTest client = null;
        if(args.length == 1)
        {
            int instances = Integer.parseInt(args[0]);
            client = new InvokerClientTest(instances);
        }
        else if(args.length == 2)
        {
            String transport = args[0];
            int port = Integer.parseInt(args[1]);
            client = new InvokerClientTest(transport, port);
        }
        else if(args.length == 3)
        {
            String transport = args[0];
            int port = Integer.parseInt(args[1]);
            int instances = Integer.parseInt(args[2]);
            client = new InvokerClientTest(transport, port, instances);
        }
        else
        {
            client = new InvokerClientTest(InvokerClientTest.class.getName());
            System.out.println("Using default transport (" + client.getTransport() +
                               ") and default port (" + client.getPort() + ") and " +
                               "default number of instances (" + client.getNumberOfInstances() + ")" +
                               "\nCan enter transport, port, and instances via command line.");
        }

        try
        {
            //regular class run
            //client.runInvokers();
            MultipleTestRunner runner = new MultipleTestRunner();
            runner.doRun(client, true);
        }
        catch(Throwable e)
        {
            e.printStackTrace();
            System.exit(1);
        }
        System.exit(0);
    }
}
