/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package test;

import org.jboss.logging.Logger;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.invocation.NameBasedInvocation;

import java.rmi.server.UID;

/**
 * This is the actual concrete test for the invoker client to test oneway calls (client and server based).
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class OnewayInvokerClientTest extends AbstractInvokerTest
{
   private static final Logger log = Logger.getLogger(OnewayInvokerClientTest.class);

   private String sessionId = new UID().toString();

   private Client client;

   private static final String NAME = "OnewayInvokerClientTest.class";

   public OnewayInvokerClientTest(String name)
   {
      super(NAME);
   }

   public OnewayInvokerClientTest(int numberOfInstances)
   {
      super(NAME, numberOfInstances);
   }

   public OnewayInvokerClientTest(String transport, int port)
   {
      super(NAME, transport, port);
   }

   public OnewayInvokerClientTest(String transport, int port, int numberOfInstances)
   {
      super(NAME, transport, port, numberOfInstances);
   }

   public void init()
   {
      try
      {
         InvokerLocator locator = new InvokerLocator(transport + "://localhost:" + port);
         client = new Client(locator, "test");
         client.connect();
      }
      catch (Exception e)
      {
         log.error(e.getMessage(), e);
      }
   }

   public void runInvokers() throws Throwable
   {
      startup(getNumberOfInstances());
      try
      {
         log.debug("\n\n*****************************" +
                   "Calling testOnewayServerInvocation" +
                   "*******************************\n");
         testOnewayServerInvocation();
         log.debug("\n\n*****************************" +
                   "Calling testOnewayClientInvocation" +
                   "*******************************\n");
         testOnewayClientInvocation();
      }
      finally
      {
         shutdown();
      }
   }

   /**
    * Test simple oneway client invocation
    *
    * @throws Throwable
    */
   public void testOnewayServerInvocation() throws Throwable
   {
      try
      {
         log.debug("running testOnewayClientCallback()");

         sessionId = new UID().toString();
         init();

         sessionId = client.getSessionId();

         log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

         // simple invoke
         makeServerOnewayInvocation("foo", "bar");
      }
      finally
      {
         if (client != null)
         {
            client.disconnect();
         }
      }
   }

   protected void makeServerOnewayInvocation(String method, String param) throws Throwable
   {
      client.invokeOneway(new NameBasedInvocation(method,
            new Object[]{param},
            new String[]{String.class.getName()}),
            null,
            false);

   }

   /**
    * Test simple oneway client invocation
    *
    * @throws Throwable
    */
   public void testOnewayClientInvocation() throws Throwable
   {
      try
      {
         log.debug("running testOnewayClientCallback()");

         sessionId = new UID().toString();
         init();

         sessionId = client.getSessionId();

         log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

         // simple invoke
         makeClientOnewayInvocation("foo", "bar");
      }
      finally
      {
         if (client != null)
         {
            client.disconnect();
         }
      }
   }

   protected void makeClientOnewayInvocation(String method, String param) throws Throwable
   {
      client.invokeOneway(new NameBasedInvocation(method,
            new Object[]{param},
            new String[]{String.class.getName()}),
            null,
            true);

   }

   protected Object makeInvocation(String method, String param) throws Throwable
   {
       Object ret = client.invoke(new NameBasedInvocation(method,
                                                          new Object[]{param},
                                                          new String[]{String.class.getName()}),
                                  null);

       return ret;
   }


   public static void main(String[] args)
   {
      OnewayInvokerClientTest client = null;
      if (args.length == 1)
      {
         int instances = Integer.parseInt(args[0]);
         client = new OnewayInvokerClientTest(instances);
      }
      else if (args.length == 2)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         client = new OnewayInvokerClientTest(transport, port);
      }
      else if (args.length == 3)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         int instances = Integer.parseInt(args[2]);
         client = new OnewayInvokerClientTest(transport, port, instances);
      }
      else
      {
         client = new OnewayInvokerClientTest(OnewayInvokerClientTest.class.getName());
         System.out.println("Using default transport (" + client.getTransport() +
               ") and default port (" + client.getPort() + ") and " +
               "default number of instances (" + client.getNumberOfInstances() + ")" +
               "\nCan enter transport, port, and instances via command line.");
      }

      try
      {
         //regular class run
         //client.runInvokers();
         MultipleTestRunner runner = new MultipleTestRunner();
         runner.doRun(client, true);
      }
      catch (Throwable e)
      {
         e.printStackTrace();
         System.exit(1);
      }
      System.exit(0);
   }
}
