/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package test.classloading.client;

import org.apache.log4j.FileAppender;
import org.apache.log4j.Level;
import org.apache.log4j.SimpleLayout;
import org.jboss.logging.Logger;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.invocation.NameBasedInvocation;
import test.AbstractInvokerTest;
import test.MultipleTestRunner;

import java.io.IOException;

/**
 * This is the actual concrete test for the invoker client.
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class InvokerClientTest extends AbstractInvokerTest
{
   private static final Logger log = Logger.getLogger(InvokerClientTest.class);

   private Client client;

   private static final String NAME = "InvokerClientTest.class";

   public InvokerClientTest(String name)
   {
      super(NAME);
   }

   public InvokerClientTest(int numberOfInstances)
   {
      super(NAME, numberOfInstances);
   }

   public InvokerClientTest(String transport, int port)
   {
      super(NAME, transport, port);
   }

   public InvokerClientTest(String transport, int port, int numberOfInstances)
   {
      super(NAME, transport, port, numberOfInstances);
   }

   public void init()
   {
      try
      {
         InvokerLocator locator = new InvokerLocator(transport + "://localhost:" + port);
         client = new Client(locator, "mock");
         client.connect();
      }
      catch (Exception e)
      {
         log.error(e.getMessage(), e);
      }
   }

   public void runInvokers() throws Throwable
   {
      startup(getNumberOfInstances());
      try
      {
         testClassloading();
      }
      finally
      {
         shutdown();
      }
   }


   public void testClassloading() throws Throwable
   {
      try
      {
         log.debug("running testClassloading()");

         init();

         log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

         // simple invoke passing Foo, which should not exist within server classloader.
         Foo foo = new Foo();
         Object ret = makeInvocation("printFoo", foo);
         System.out.println("return of " + ret);
         assertTrue("Result of testClassloading() invocation of printFoo.", "Bar::toString()".equals(ret.toString()));
      }
      catch (Exception e)
      {
         System.err.println("Error making invocation.");
         e.printStackTrace();
         assertNotNull("Failure making invocation.  " + e, null);
      }
      finally
      {
         if (client != null)
         {
            client.disconnect();
         }
      }
   }

   private Object makeInvocation(String method, Object param) throws Throwable
   {
      Object ret = client.invoke(new NameBasedInvocation(method,
            new Object[]{param},
            new String[]{param.getClass().getName()}),
            null);

      return ret;
   }

   public static void main(String[] args)
   {

      org.apache.log4j.BasicConfigurator.configure();
      org.apache.log4j.Category.getRoot().setLevel(Level.DEBUG);
      try
      {
         org.apache.log4j.Category.getRoot().addAppender(new FileAppender(new SimpleLayout(), "client_test.log"));
      }
      catch (IOException e)
      {
         e.printStackTrace();
      }


      InvokerClientTest client = null;
      if (args.length == 1)
      {
         int instances = Integer.parseInt(args[0]);
         client = new InvokerClientTest(instances);
      }
      else if (args.length == 2)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         client = new InvokerClientTest(transport, port);
      }
      else if (args.length == 3)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         int instances = Integer.parseInt(args[2]);
         client = new InvokerClientTest(transport, port, instances);
      }
      else
      {
         client = new InvokerClientTest(InvokerClientTest.class.getName());
         System.out.println("Using default transport (" + client.getTransport() +
               ") and default port (" + client.getPort() + ") and " +
               "default number of instances (" + client.getNumberOfInstances() + ")" +
               "\nCan enter transport, port, and instances via command line.");
      }

      try
      {
         //regular class run
         //client.runInvokers();
         MultipleTestRunner runner = new MultipleTestRunner();
         runner.doRun(client, true);
      }
      catch (Throwable e)
      {
         e.printStackTrace();
         System.exit(1);
      }
      System.exit(0);
   }
}
