/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.test;

import java.util.ArrayList;
import java.util.Hashtable;

import javax.management.ObjectName;
import javax.naming.InitialContext;

import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

/**
 * Derived implementation of JBossTestServices for cluster testing.
 *
 * @author <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 * @author Scott.Stark@jboss.org
 * @version $Revision: 1.1.4.6 $
 */
public class JBossTestClusteredServices extends JBossTestServices
{
   protected ArrayList adaptors = null;
   protected ArrayList servers = null;
   protected ArrayList namingURLs = null;
   protected ArrayList httpURLs = null;
   
   // Constructors --------------------------------------------------
   
   public JBossTestClusteredServices(String className)
   {
      super(className);
   }

   int getServerCount()
   {
      return servers.size();
   }

   RMIAdaptor[] getAdaptors() throws Exception
   {
      init();
      RMIAdaptor[] tmp = new RMIAdaptor[adaptors.size()];
      adaptors.toArray(tmp);
      return tmp;
   }

   RMIAdaptor getAdaptor(int index) throws Exception
   {
      init();
      RMIAdaptor adaptor = (RMIAdaptor) adaptors.get(index);
      return adaptor;
   }

   String[] getServers() throws Exception
   {
      init();
      String[] tmp = new String[servers.size()];
      servers.toArray(tmp);
      return tmp;
   }

   String getServer(int index) throws Exception
   {
      init();
      String server = (String) servers.get(index);
      return server;
   }

   /**
    * Get the JNDI provider urls for the cluster nodes
    *
    * @return
    * @throws Exception
    */
   String[] getNamingURLs() throws Exception
   {
      init();
      String[] tmp = new String[namingURLs.size()];
      namingURLs.toArray(tmp);
      return tmp;
   }

   String getNamingURL(int index) throws Exception
   {
      init();
      String server = (String) namingURLs.get(index);
      return server;
   }

   /**
    * Get the default web container urls for the cluster nodes
    *
    * @return
    * @throws Exception
    */
   String[] getHttpURLs() throws Exception
   {
      init();
      String[] tmp = new String[httpURLs.size()];
      httpURLs.toArray(tmp);
      return tmp;
   }

   String getHttpURL(int index) throws Exception
   {
      init();
      String server = (String) httpURLs.get(index);
      return server;
   }

   /**
    * Override to invoke the operation on all servers
    *
    * @param name
    * @param method
    * @param args
    * @param sig
    * @return
    * @throws Exception
    */
   protected Object invoke(ObjectName name, String method, Object[] args,
      String[] sig)
      throws Exception
   {
      RMIAdaptor[] adaptors = getAdaptors();

      Object result = null;
      for (int i = 0; i < adaptors.length; i++)
      {
         RMIAdaptor adaptor = adaptors[i];
         log.debug("Using RMIAdaptor: " + adaptor);
         result = invoke(adaptor, name, method, args, sig);
      }

      return result;

   }

   public void init() throws Exception
   {
      if (initialContext == null)
      {
         initialContext = new InitialContext();
      }
      if (adaptors == null)
      {
         adaptors = new ArrayList();
         servers = new ArrayList();
         namingURLs = new ArrayList();
         httpURLs = new ArrayList();
         String adaptorName = System.getProperty("jbosstest.server.name");
         if (adaptorName == null)
            adaptorName = "jmx/invoker/RMIAdaptor";

         Hashtable env = new Hashtable();
         env.put("java.naming.factory.initial", "org.jnp.interfaces.NamingContextFactory");
         env.put("java.naming.factory.url.pkgs", "org.jnp.interfaces");

         // Look for jbosstest.cluster.nodeN properties for the server names
         String node = "jbosstest.cluster.node";
         int count = 0;
         while (count < 10)
         {
            String prop = node + count;
            String host = System.getProperty(prop);
            count++;
            if (host == null)
               break;
            log.info(prop + " = " + host);
            servers.add(host);
            // See if there is a jbosstest.cluster.nodeN.jndi.url
            String urlProp = prop + ".jndi.url";
            String urlDefault = "jnp://" + host + ":1099";
            String urlValue = System.getProperty(urlProp, urlDefault);
            log.debug("JNDI Url for node=" + count + " is:" + urlValue);
            namingURLs.add(urlValue);
            env.put("java.naming.provider.url", urlValue);
            // Lookup the adaptor
            InitialContext tmpCtx = new InitialContext(env);
            RMIAdaptor adaptor = (RMIAdaptor) tmpCtx.lookup(adaptorName);
            adaptors.add(adaptor);

            // See if there is a jbosstest.cluster.nodeN.http.url
            urlProp = prop + ".http.url";
            urlDefault = "http://" + host + ":8080";
            urlValue = System.getProperty(urlProp, urlDefault);
            log.debug("Http Url for node=" + count + " is:" + urlValue);
            httpURLs.add(urlValue);
         }

         if (adaptors.size() == 0)
            throw new IllegalStateException("No jbosstest.cluster.node values found");
      }
   }

   /**
    * This method gives overriding testcases to set the cluster servernames
    */
   public void setServerNames(String[] snames)
   {
      if (snames == null) return;
      for (int i = 0; i < snames.length; i++)
      {
         servers.add((String) snames[i]);
      }
   }

}
