/*
 *
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.test.cache.perf.basic;


import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.PropertyConfigurator;
import org.jboss.cache.TreeCache;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.transaction.DummyTransactionManager;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.transaction.UserTransaction;
import java.text.DecimalFormat;
import java.text.FieldPosition;
import java.util.ArrayList;
import java.util.Properties;

/**
 * Replicated synchronous mode performance test for transactional TreeCache.
 *
 * @version $Revision: 1.1.6.1 $
 * @author<a href="mailto:bwang@jboss.org">Ben Wang</a> May 20 2003
 */
public class ReplicatedSyncPerfTestCase extends TestCase
{
   TreeCache cache1_, cache2_, cache3_;
   int cachingMode_ = TreeCache.REPL_SYNC;
   final String groupName_ = "TreeCacheTestGroup";
   final static Properties p_;
//   final static Logger log_ = Logger.getLogger(ReplicatedSyncPerfAopTest.class);
   String oldFactory_ = null;
   final String FACTORY = "org.jboss.cache.transaction.DummyContextFactory";

   ArrayList nodeList_;
   // (4, 4) combination will generate 340 nodes.
   static final int depth_ = 3;
   static final int children_ = 4;
   DummyTransactionManager tm_;

   static
   {
      p_ = new Properties();
      p_.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.cache.transaction.DummyContextFactory");
   }

   public ReplicatedSyncPerfTestCase(String name)
   {
      super(name);
   }

   public void setUp() throws Exception
   {
      super.setUp();

      oldFactory_ = System.getProperty(Context.INITIAL_CONTEXT_FACTORY);
      System.setProperty(Context.INITIAL_CONTEXT_FACTORY, FACTORY);

      DummyTransactionManager.getInstance();
      nodeList_ = nodeGen(depth_, children_);
      tm_ = new DummyTransactionManager();

      log("ReplicatedSyncPerfAopTest: cacheMode=REPL_SYNC");
   }

   public void tearDown() throws Exception
   {
      super.tearDown();

      DummyTransactionManager.destroy();

      if (oldFactory_ != null) {
         System.setProperty(Context.INITIAL_CONTEXT_FACTORY, oldFactory_);
         oldFactory_ = null;
      }

   }

   TreeCache createCache() throws Exception
   {
      TreeCache cache = new TreeCache();
      PropertyConfigurator config = new PropertyConfigurator();
      config.configure(cache, "META-INF/replSync-service.xml"); // use generic syncRepl xml
      cache.setIsolationLevel(IsolationLevel.REPEATABLE_READ);
      cache.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      return cache;
   }

   void destroyCache(TreeCache cache) throws Exception
   {
      cache.stopService();
      cache = null;
   }


    /** Executes a lot of put() operations against the cache */
    public void testPuts() throws Exception
   {
        UserTransaction tx=null;
        log("=== 1 cache with transaction (no concurrent access) many puts ===");
        cache1_ = createCache();
        cache1_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
        cache1_.setCacheMode(TreeCache.REPL_SYNC);
        cache1_.startService();

        tx=(UserTransaction) new InitialContext(p_).lookup("UserTransaction");

        int nOps = 1000, realOps=0;
        long time1 = System.currentTimeMillis();

        try {
            for(int i=0; i < nOps; i++) {
                tx.begin();
                cache1_.put("/bela/ban", "name", "Bela Ban");
                tx.commit();
                realOps++;
            }
        }
        catch(Throwable t) {
            if(tx != null)
                tx.rollback();
        }
        long time2 = System.currentTimeMillis();
        assertEquals(nOps, realOps);

        double d = (double) (time2 - time1) / realOps;
        log("Time elapsed for _add is " + (time2 - time1) + " with " + realOps
                + " operations. Average per ops is: " + d + " msec.");

       destroyCache(cache1_);
    }

   public void testOneCacheTx() throws Exception
   {
      log("=== 1 cache with transaction (no concurrent access) ===");
      cache1_ = createCache();
      cache1_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache1_.startService();

      // Formating
      DecimalFormat form = new DecimalFormat("#.00");
      FieldPosition fieldPos = new FieldPosition(0);
      StringBuffer dumbStr = new StringBuffer();
      boolean hasTx = true;
      boolean oneTxOnly = false;

      // Step 1. Add entries to the cache
      System.out.println("-- before add: number of locks held is " + cache1_.getNumberOfLocksHeld());
      long time1 = System.currentTimeMillis();
      int nOps = _add(cache1_, hasTx, oneTxOnly);
      long time2 = System.currentTimeMillis();
      System.out.println("-- after add: number of locks held is " + cache1_.getNumberOfLocksHeld());
      double d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _add is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 2. Query the cache
      System.out.println("-- before get: number of locks held is " + cache1_.getNumberOfLocksHeld());
      time1 = System.currentTimeMillis();
      nOps = _get(cache1_, hasTx, oneTxOnly);
      time2 = System.currentTimeMillis();
      System.out.println("-- after get: number of locks held is " + cache1_.getNumberOfLocksHeld());
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _get is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 3. Remove entries from the cache
      System.out.println("-- before remove: number of locks held is " + cache1_.getNumberOfLocksHeld());
      time1 = System.currentTimeMillis();
      nOps = _remove(cache1_, hasTx, oneTxOnly);
      time2 = System.currentTimeMillis();
      System.out.println("-- after remove: number of locks held is " + cache1_.getNumberOfLocksHeld());
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _remove is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");

      destroyCache(cache1_);
   }

   public void test2CachesTx() throws Exception
   {
      log("=== 2 caches with transaction (no concurrent access) ===");
      cache1_ = createCache();
      cache2_ = createCache();
      cache1_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache2_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache1_.startService();
      cache2_.startService();

      // Formating
      DecimalFormat form = new DecimalFormat("#.00");
      FieldPosition fieldPos = new FieldPosition(0);
      StringBuffer dumbStr = new StringBuffer();
      boolean hasTx = true;
      boolean oneTxOnly = false;

      // Step 1. Add entries to the cache
      long time1 = System.currentTimeMillis();
      int nOps = _add(cache1_, hasTx, oneTxOnly);
      long time2 = System.currentTimeMillis();
      double d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _add is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 2. Query the cache
      time1 = System.currentTimeMillis();
      nOps = _get(cache1_, hasTx, oneTxOnly);
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _get is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 3. Remove entries from the cache
      time1 = System.currentTimeMillis();
      nOps = _remove(cache2_, hasTx, oneTxOnly);  // Note we remove nodes from cache2.
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _remove is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");

      destroyCache(cache1_);
      destroyCache(cache2_);
   }

   public void test2CachesOneTxOnly() throws Exception
   {
      log("=== 2 caches with single transaction only (no concurrent access) ===");
      cache1_ = createCache();
      cache2_ = createCache();
      cache1_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache2_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache1_.startService();
      cache2_.startService();

      // Formating
      DecimalFormat form = new DecimalFormat("#.00");
      FieldPosition fieldPos = new FieldPosition(0);
      StringBuffer dumbStr = new StringBuffer();
      boolean hasTx = true;
      boolean oneTxOnly = true;

      // Step 1. Add entries to the cache
      long time1 = System.currentTimeMillis();
      int nOps = _add(cache1_, hasTx, oneTxOnly);
      long time2 = System.currentTimeMillis();
      double d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _add is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 2. Query the cache
      time1 = System.currentTimeMillis();
      nOps = _get(cache1_, hasTx, oneTxOnly);
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _get is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 3. Remove entries from the cache
      time1 = System.currentTimeMillis();
      nOps = _remove(cache2_, hasTx, oneTxOnly);  // Note we remove nodes from cache2.
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _remove is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");

      destroyCache(cache1_);
      destroyCache(cache2_);
   }

   public void test3CachesTx() throws Exception
   {
      log("=== 3 caches with transaction (no concurrent access) ===");
      cache1_ = createCache();
      cache2_ = createCache();
      cache3_ = createCache();
      cache1_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache2_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache2_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache3_.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache1_.startService();
      cache2_.startService();
      cache3_.startService();

      // Formating
      DecimalFormat form = new DecimalFormat("#.00");
      FieldPosition fieldPos = new FieldPosition(0);
      StringBuffer dumbStr = new StringBuffer();
      boolean hasTx = true;
      boolean oneTxOnly = false;

      // Step 1. Add entries to the cache
      long time1 = System.currentTimeMillis();
      int nOps = _add(cache1_, hasTx, oneTxOnly);
      long time2 = System.currentTimeMillis();
      double d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _add is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 2. Query the cache
      time1 = System.currentTimeMillis();
      nOps = _get(cache2_, hasTx, oneTxOnly);   // Note query is from cache2
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _get is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");
      dumbStr = new StringBuffer();

      // Step 3. Remove entries from the cache
      time1 = System.currentTimeMillis();
      nOps = _remove(cache3_, hasTx, oneTxOnly);   // Note remove is from cache3
      time2 = System.currentTimeMillis();
      d = (double) (time2 - time1) / nOps;
      log("Time elapsed for _remove is " + (time2 - time1) + " with " + nOps
            + " operations. Average per ops is: " + form.format(d, dumbStr, fieldPos) +
            " msec.");

      destroyCache(cache1_);
      destroyCache(cache2_);
      destroyCache(cache3_);
   }

   private int _add(TreeCache cache, boolean hasTx, boolean oneTxOnly) throws Exception
   {
      UserTransaction tx = null;
      if (hasTx) {
         tx = (UserTransaction) new InitialContext(p_).lookup("UserTransaction");
      }

      if (hasTx && oneTxOnly) {
         tx.begin();
      }

      for (int i = 0; i < nodeList_.size(); i++) {
         String key = Integer.toString(i);
         String value = Integer.toString(i);
         if (hasTx && !oneTxOnly) {
            tx.begin();
            cache.put((String) nodeList_.get(i), key, value);
            tx.commit();
         } else {
            cache.put((String) nodeList_.get(i), key, value);
         }
      }

      if (hasTx && oneTxOnly) {
         tx.commit();
      }

      return nodeList_.size();
   }

   private int _get(TreeCache cache, boolean hasTx, boolean oneTxOnly) throws Exception
   {
      UserTransaction tx = null;
      if (hasTx) {
         tx = (UserTransaction) new InitialContext(p_).lookup("UserTransaction");
      }

      if (hasTx && oneTxOnly) {
         tx.begin();
      }

      for (int i = 0; i < nodeList_.size(); i++) {
         String key = Integer.toString(i);
         if (hasTx && !oneTxOnly) {
            tx.begin();
            cache.get((String) nodeList_.get(i), key);
            tx.commit();
         } else {
            cache.get((String) nodeList_.get(i), key);
         }
      }

      if (hasTx && oneTxOnly) {
         tx.commit();
      }

      return nodeList_.size();
   }

   private int _remove(TreeCache cache, boolean hasTx, boolean oneTxOnly) throws Exception
   {
      UserTransaction tx = null;
      if (hasTx) {
         tx = (UserTransaction) new InitialContext(p_).lookup("UserTransaction");
      }

      if (hasTx && oneTxOnly) {
         tx.begin();
      }

      for (int i = 0; i < nodeList_.size(); i++) {
         String key = Integer.toString(i);
         if (hasTx && !oneTxOnly) {
            tx.begin();
            cache.remove((String) nodeList_.get(i), key);
            tx.commit();
         } else {
            cache.remove((String) nodeList_.get(i), key);
         }
      }

      if (hasTx && oneTxOnly) {
         tx.commit();
      }

      return nodeList_.size();
   }

   /**
    * Generate the tree nodes quasi-exponentially. I.e., depth is the level
    * of the hierarchy and children is the number of children under each node.
    */
   private ArrayList nodeGen(int depth, int children)
   {
      ArrayList strList = new ArrayList();
      ArrayList oldList = new ArrayList();
      ArrayList newList = new ArrayList();

      oldList.add("/");
      newList.add("/");
      strList.add("/");

      while (depth > 0) {
         // Trying to produce node name at this depth.
         newList = new ArrayList();
         for (int i = 0; i < oldList.size(); i++) {
            for (int j = 0; j < children; j++) {
               String tmp = (String) oldList.get(i);
               tmp += Integer.toString(j);
               if (depth != 1) tmp += "/";
               newList.add(tmp);
            }
         }
         strList.addAll(newList);
         oldList = newList;
         depth--;
      }

      log("Nodes generated: " + strList.size());
      return strList;
   }

   public static Test suite() throws Exception
   {
      return new TestSuite(ReplicatedSyncPerfTestCase.class);
   }

   private void log(String str)
   {
//     System.out.println(this.getClass().getName() +": " +str);
      System.out.println(str);
   }

}
