/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

// $Id: JmxUnitTestCase.java,v 1.1.1.1.4.2 2004/11/05 18:18:08 tdiesler Exp $

package org.jboss.test.jbossnet.jmx;

import com.ibm.wsdl.factory.WSDLFactoryImpl;
import junit.framework.Test;
import org.apache.axis.client.Service;
import org.jboss.net.axis.AxisInvocationHandler;
import org.jboss.net.jmx.MBeanInvocationHandler;
import org.jboss.net.jmx.adaptor.RemoteAdaptor;
import org.jboss.net.jmx.adaptor.RemoteAdaptorInvocationHandler;
import org.jboss.test.jbossnet.JBossNetTestBase;
import org.xml.sax.InputSource;

import javax.management.ObjectName;
import javax.wsdl.Definition;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLReader;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;

/**
 * Tests remote accessibility of JMX services
 * @since 11. Oktober 2001
 * @author <a href="mailto:Christoph.Jung@infor.de">Christoph G. Jung</a>
 * @author <a href="mailto:pbraswell@utopiansoft.com">Peter Braswell</a>
 * @version $Revision: 1.1.1.1.4.2 $
 */
public class JmxUnitTestCase extends JBossNetTestBase
{
   // static that holds the configured Axis jmx name
   private static String AXIS_JMX_NAME = "jboss.net:service=Axis";

   /* Member variables */
   private String JMX_SERVICE_SOAP_ACTION = "JMXTest";
   private String JMX_ADAPTOR_SOAP_ACTION = "RemoteAdaptor";
   private String JMX_DYNAMIC_SERVICE_SOAP_ACTION = "JMXDynamicTest";

   /** the map of methods to interface names */
   private Map interfaceMap = new AxisInvocationHandler.DefaultInterfaceMap();

   /** the map of methods to method names */
   private Map methodMap = new AxisInvocationHandler.DefaultMethodMap();

   // Constructors --------------------------------------------------
   public JmxUnitTestCase(String name)
   {
      super(name);
   }

   /** where the config is stored */
   protected String getAxisConfiguration()
   {
      return "jbossnet/jmx/client/" + super.getAxisConfiguration();
   }

   /** tests a very (untyped) basic call through the normal invocation handler */
   public void testBasic() throws Exception
   {
      log.info("+++ testBasic");
      AxisInvocationHandler handler =
              createAxisInvocationHandler(JMX_ADAPTOR_SOAP_ACTION);
      assertEquals("Testing basic invocation",
              "jboss",
              handler.invoke("RemoteAdaptor", "getDefaultDomain", new Object[0]));
      assertEquals("Testing complex invocation",
              Boolean.TRUE,
              handler.invoke("RemoteAdaptor",
                      "isRegistered",
                      new Object[]{new ObjectName(AXIS_JMX_NAME)}));
   }

   /** tests a very (untyped) basic call through the mbean invocation handler */
   public void testMBeanHandler() throws Exception
   {
      log.info("+++ testMBeanHandler");
      MBeanInvocationHandler handler =
              createMBeanInvocationHandler(JMX_ADAPTOR_SOAP_ACTION);
      assertEquals("Testing mbean specific invocation",
              "jboss",
              handler.invoke("RemoteAdaptor",
                      "getDefaultDomain",
                      new Object[0],
                      new Class[0]));
      assertEquals("Testing custom serializer",
              Boolean.TRUE,
              handler.invoke("RemoteAdaptor",
                      "isRegistered",
                      new Object[]{new ObjectName(AXIS_JMX_NAME)},
                      new Class[]{ObjectName.class}));
   }

   /** tests the (typed) adaptor access */
   public void testAdaptor() throws Exception
   {
      log.info("+++ testAdaptor");
      RemoteAdaptor handler = createRemoteAdaptor(JMX_ADAPTOR_SOAP_ACTION);
      assertEquals("Testing handler", "jboss", handler.getDefaultDomain());
      assertTrue("Testing handler with custom serializer",
              handler.isRegistered(new ObjectName(AXIS_JMX_NAME)));
   }

   public void testGetter() throws Exception
   {
      MBeanInvocationHandler handler =
              createMBeanInvocationHandler(JMX_SERVICE_SOAP_ACTION);
      String str = (String)handler.invoke("JMXTest", // serviceName
              "getTestString", // methodName
              null, // arguments
              null); // classes
      assertEquals(str, "JMX_TEST_STRING");
   }

   public void testDynamicGetter() throws Exception
   {
      MBeanInvocationHandler handler =
              createMBeanInvocationHandler(JMX_DYNAMIC_SERVICE_SOAP_ACTION);
      String str = (String)handler.invoke("JMXDynamicTest", // serviceName
              "getTestString", // methodName
              null, // arguments
              null); // classes
      assertEquals(str, "JMX_TEST_STRING");
   }

   public void testSetter() throws Exception
   {
      MBeanInvocationHandler handler =
              createMBeanInvocationHandler(JMX_SERVICE_SOAP_ACTION);
      handler.invoke("JMXTest", // serviceName
              "setTestString", // methodName
              new String[]{"foo-dog"}, // arguments
              new Class[]{String.class}); // classes
      // invoke the getter and compare the answer with the 
      // set string value
      String str = (String)handler.invoke("JMXTest", // serviceName
              "getTestString", // methodName
              null, // arguments
              null); // classes
      assertEquals(str, "foo-dog");
   }

   public void testSetterDynamic() throws Exception
   {
      MBeanInvocationHandler handler =
              createMBeanInvocationHandler(JMX_DYNAMIC_SERVICE_SOAP_ACTION);
      handler.invoke("JMXDynamicTest", // serviceName
              "setTestString", // methodName
              new String[]{"foo-dog"}, // arguments
              new Class[]{String.class}); // classes
      // invoke the getter and compare the answer with the 
      // set string value
      String str = (String)handler.invoke("JMXDynamicTest", // serviceName
              "getTestString", // methodName
              null, // arguments
              null); // classes
      assertEquals(str, "foo-dog");
   }

   public void testMethodInvoke() throws Exception
   {
      MBeanInvocationHandler handler =
              createMBeanInvocationHandler(JMX_SERVICE_SOAP_ACTION);
      handler.invoke("jboss.net:service=JMXTestMBean", // serviceName
              "noopOperation", // methodName
              null, // arguments
              null); // classes
   }

   public void testDynamicMethodInvoke() throws Exception
   {
      MBeanInvocationHandler handler =
              createMBeanInvocationHandler(JMX_DYNAMIC_SERVICE_SOAP_ACTION);
      handler.invoke("jboss.net:service=JMXDynamicTestMBean", // serviceName
              "noopOperation", // methodName
              null, // arguments
              null); // classes
   }

   /** tests wsdl access */
   public void checkWsdlOf(String service) throws Exception
   {
      // check if the WSDL is available
      URL urlWSDL = new URL(SERVICES_LOCATION + "/" + service + "?wsdl");
      InputStream isWSDL = urlWSDL.openStream();

      // check if it is a valid WSDL
      WSDLFactory wsdlFactory = new WSDLFactoryImpl();
      WSDLReader wsdlReader = wsdlFactory.newWSDLReader();
      Definition wsdlDefinition =
              wsdlReader.readWSDL(null, new InputSource(isWSDL));
      assertNotNull("cannot obtain the wsdl", wsdlDefinition);
   }

   /** tests wsdl access */
   public void testWsdl() throws Exception
   {
      checkWsdlOf(JMX_SERVICE_SOAP_ACTION);
      checkWsdlOf(JMX_DYNAMIC_SERVICE_SOAP_ACTION);
   }

   /** creates a new service */
   private Service createService()
   {
      Service result = new Service();
      result.setMaintainSession(true);
      return result;
   }

   /** creates a new Axis service using the test engine*/
   private AxisInvocationHandler createAxisInvocationHandler(String endpoint, String soapAction)
           throws Exception
   {
      return new AxisInvocationHandler(new URL(endpoint),
              soapAction,
              createService(),
              methodMap,
              interfaceMap);
   }

   /** creates a new Axis service using the test engine*/
   private AxisInvocationHandler createAxisInvocationHandler(String soapAction)
           throws Exception
   {
      return createAxisInvocationHandler(SERVICES_LOCATION,
              soapAction);
   }

   /** creates a new Axis service using the test engine*/
   private MBeanInvocationHandler createMBeanInvocationHandler(String soapAction)
           throws Exception
   {
      return new MBeanInvocationHandler(new URL(SERVICES_LOCATION),
              soapAction,
              createService(),
              methodMap,
              interfaceMap);
   }

   /** creates a new remote adaptor using the test engine*/
   private RemoteAdaptor createRemoteAdaptor(String soapAction)
           throws Exception
   {
      return RemoteAdaptorInvocationHandler.createRemoteAdaptor(createMBeanInvocationHandler(soapAction));
   }

   /**
    * Method suite, deploys additional jmx bean+web service
    * @return
    * @throws Exception
    */

   public static Test suite() throws Exception
   {
      return getDeploySetup(JmxUnitTestCase.class, "jbossnet-jmx.sar");
   }
}
