/*
 * JBoss, the OpenSource WebOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.web.tomcat.tc4;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;
import java.security.ProtectionDomain;

import javax.management.NotificationListener;
import javax.management.Notification;

import org.apache.catalina.Logger;
import org.apache.catalina.startup.Embedded;

import org.jboss.deployment.DeploymentInfo;
import org.jboss.web.AbstractWebContainer;
import org.jboss.web.AbstractWebDeployer;
import org.jboss.web.tomcat.statistics.InvocationStatistics;
import org.jboss.web.tomcat.Log4jLogger;
import org.jboss.web.tomcat.security.JBossSecurityMgrRealm;
import org.jboss.system.server.ServerImplMBean;
import org.jboss.system.server.Server;

import org.w3c.dom.Element;


/** An implementation of the AbstractWebContainer for the Jakarta Tomcat
 4.1 servlet container. This uses the org.apache.catalina.startup.Embedded as
 the integration class. It does not parse the catalina server.xml in the
 catalina distribution. Rather, it parses a subset of the server.xml syntax
 and obtains this configuration information from the Config attribute.

 @see org.jboss.web.AbstractWebContainer
 @see org.apache.catalina.startup.Embedded

 @author Scott.Stark@jboss.org
 @version $Revision: 1.1.1.1.2.9 $
 */
public class EmbeddedTomcatService extends AbstractWebContainer
   implements EmbeddedTomcatServiceMBean, NotificationListener
{
   // Constants -----------------------------------------------------
   public static final String NAME = "EmbeddedCatalina4.1.x";

   /** The embedded instance used to configure catalina */
   private EmbeddedCatalina catalina;
   /** The catalina debug level */
   private int debugLevel;
   /** The value to use for the catalina.home System property */
   private String catalinaHome;
   /** The value to use for the catalina.base System property */
   private String catalinaBase;
   /** A flag indicating if the JBoss Loader should be used */
   private boolean useJBossWebLoader = true;
   /** Any extended configuration information specified via a config
    element in the mbean definition.
    */
   private Element extendedConfig;
   /** A flag indicating if the working dir for a war deployment should be
    delete when the war is undeployed.
    */
   private boolean deleteWorkDirs = true;
   /** Which snapshot mode should be used in clustered environment?
    Default: instant
    */
   private String snapshotMode = "instant"; // instant or interval
   /** With IntervalSnapshotManager use this interval (in ms) for snapshotting */
   private int snapshotInterval = 1000;

   /** Should the clustering code use a local cache for the sessions? */
   private boolean useLocalCache = true;

   /** Get the request attribute name under which the JAAS Subject is store */
   private String subjectAttributeName = null;

   /** The web ctx invocation statistics */
   private InvocationStatistics stats = new InvocationStatistics();

   public EmbeddedTomcatService()
   {
   }

   public String getName()
   {
      return NAME;
   }

   public String getCatalinaHome()
   {
      return this.catalinaHome;
   }

   public void setCatalinaHome(String catalinaHome)
   {
      this.catalinaHome = catalinaHome;
   }

   public String getCatalinaBase()
   {
      return this.catalinaBase;
   }

   public void setCatalinaBase(String catalinaBase)
   {
      this.catalinaBase = catalinaBase;
   }

   public boolean getUseJBossWebLoader()
   {
      return useJBossWebLoader;
   }

   public void setUseJBossWebLoader(boolean flag)
   {
      this.useJBossWebLoader = flag;
   }

   /** Get the delete work dirs on undeployment flag.
    @see #setDeleteWorkDirs(boolean)
    */
   public boolean getDeleteWorkDirs()
   {
      return this.deleteWorkDirs;
   }

   /** Set the delete work dirs on undeployment flag. By default catalina
    does not delete its working directories when a context is stopped and
    this can cause jsp pages in redeployments to not be recompiled if the
    timestap of the file in the war has not been updated. This defaults to true.
    */
   public void setDeleteWorkDirs(boolean flag)
   {
      this.deleteWorkDirs = flag;
   }

   /** Set the snapshot mode. Currently supported: instant or interval */
   public void setSnapshotMode(String mode)
   {
      this.snapshotMode = mode;
   }
   /** Get the snapshot mode */
   public String getSnapshotMode()
   {
      return this.snapshotMode;
   }

   /** Set the snapshot interval in milliseconds for snapshot mode = interval */
   public void setSnapshotInterval(int interval)
   {
      this.snapshotInterval = interval;
   }
   /** Get the snapshot interval */
   public int getSnapshotInterval()
   {
      return this.snapshotInterval;
   }

   public boolean isUseLocalCache()
   {
      return useLocalCache;
   }

   public void setUseLocalCache(boolean useLocalCache)
   {
      this.useLocalCache = useLocalCache;
   }

   public String getSubjectAttributeName()
   {
      return this.subjectAttributeName;
   }
   public void setSubjectAttributeName(String name)
   {
      this.subjectAttributeName = name;
   }

   public Element getConfig()
   {
      return this.extendedConfig;
   }

   /** This method is invoked to import an arbitrary XML configuration tree.
    Subclasses should override this method if they support such a configuration
    capability. This implementation does nothing.
    */
   public void setConfig(Element config)
   {
      this.extendedConfig = config;
   }

   /** Get the active thread count */
   public int getActiveThreadCount()
   {
      return stats.concurrentCalls;
   }

   /** Get the maximal active thread count */
   public int getMaxActiveThreadCount()
   {
      return stats.maxConcurrentCalls;
   }

   public InvocationStatistics getStats()
   {
      return stats;
   }

   public void resetStats()
   {
      stats.resetStats();
   }

   public void startService() throws Exception
   {
      // Start create the embeded catalina container but don't let it overwrite the thread class loader
      ClassLoader cl = Thread.currentThread().getContextClassLoader();
      ClassLoader parent = cl;
      while (parent != null)
      {
         log.trace(parent);
         URL[] urls = super.getClassLoaderURLs(parent);
         for (int u = 0; u < urls.length; u++)
            log.trace("  " + urls[u]);
         parent = parent.getParent();
      }

      // Determine the catalina debug level from the enabled priority
      debugLevel = 0;
      if (log.isTraceEnabled())
         debugLevel = 3;
      log.debug("Setting catalina debug level to: " + debugLevel);

      try
      {
         // Set the catalina.home property from the Embedded class location
         if (catalinaHome == null)
         {
            ProtectionDomain pd = Embedded.class.getProtectionDomain();
            URL homeURL = pd.getCodeSource().getLocation();
            String homePath = homeURL.getFile();
            File homeDir = new
               File(homePath).getParentFile().getParentFile().getParentFile();
            catalinaHome = homeDir.getCanonicalPath();
         }
         if (catalinaBase == null)
            catalinaBase = catalinaHome;
         log.debug("Setting catalina.home to: " + catalinaHome);
         log.debug("Setting catalina.base to: " + catalinaBase);
         System.setProperty("catalina.home", catalinaHome);
         System.setProperty("catalina.base", catalinaBase);
         initCatalina(cl);
         catalina.start();
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(cl);
      }
      log.info("OK");

      // Register for notification of the overall server startup
      server.addNotificationListener(ServerImplMBean.OBJECT_NAME, this, null, null);

      // Invoke the super method to register as a deployer
      super.startService();
   }

   public void stopService() throws Exception
   {
      super.stopService();
      if (catalina != null)
      {
         catalina.stop();
      }
   }

   public void startConnectors() throws Exception
   {
      catalina.startConnectors();
   }
   public void stopConnectors() throws Exception
   {
      catalina.stopConnectors();      
   }

   public void handleNotification(Notification msg, Object handback)
   {
      String type = msg.getType();
      if( type.equals(Server.START_NOTIFICATION_TYPE) )
      {
         log.info("Saw "+type+" notification, starting connectors");
         try
         {
            startConnectors();
         }
         catch(Exception e)
         {
            log.warn("Failed to startConnectors", e);
         }
      }
   }

   /** Create and configure a org.apache.catalina.startup.Embedded
    instance. We do not use the server.xml file as we obtain all
    of the required customization from our mbean properties.
    */
   private void initCatalina(ClassLoader parent) throws Exception
   {
      Logger jbossLog = new Log4jLogger(this.log);
      JBossSecurityMgrRealm jbossRealm = new JBossSecurityMgrRealm();
      catalina = new EmbeddedCatalina(jbossLog, jbossRealm);
      catalina.setDebug(debugLevel);
      catalina.setUseNaming(false);

      // Deploy default web.xml to catalina.home, if running from a .sar
      try
      {
         ClassLoader cl = Thread.currentThread().getContextClassLoader();
         InputStream is = cl.getResourceAsStream("web.xml");
         if (is != null)
         {
            File confDir = new File(catalinaHome, "conf");
            confDir.mkdirs();
            File webXml = new File(catalinaHome, "conf/web.xml");
            FileOutputStream os = new FileOutputStream(webXml);
            byte[] buf = new byte[512];
            while (true)
            {
               int n = is.read(buf);
               if (n < 0)
               {
                  break;
               }
               os.write(buf, 0, n);
            }
            os.close();
            is.close();
         }
         else
         {
            log.info("Assuming Tomcat standalone");
         }
      }
      catch (Exception e)
      {
         // Ignore
      }

      // Apply any extended configuration
      ConfigHandler handler = new ConfigHandler(log);
      handler.applyHostConfig(extendedConfig, catalina, debugLevel > 0);
   }

   public AbstractWebDeployer getDeployer(DeploymentInfo di) throws Exception
   {
      ClassLoader loader = Thread.currentThread().getContextClassLoader();
      Class deployerClass = loader.loadClass("org.jboss.web.tomcat.tc4.TomcatDeployer");
      AbstractWebDeployer deployer = (AbstractWebDeployer) deployerClass.newInstance();
      DeployerConfig config = new DeployerConfig();
      config.setJava2ClassLoadingCompliance(this.java2ClassLoadingCompliance);
      config.setUnpackWars(this.unpackWars);
      config.setLenientEjbLink(this.lenientEjbLink);
      config.setCatalina(catalina);
      config.setDebugLevel(debugLevel);
      config.setDeleteWorkDirs(deleteWorkDirs);
      config.setServiceName(serviceName);
      config.setSnapshotInterval(this.snapshotInterval);
      config.setSnapshotMode(this.snapshotMode);
      config.setUseLocalCache(this.useLocalCache);
      config.setStats(stats);
      config.setSubjectAttributeName(this.subjectAttributeName);
      config.setUseJBossWebLoader(this.useJBossWebLoader);
      deployer.setServer(server);
      deployer.init(config);
      return deployer;
   }
}
