/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
 
package org.jboss.monitor.alarm;

import javax.management.Notification;

/**
 * AlarmNotification
 *
 * @author  <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 *
 * @version $Revision: 1.1.2.3 $
**/
public class AlarmNotification
   extends Notification
{
   // Private Data --------------------------------------------------

   /** the alarm state */
   private int alarmState;
   
   /** the alarm severity */
   private int severity;
   
   // CTORS ---------------------------------------------------------
   
   /**
    * Complete CTOR, creates an AlarmNotification object
    *
    * Note:
    *   STATE_CLEARED forces severity to SEVERITY_NORMAL
    *   STATE_CREATED/CHANGED have valid severities WARNING to UNKNOWN
    *   STATE_NONE has valid severities NORMAL to UNKNOWN
    * Also:
    *   Out-of-range states are mapped to STATE_NONE
    *   Out-of-range severities are mapped to SEVERITY_UNKNOWN
   **/
   public AlarmNotification(
      String type, Object source,
      long sequenceNumber, long timeStamp, String message,
      int alarmState, int severity
   )
   {
      super(type, source, sequenceNumber, timeStamp, message);
      
      switch (alarmState) {
         
         case Alarm.STATE_CLEARED:
            this.alarmState = Alarm.STATE_CLEARED;
            // forces severity=SEVERITY_NORMAL
            this.severity = Alarm.SEVERITY_NORMAL;
            break;
            
         case Alarm.STATE_CREATED:
         case Alarm.STATE_CHANGED:
            this.alarmState = alarmState;
            // can't have SEVERITY_NORMAL!
            if (severity > Alarm.SEVERITY_NORMAL && severity <= Alarm.SEVERITY_UNKNOWN)
               this.severity = severity;
            else // handle out of range severity as SEVERITY_UNKNOWN
               this.severity = Alarm.SEVERITY_UNKNOWN;
            break;            

         case Alarm.STATE_NONE:
         default: // handle out of range alarmState as STATE_NONE
            this.alarmState = Alarm.STATE_NONE;
            if (severity >= Alarm.SEVERITY_NORMAL && severity <= Alarm.SEVERITY_UNKNOWN)
               this.severity = severity;
            else // handle out of range severity as SEVERITY_UNKNOWN
               this.severity = Alarm.SEVERITY_UNKNOWN;
            break;            
      }
   }
   
   // Accessors -----------------------------------------------------
   
   /**
    * Gets alarm severity
   **/
   public int getSeverity()
   {
      return this.severity;
   }
   
   /**
    * Gets alarm state
   **/
   public int getAlarmState()
   {
      return this.alarmState;
   }
   
   // Object stuff --------------------------------------------------
   
   /**
    * toString()
   **/
   public String toString()
   {
      StringBuffer sbuf = new StringBuffer(256);
      
      sbuf.append(AlarmNotification.class.getName());
      sbuf.append(" [ source=").append(this.getSource());
      sbuf.append(", type=").append(this.getType());
      sbuf.append(", sequenceNumber=").append(this.getSequenceNumber());
      sbuf.append(", timeStamp=").append(this.getTimeStamp());
      sbuf.append(", message=").append(this.getMessage());
      sbuf.append(", userData=").append(this.getUserData());
      sbuf.append(", alarmState=").append(Alarm.STATE_STRINGS[this.alarmState]);
      sbuf.append(", severity=").append(Alarm.SEVERITY_STRINGS[this.severity]);
      sbuf.append(" ]");
      
      return sbuf.toString();
   }
}
