package org.jboss.cache.interceptors;

import org.jgroups.blocks.MethodCall;
import org.jboss.cache.TreeCache;
import org.jboss.cache.GlobalTransaction;
import org.jboss.cache.TransactionTable;

import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

/**
 * Always at the end of the chain, directly in front of the cache. Simply calls into the cache using reflection.
 * If the call resulted in a modification, add the Modification to the end of the modification list
 * keyed by the current transaction
 * @author Bela Ban
 * @version $Id: CallInterceptor.java,v 1.1.4.3 2004/12/30 17:08:24 starksm Exp $
 */
public class CallInterceptor extends Interceptor {
   TransactionManager tx_mgr=null;
   TransactionTable   tx_table=null;


   public void setCache(TreeCache cache) {
      super.setCache(cache);
      tx_mgr=cache.getTransactionManager();
      tx_table=cache.getTransactionTable();
   }

   public Object invoke(MethodCall m) throws Throwable {
      Object retval=null;
      Transaction tx=null;

      if(tx_mgr != null)
         tx=tx_mgr.getTransaction();

      try {
         retval=m.invoke(cache);
      }
      catch(Throwable t) {
         retval=t;
      }

      if(tx != null && (isActive(tx) || isPreparing(tx))) {
         if(retval != null && retval instanceof Throwable) {
            tx.setRollbackOnly(); // no exception, regular return
         }
         else {
            if(TreeCache.isCrudMethod(m.getMethod())) {
               // if method is a CRUD (Create/Remove/Update/Delete) method: add it to the modification
               // list, otherwise skip (e.g. get() is not added)
               // add the modification to the TX's modification list. this is used to later
               // (on TX commit) send all modifications done in this TX to all members
               GlobalTransaction gtx=tx_table.get(tx);
               if(gtx == null) {
                  if(log.isDebugEnabled())
                     log.debug("didn't find GlobalTransaction for " + tx + "; won't add modification to transaction list");
               }
               else {
                  tx_table.addModification(gtx, m);
               }
            }
         }
      }
      if(retval != null && retval instanceof Throwable)
         throw (Throwable)retval;
      return retval;
   }
}
