// $Id: LockingPolicy.java,v 1.1 2003/10/30 01:21:19 bwang00 Exp $

/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 * Created on Jan 18, 2003
 */
package org.jboss.cache;

import org.jgroups.blocks.Xid;

/**
 * Defines how read or write-locks are acquired. Implementations will typically
 * maintain some form of locking table, where they associate transactions and
 * the locks acquired by a transaction.
 * @author Bela Ban
 * @version $Revision: 1.1 $
 */
public interface LockingPolicy {

    /** Sets the cache on which locks are acquired */
    void setCache(CacheImpl cache);

    /**
     * Acquires a read lock for a given transaction. If <tt>dirty</tt> is true,
     * then no lock needs to be acquired.
     * @param transaction The transaction with which the read-lock is to be
     * associated. If the transaction already holds a read or write-lock, no new
     * lock will be acquired, but we will proceed immediately
     * @param dirty Whether of not a lock needs to be acquired
     * @param lock_acquisition_timeout Number of milliseconds within which the
     * lock has to be acquired. A value of <=0 means to wait forever.
     * @throws LockingException Thrown if the lock cannot be acquired within
     * <tt>lock_acquisition_timeout</tt> milliseconds
     */
    void getReadLock(Xid transaction, boolean dirty, long lock_acquisition_timeout)
            throws LockingException;

    /**
     *
     * @param transaction The transaction with which the write-lock is to be
     * associated. If the transaction already holds a write-lock, no new lock
     * will be acquired, but we will proceed immediately. If the transaction
     * already holds a read-lock, the read-lock will have to be upgraded to a
     * write-lock. If this is not possible, an UpgradeException will need to be
     * thrown.
     * @param lock_acquisition_timeout Number of milliseconds within which the
     * lock has to be acquired. A value of <=0 means to wait forever.
     * @param entire_map Whether or not the entire map needs to be locked. This
     * will only happen on {@link Cache#putAll(java.util.Map)}.
     * @throws LockingException Thrown if the lock cannot be acquired within
     * <tt>lock_acquisition_timeout</tt> milliseconds
     * @throws UpgradeException Thrown when a read-lock cannot be upgraded to a
     * write-lock
     */
    void getWriteLock(Xid transaction, long lock_acquisition_timeout, boolean entire_map)
            throws LockingException, UpgradeException;

    /**
     * Releases all locks held by a given transaction
     * @param transaction The transaction
     */
    void releaseLocks(Xid transaction);
}
