/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
package org.jboss.resource.adapter.jms;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.StreamMessage;
import javax.jms.TextMessage;

/**
 * A wrapper for a message consumer
 *
 * @author <a href="mailto:adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.2.2.1 $
 */
public class JmsMessageConsumer implements MessageConsumer
{
   // Constants -----------------------------------------------------
   
   // Attributes ----------------------------------------------------

   /** The wrapped message consumer */
   MessageConsumer consumer;
   
   /** The session for this consumer */
   JmsSession session;
   
   // Static --------------------------------------------------------
   
   // Constructors --------------------------------------------------

   /**
    * Create a new wrapper
    * 
    * @param consumer the consumer
    * @param session the session
    */
   public JmsMessageConsumer(MessageConsumer consumer, JmsSession session)
   {
      this.consumer = consumer;
      this.session = session;
   }
   
   // Public --------------------------------------------------------
   
   // MessageConsumer implementation --------------------------------

   public void close() throws JMSException
   {
      try
      {
         closeConsumer();
      }
      finally
      {
         session.removeConsumer(this);
      }
   }
   
   public MessageListener getMessageListener() throws JMSException
   {
      session.checkStrict();
      return consumer.getMessageListener();
   }
   
   public String getMessageSelector() throws JMSException
   {
      return consumer.getMessageSelector();
   }
   
   public Message receive() throws JMSException
   {
      Message message = consumer.receive();
      if (message == null)
         return null;
      else
         return wrapMessage(message);
   }

   public Message receive(long timeout) throws JMSException
   {
      Message message = consumer.receive(timeout);
      if (message == null)
         return null;
      else
         return wrapMessage(message);
   }

   public Message receiveNoWait() throws JMSException
   {
      Message message = consumer.receiveNoWait();
      if (message == null)
         return null;
      else
         return wrapMessage(message);
   }
   
   public void setMessageListener(MessageListener listener) throws JMSException
   {
      session.checkStrict();
      if (listener == null)
         consumer.setMessageListener(null);
      else
         consumer.setMessageListener(wrapMessageListener(listener));
   }
   
   // Package protected ---------------------------------------------

   void closeConsumer() throws JMSException
   {
      consumer.close();
   }
   
   Message wrapMessage(Message message)
   {
      if (message instanceof BytesMessage)
         return new JmsBytesMessage((BytesMessage) message, session);
      else if (message instanceof MapMessage)
         return new JmsMapMessage((MapMessage) message, session);
      else if (message instanceof ObjectMessage)
         return new JmsObjectMessage((ObjectMessage) message, session);
      else if (message instanceof StreamMessage)
         return new JmsStreamMessage((StreamMessage) message, session);
      else if (message instanceof TextMessage)
         return new JmsTextMessage((TextMessage) message, session);
      return new JmsMessage(message, session);
   }
   
   MessageListener wrapMessageListener(MessageListener listener)
   {
      return new JmsMessageListener(listener, this);
   }
   
   // Protected -----------------------------------------------------
   
   // Private -------------------------------------------------------
   
   // Inner classes -------------------------------------------------
}
