/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.axis.encoding.ser;

import org.apache.axis.utils.Messages;

import javax.xml.namespace.QName;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.util.SimpleTimeZone;

/**
 * The DateSerializer deserializes a Date.  Much of the work is done in the
 * base class.
 *
 * @author Sam Ruby (rubys@us.ibm.com)
 *         Modified for JAX-RPC @author Rich Scheuerle (scheu@us.ibm.com)
 */
public class DateDeserializer extends SimpleDeserializer
{

   private static SimpleDateFormat zulu =
           new SimpleDateFormat("yyyy-MM-dd");
           
   private static SimpleDateFormat zuluUTC =
           new SimpleDateFormat("yyyy-MM-dd");
   //  0123456789 0 123456789

   static
   {
      zuluUTC.setTimeZone(TimeZone.getTimeZone("GMT"));
   }

   /**
    * The Deserializer is constructed with the xmlType and
    * javaType
    */
   public DateDeserializer(Class javaType, QName xmlType)
   {
      super(javaType, xmlType);
   }

   // Adjusts the passed in date by an xsd:date zone string and returns the zone offset
   private int adjustByZone(Date date, String zone)
   {
      int sign, hours, minutes;

      if (zone.charAt(0) == 'Z')
      {
         return 0;
      }
      else if (zone.charAt(0) == '-')
      {
         // The sign is the inverse since we are moving from UTC to the timezone
         sign = 1;
      }
      else if (zone.charAt(0) == '+')
      {
         sign = -1;
      }
      else
      {
         throw new NumberFormatException(Messages.getMessage("badTimezone00")); 
      }
      
      if (zone.charAt(3) != ':')
      {
         throw new NumberFormatException(Messages.getMessage("badTimezone00")); 
      }
     
      try
      {
         hours = Integer.parseInt(zone.substring(1, 3));
         minutes = Integer.parseInt(zone.substring(4, 6));
      }
      catch (Exception e)
      {
         throw new NumberFormatException(Messages.getMessage("badTimezone00")); 
      }

      // 3.2.7.3 - magnitude
      if (hours > 14 || (hours == 14 && minutes > 0) || minutes > 59)
      {
         throw new NumberFormatException(Messages.getMessage("badTimezone00")); 
      }

      int offset = sign * (hours * 60 + minutes) * 60 * 1000;

      date.setTime(date.getTime() + offset);

      // We resture the zone since we are now moving from the timezone to UTC
      return -offset;
   }
      
   /**
    * The simple deserializer provides most of the stuff.
    * We just need to override makeValue().
    */
   public Object makeValue(String source)
   {
      Calendar calendar;
      Date result;
      boolean bc = false;
      String zonePortion = null;

      if (source == null)
      {
         throw new NumberFormatException(Messages.getMessage("badDate00"));
      }
      
      if (source.charAt(0) == '+')
         source = source.substring(1);

      if (source.charAt(0) == '-')
      {
         source = source.substring(1);
         bc = true;
      }

      if (source.length() < 10)
         throw new NumberFormatException(Messages.getMessage("badDate00"));

      if (source.charAt(4) != '-' || source.charAt(7) != '-')
         throw new NumberFormatException(Messages.getMessage("badDate00"));

      try
      {
         if (source.length() > 10) 
         {
            zonePortion = source.substring(10);

            synchronized (zuluUTC)
            {
               result = zuluUTC.parse(source.substring(0, 10));
            }
            int offset = adjustByZone(result, zonePortion);
            TimeZone timeZone = (offset == 0) ? TimeZone.getTimeZone("GMT")
                                              : new SimpleTimeZone(offset, "GMT offset");
            calendar = Calendar.getInstance(timeZone);
         }
         else
         {
            synchronized (zulu)
            {
               result = zulu.parse(source);
            }
            calendar = Calendar.getInstance();
         }
      }
      catch (NumberFormatException e)
      {
         throw e;
      }
      catch (Exception e)
      {
         throw new NumberFormatException(e.toString());
      }

      calendar.setTime(result);

      // support dates before the Christian era
      if (bc)
      {
         calendar.set(Calendar.ERA, GregorianCalendar.BC);
         result = calendar.getTime();
      }

      if (super.javaType == Calendar.class)
      {
         return calendar;
      }

      return result;
   }
}
