/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 *
 * Created on Dec 29, 2003
 */
package org.jboss.net.axis.transport.mailto;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.activation.DataSource;

import org.apache.axis.AxisFault;
import org.apache.axis.Message;
import org.apache.axis.MessageContext;
import org.apache.axis.message.SOAPEnvelope;
import org.apache.axis.soap.SOAPConstants;
import org.apache.log4j.Logger;


/**
 * <dl>
 * <dt><b>Title: </b><dd>SOAP 1.2 DataSource</dd>
 * <p>
 * <dt><b>Description: </b><dd>Datasource capable of handling SOAP 1.2 messages with the application/soap+xml
 * content type.</dd>
 * <p>
 * </dl>
 * @author <a href="mailto:jasone@greenrivercomputing.com">Jason Essington</a>
 * @version $Revision: 1.1.4.1 $
 */
public class SOAPDataSource implements DataSource
{
   private Logger log = Logger.getLogger(getClass());
   private Message soapMessage = null;

   /**
    * constructor to create a response message (one with no action parameter)
    * @param msg soapMessage
    */
   public SOAPDataSource(Message msg)
   {
      this.soapMessage = msg;
   }

   /* (non-Javadoc)
    * @see javax.activation.DataSource#getInputStream()
    */
   public InputStream getInputStream() throws IOException
   {
      /*
       * This method will get the whole soap message, but this datasource is really only set up to handle the 
       * soap envelope sans attachments.
       * TODO handle attachments
       */
      return new ByteArrayInputStream(this.soapMessage.getSOAPPartAsBytes());
   }

   /**
    * NOT IMPLEMENTED!
    */
   public OutputStream getOutputStream() throws IOException
   {
      throw new IOException("getOutputStream() is not implemented in SOAP12DataSource.");
   }

   /* (non-Javadoc)
    * @see javax.activation.DataSource#getContentType()
    */
   public String getContentType()
   {
      SOAPEnvelope envelope = null;
      try
      {
         envelope = soapMessage.getSOAPEnvelope();
      } catch (AxisFault e)
      {
         log.warn("Unable to get SOAPEnvelope from the SOAPMessage.", e);
      }
      
      boolean soap12 = (envelope != null && envelope.getSOAPConstants() == SOAPConstants.SOAP12_CONSTANTS);
       
      MessageContext mctx = this.soapMessage.getMessageContext();
      SOAPConstants sc = mctx.getSOAPConstants();

      String ct = "text/xml";
      try
      {
         ct = soapMessage.getContentType(sc);
      } catch (AxisFault e)
      {
         log.warn("Failed to determine content type from the SOAPMessage.", e);
         ct = sc.getContentType();
      } 
      
      if (soap12)
      {
         // we need to add the action parameter to the content type for SOAP 1.2 requests
         if (mctx.useSOAPAction())
         {
            ct += ct +"; action=\""+mctx.getSOAPActionURI()+"\"";
         }
      }
      
      return ct;
   }

   /* (non-Javadoc)
    * @see javax.activation.DataSource#getName()
    */
   public String getName()
   {
      // TODO return something more intelligent here
      return "SOAP Message";
   }

}
