/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.mx.loading;

import EDU.oswego.cs.dl.util.concurrent.ConcurrentReaderHashMap;
import org.jboss.logging.Logger;
import org.jboss.mx.server.ServerConstants;
import org.jboss.util.loading.Translator;

import java.net.URL;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Vector;
import javax.management.loading.ClassLoaderRepository;

/**
 *
 * @see javax.management.loading.DefaultLoaderRepository
 * @see org.jboss.mx.loading.BasicLoaderRepository
 *
 * @author  <a href="mailto:juha@jboss.org">Juha Lindfors</a>.
 * @author Scott.Stark@jboss.org
 * @version $Revision: 1.4.4.8 $
 */
public abstract class LoaderRepository
   implements ServerConstants, ClassLoaderRepository
{

   // Attributes ----------------------------------------------------
   protected static Vector loaders = new Vector();
   protected static LoaderRepository instance = null;
   protected Translator translator = null;

   /** The loaded classes cache, HashMap<String, Class>.
    * Access synchronized via this.classes monitor.
    */
   private ConcurrentReaderHashMap classes = new ConcurrentReaderHashMap();


   /**
    * Native signature to class map
    */
   protected static final HashMap nativeClassBySignature;

   // Static --------------------------------------------------------
   private static final Logger log = Logger.getLogger(LoaderRepository.class);

   /**
    * Construct the native class map
    */
   static
   {
     nativeClassBySignature = new HashMap();
     nativeClassBySignature.put(boolean.class.getName(),
                                Boolean.TYPE);
     nativeClassBySignature.put(byte.class.getName(),
                                Byte.TYPE);
     nativeClassBySignature.put(char.class.getName(),
                                Character.TYPE);
     nativeClassBySignature.put(double.class.getName(),
                                Double.TYPE);
     nativeClassBySignature.put(float.class.getName(),
                                Float.TYPE);
     nativeClassBySignature.put(int.class.getName(),
                                Integer.TYPE);
     nativeClassBySignature.put(long.class.getName(),
                                Long.TYPE);
     nativeClassBySignature.put(short.class.getName(),
                                Short.TYPE);
     nativeClassBySignature.put(void.class.getName(),
                                Void.TYPE);
   }

   public synchronized static LoaderRepository getDefaultLoaderRepository()
   {

      if (instance != null)
         return instance;

      ClassLoader cl = Thread.currentThread().getContextClassLoader();
      String className = System.getProperty(LOADER_REPOSITORY_CLASS_PROPERTY, DEFAULT_LOADER_REPOSITORY_CLASS);
      System.setProperty(LOADER_REPOSITORY_CLASS_PROPERTY, className);

      try
      {
         Class repository = cl.loadClass(className);
         instance = (LoaderRepository)repository.newInstance();

         return instance;
      }
      catch (ClassNotFoundException e)
      {
         throw new Error("Cannot instantiate default loader repository class. Class " + className + " not found.");
      }
      catch (ClassCastException e)
      {
         throw new Error("Cannot instantiate default loader repository class. The target class is not an instance of LoaderRepository interface.");
      }
      catch (Exception e)
      {
         throw new Error("Error creating default loader repository: " + e.toString());
      }
   }

   // Public --------------------------------------------------------
   public Vector getLoaders()
   {
      return loaders;
   }

   public URL[] getURLs()
   {
      return null;
   }

   public Class getCachedClass(String classname)
   {
       return (Class)classes.get(classname);
   }

   public Translator getTranslator()
   {
      return translator;
   }

   public void setTranslator(Translator t)
   {
      translator = t;
   }

   /** Create UnifiedClassLoader and optionally add it to the repository
    * @param url the URL to use for class loading
    * @param addToRepository a flag indicating if the UCL should be added to
    *    the repository
    * @return the UCL instance
    * @throws Exception
    */
   public abstract UnifiedClassLoader newClassLoader(final URL url, boolean addToRepository)
      throws Exception;
   /** Create UnifiedClassLoader and optionally add it to the repository
    * @param url the URL to use for class loading
    * @param origURL an orignal URL to use as the URL for the UCL CodeSource.
    * This is useful when the url is a local copy that is difficult to use for
    * security policy writing.
    * @param addToRepository a flag indicating if the UCL should be added to
    *    the repository
    * @return the UCL instance
    * @throws Exception
    */
   public abstract UnifiedClassLoader newClassLoader(final URL url, final URL origURL,
      boolean addToRepository)
      throws Exception;

   /** Load the given class from the repository.
    * @param className
    * @return
    * @throws ClassNotFoundException
    */
   public abstract Class loadClass(String className) throws ClassNotFoundException;

   /** 
    * Load the given class from the repository using classloaders before this classloader.
    * 
    * @param cl the classloader
    * @param className the class name
    * @return the class
    * @throws ClassNotFoundException for not found
    */
   public Class loadClassBefore(ClassLoader cl, String className) throws ClassNotFoundException
   {
      throw new UnsupportedOperationException("NYI");
   }

   /** Load the given class from the repository
    * @param name
    * @param resolve
    * @param cl
    * @return
    * @throws ClassNotFoundException
    */
   public abstract Class loadClass(String name, boolean resolve, ClassLoader cl)
      throws ClassNotFoundException;

   /** Find a resource URL for the given name
    *
    * @param name the resource name
    * @param cl the requesting class loader
    * @return The resource URL if found, null otherwise
    */
   public abstract URL getResource(String name, ClassLoader cl);
   /** Find all resource URLs for the given name. Since this typically
    * entails an exhuastive search of the repository it can be a relatively
    * slow operation.
    *
    * @param name the resource name
    * @param cl the requesting class loader
    * @param urls a list into which the located resource URLs will be placed
    */
   public abstract void getResources(String name, ClassLoader cl, List urls);

   /** Not used.
    * @param loader
    * @param className
    * @return
    * @throws ClassNotFoundException
    */
   public abstract Class loadClassWithout(ClassLoader loader, String className)
      throws ClassNotFoundException;
   /** Add a class loader to the repository
    */
   public abstract void addClassLoader(ClassLoader cl);
   /** Update the set of URLs known to be associated with a previously added
    * class loader.
    *
    * @param cl
    * @param url
    */
   public abstract boolean addClassLoaderURL(ClassLoader cl, URL url);
   /** Remove a cladd loader from the repository.
    * @param cl
    */
   public abstract void removeClassLoader(ClassLoader cl);

   /**
    * Return the class of a java native type
    * @return the class, or null if className is not a native class name
    */
   public static final Class getNativeClassForName(String className)
   {
      // Check for native classes
      return (Class)nativeClassBySignature.get(className);
   }

   /**
    * Add a class to the the cache
    */
   void cacheLoadedClass(String name, Class cls, ClassLoader cl)
   {
       synchronized( classes )
       {
          // Update the global cache
          Object prevClass = classes.put(name, cls);
          if( log.isTraceEnabled() )
          {
             log.trace("cacheLoadedClass, classname: "+name+", class: "+cls
                +", ucl: "+cl);
          }

           /**
            * TODO: Adding this implementation is a hack for jmx 1.2 checkin.
            * Had to add this because need getCachedClass() to work.
            * However, this method does not add loaded classes to collection
            * to be unloaded when called to remove classloader.
            * Hopefully this will be a short term workaround.
            * Contact telrod@e2technologies.net if you have questions. -TME
            */
       }
   }

   void clear()
   {
      classes.clear();
   }
}
