/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.mq;

import java.io.PrintWriter;
import java.io.PrintStream;

import javax.jms.JMSException;

import org.jboss.util.NestedThrowable;
import org.jboss.util.NestedException;

/**
 * A common superclass for <tt>JMSException</tt> classes that can contain a
 * nested <tt>Throwable</tt> detail object.
 * 
 * @author <a href="mailto:jason@planet57.com">Jason Dillon</a>
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @version <tt>$Revision: 1.3.2.2 $</tt>
 */
public class SpyJMSException extends JMSException implements NestedThrowable
{
   // Constants -----------------------------------------------------

   /** The serialVersionUID */
   static final long serialVersionUID = 5216406958161784593L;

   // Attributes ----------------------------------------------------

   /** The nested throwable */
   protected Throwable nested;

   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------

   /**
	 * Construct a <tt>SpyJMSException</tt> with the specified detail message.
	 * 
	 * @param msg Detail message.
	 */
   public SpyJMSException(final String msg)
   {
      super(msg);
      this.nested = null;
   }

   /**
	 * Construct a <tt>SpyJMSException</tt> with the specified detail message
	 * and error code.
	 * 
	 * @param msg Detail message.
	 * @param code Error code.
	 */
   public SpyJMSException(final String msg, final String code)
   {
      super(msg, code);
      this.nested = null;
   }

   /**
	 * Construct a <tt>SpyJMSException</tt> with the specified detail message
	 * and nested <tt>Throwable</tt>.
	 * 
	 * @param msg Detail message.
	 * @param nested Nested <tt>Throwable</tt>.
	 */
   public SpyJMSException(final String msg, final Throwable nested)
   {
      super(msg);
      this.nested = nested;
      NestedThrowable.Util.checkNested(this, nested);
   }

   // Public --------------------------------------------------------

   /**
	 * Construct a <tt>SpyJMSException</tt> with the specified nested <tt>Throwable</tt>.
	 * 
	 * @param nested Nested <tt>Throwable</tt>.
	 */
   public SpyJMSException(final Throwable nested)
   {
      this(nested.getMessage(), nested);
   }

   // NestedException implementation --------------------------------

   public void setLinkedException(final Exception e)
   {
      this.nested = e;
   }

   public Exception getLinkedException()
   {
      // jason: this is bad, but whatever... the jms folks should have had more
      // insight
      if (nested == null)
         return this;
      if (nested instanceof Exception)
         return (Exception) nested;
      return new NestedException(nested);
   }

   // Throwable overrides -------------------------------------------

   public Throwable getNested()
   {
      return nested;
   }

   public Throwable getCause()
   {
      return nested;
   }

   public String getMessage()
   {
      return NestedThrowable.Util.getMessage(super.getMessage(), nested);
   }

   public void printStackTrace(final PrintStream stream)
   {
      if (nested == null || NestedThrowable.PARENT_TRACE_ENABLED)
         super.printStackTrace(stream);
      NestedThrowable.Util.print(nested, stream);
   }

   public void printStackTrace(final PrintWriter writer)
   {
      if (nested == null || NestedThrowable.PARENT_TRACE_ENABLED)
         super.printStackTrace(writer);
      NestedThrowable.Util.print(nested, writer);
   }

   public void printStackTrace()
   {
      printStackTrace(System.err);
   }

   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------
}