/*
 * JBossMQ, the OpenSource JMS implementation
 * 
 * Distributable under LGPL license. See terms of license at gnu.org.
 */

package org.jboss.mq;

import javax.jms.JMSException;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import org.jboss.logging.Logger;

/**
 * This class implements the XAResource interface for used with an XASession.
 * 
 * @author Hiram Chirino (Cojonudo14@hotmail.com)
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @version $Revision: 1.5.2.4 $
 */
public class SpyXAResource implements XAResource
{
   // Constants -----------------------------------------------------

   /** The log */
   private static final Logger log = Logger.getLogger(SpyXAResource.class);
   
   /** Whether trace is enabled */
   private static boolean trace = log.isTraceEnabled();
   
   // Attributes ----------------------------------------------------

   /** The session */
   SpySession session;
   
   // Static --------------------------------------------------------
   
   // Constructors --------------------------------------------------

   /**
    * Create a new SpyXAResource
    *
    * @param session the session
    */
   SpyXAResource(SpySession session)
   {
      trace = log.isTraceEnabled();
      
      this.session = session;
      
      if (trace)
         log.trace("Created " + this);
   }
   
   // Public --------------------------------------------------------
   
   // XAResource implementation -------------------------------------

   public boolean setTransactionTimeout(int arg1) throws XAException
   {
      return false;
   }

   public int getTransactionTimeout() throws XAException
   {
      return 0;
   }

   public boolean isSameRM(XAResource arg1) throws XAException
   {
      if (!(arg1 instanceof SpyXAResource))
         return false;
      return ((SpyXAResource) arg1).session.connection.spyXAResourceManager == session.connection.spyXAResourceManager;
   }
   public void commit(Xid xid, boolean onePhase) throws XAException
   {
      if (trace)
         log.trace("Commit xid=" + xid + ", onePhase=" + onePhase + " " + this);

      try
      {
         session.connection.spyXAResourceManager.commit(xid, onePhase);
      }
      catch (JMSException e)
      {
         throw new SpyXAException(XAException.XAER_RMERR, e);
      }
   }

   public void end(Xid xid, int flags) throws XAException
   {
      if (trace)
         log.trace("End xid=" + xid + ", flags=" + flags + " " +this);

      synchronized (session.runLock)
      {

         switch (flags)
         {
            case TMSUSPEND :
               session.unsetCurrentTransactionId(xid);
               session.connection.spyXAResourceManager.suspendTx(xid);
               break;
            case TMFAIL :
               session.unsetCurrentTransactionId(xid);
               session.connection.spyXAResourceManager.endTx(xid, false);
               break;
            case TMSUCCESS :
               session.unsetCurrentTransactionId(xid);
               session.connection.spyXAResourceManager.endTx(xid, true);
               break;
         }
      }
   }

   public void forget(Xid xid) throws XAException
   {
      if (trace)
         log.trace("Forget xid=" + xid + " " + this);
   }

   public int prepare(Xid xid) throws XAException
   {
      if (log.isTraceEnabled())
         log.trace("Prepare xid=" + xid + " " + this);

      try
      {
         return session.connection.spyXAResourceManager.prepare(xid);
      }
      catch (JMSException e)
      {
         throw new SpyXAException(XAException.XAER_RMERR, e);
      }
   }

   public Xid[] recover(int arg1) throws XAException
   {
      if (log.isTraceEnabled())
         log.trace("Recover arg1=" + arg1 + " " + this);

      return new Xid[0];
   }

   public void rollback(Xid xid) throws XAException
   {
      if (log.isTraceEnabled())
         log.trace("Rollback xid=" + xid + " " + this);

      try
      {
         session.connection.spyXAResourceManager.rollback(xid);
      }
      catch (JMSException e)
      {
         throw new SpyXAException(XAException.XAER_RMERR, e);
      }
   }

   public void start(Xid xid, int flags) throws XAException
   {
      if (log.isTraceEnabled())
         log.trace("Start xid=" + xid + ", flags=" + flags + " " + this);

      boolean convertTx = false;
      if (session.getCurrentTransactionId() != null)
      {
         if (flags == TMNOFLAGS && session.getCurrentTransactionId() instanceof Long)
         {
            convertTx = true;
         }
         else
         {
            throw new XAException(XAException.XAER_OUTSIDE);
         }
      }

      synchronized (session.runLock)
      {

         switch (flags)
         {
            case TMNOFLAGS :
               if (convertTx)
               {
                  // it was an anonymous TX, TM is now taking control over it.
                  // convert it over to a normal XID tansaction.
                  session.setCurrentTransactionId(session.connection.spyXAResourceManager
                        .convertTx((Long) session.getCurrentTransactionId(), xid));
               }
               else
               {
                  session.setCurrentTransactionId(session.connection.spyXAResourceManager.startTx(xid));
               }
               break;
            case TMJOIN :
               session.setCurrentTransactionId(session.connection.spyXAResourceManager.joinTx(xid));
               break;
            case TMRESUME :
               session.setCurrentTransactionId(session.connection.spyXAResourceManager.resumeTx(xid));
               break;
         }
         session.runLock.notify();
      }
   }
   
   // Object overrides ----------------------------------------------
   
   public String toString()
   {
      return "SpyXAResource[session=" + session + ']';
   }
   
   // Package protected ---------------------------------------------
   
   // Protected -----------------------------------------------------
   
   // Private -------------------------------------------------------
   
   // Inner classes -------------------------------------------------
}