/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
package org.jboss.remoting;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.StreamCorruptedException;

import org.jboss.logging.Logger;
import org.jboss.remoting.loading.ClassByteClassLoader;
import org.jboss.remoting.loading.ClassBytes;
import org.jboss.remoting.loading.ClassUtil;

/**
 * AbstractInvoker is an abstract handler part that contains common methods between both
 * client and server.
 *
 * @author <a href="mailto:jhaynie@vocalocity.net">Jeff Haynie</a>
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 * @version $Revision: 1.8.2.1 $
 */
public abstract class AbstractInvoker implements Invoker
{
    protected static boolean logExtra = Boolean.getBoolean("jboss.remoting.invoker.debug");

    protected final Logger log = Logger.getLogger(getClass());
    private boolean destroyed = false;
    protected ClassByteClassLoader classbyteloader;
    protected InvokerLocator locator;
    protected InvokerLocator localServerLocator;

    public AbstractInvoker (InvokerLocator locator)
    {
        this.classbyteloader = new ClassByteClassLoader(this.getClass().getClassLoader());
        this.locator = locator;

        localServerLocator = InvokerRegistry.getSuitableServerLocatorForRemote(locator);
    }

    /**
     * called when the invoker is no longer to be used and should release all internal resources
     */
    public void destroy()
    {
        this.destroyed = true;
    }

    /**
     * returns true if destroyed, false is not  (regardless of the connection state)
     *
     * @return
     */
    public boolean isDestroyed()
    {
        return destroyed;
    }

    /**
     * return the locator this Invoker represents
     *
     * @return
     */
    public InvokerLocator getLocator ()
    {
        return locator;
    }

    /**
     * This should be set when want to override the default behavior of automatically
     * getting s suitable locator.  This should be used want want to control what type
     * of callbacks to receive (pull or push).  Set to null to poll for callback messages.
     * This can also be used to receive callbacks using another transport and subsystem,
     * if desired.
     * @param locator
     */
    public void setClientLocator(InvokerLocator locator)
    {
        localServerLocator = locator;
    }

    /**
     * Gets the client locator.  This locator will be used by the server side
     * to make callbacks to the handler for this locator.
     * @return
     */
    public InvokerLocator getClientLocator()
    {
        return localServerLocator;
    }

    /**
     * set the classloader to use internally
     *
     * @param classloader
     */
    public synchronized void setClassLoader (ClassLoader classloader)
    {
        this.classbyteloader = new org.jboss.remoting.loading.ClassByteClassLoader(classloader);
    }

    /**
     * simple utility method for reading bytes from an input stream
     *
     * @param in
     * @return
     * @throws IOException
     */
    protected byte[] read (InputStream in)
        throws IOException
    {
        ByteArrayOutputStream out=new ByteArrayOutputStream();
        byte buf[]=new byte[4096];
        while(true)
        {
            int c = in.read(buf);
            if (c<0)
            {
                break;
            }
            out.write(buf,0,c);
        }
        return out.toByteArray();
    }

    /**
     * method is called to retrieve a byte array of a Class for a given class name
     *
     * @param className
     * @return
     */
    public byte[] getClassBytes (String className)
    {
        String cn = null;
        if (ClassUtil.isArrayClass(className))
        {
            // if requesting an array, of course, that would be found in our class path, so we
            // need to strip the class data and just return the class part, the other side
            // will properly load the class as an array
            cn = ClassUtil.getArrayClassPart(className).replace('.','/') + ".class";
        }
        else
        {
            cn = className.replace('.','/') + ".class";
        }
        if (logExtra && log.isDebugEnabled())
        {
            log.debug("trying to load class: "+className+" from path: "+cn);
        }
        InputStream in = null;
        ClassLoader cl = classbyteloader;

        if (cl==null)
        {
            cl = ClassLoader.getSystemClassLoader();
        }
        if (cl!=null)
        {
            in = cl.getResourceAsStream(cn);
            if (in!=null)
            {
                if (logExtra && log.isDebugEnabled())
                {
                    log.debug("looking for classes at: "+cl);
                }
                try
                {
                    byte data[] = read(in);
                    if (logExtra && log.isDebugEnabled())
                    {
                        log.debug("found class at classloader: "+cl);
                    }
                    return data;
                }
                catch (IOException io)
                {
                }
                finally
                {
                    if (in!=null)
                    {
                        try { in.close(); } catch (Exception ig) {}
                        in = null;
                    }
                }
            }
        }
        return null;
    }

    /**
     * add a remote class to the local loader to make the class visible to the
     * handler
     *
     * @param neededClass
     * @throws IOException
     */
    public void addClass (ClassBytes neededClass)
        throws IOException
    {
        if (classbyteloader!=null)
        {
            classbyteloader.addClass(neededClass);
        }
    }

    /**
     * load a class, taking into account the remote classes downloaded, if any. this class
     * will use the following order for finding the class:
     *
     * <OL>
     *      <LI>look at the ClassByteClassLoader associated with this handler </LI>
     *      <LI>look at the Thread ContextClassLoader (if any)</LI>
     *      <LI>look at the System ClassLoader</LI>
     * </OL>
     *
     * @param className
     * @return
     * @throws ClassNotFoundException
     */
    public Class loadClass (String className)
        throws ClassNotFoundException
    {
        if (logExtra && log.isDebugEnabled())
        {
            log.debug("... load class: "+className);
        }
        String cn = className;
        boolean array = ClassUtil.isArrayClass(cn);
        Class cl = null;
        ClassLoader thecl = null;

        if (classbyteloader!=null)
        {
            if (logExtra && log.isDebugEnabled())
            {
                log.debug("... trying classbyteclassloader - load class: "+className);
            }
            try { cl = classbyteloader.loadClass(cn); thecl = classbyteloader; } catch (Exception ex) {}
        }
        if (cl==null)
        {
            if (logExtra && log.isDebugEnabled())
            {
                log.debug("... trying context classloader - load class: "+className);
            }
            ClassLoader loader = Thread.currentThread().getContextClassLoader();
            if (loader!=null)
            {
                try { cl = loader.loadClass(className); thecl = loader; } catch (Exception ex) {}
            }
        }
        if (cl==null)
        {
            if (array)
            {
                cn = ClassUtil.getArrayClassPart(className);
            }
            cl = ClassLoader.getSystemClassLoader().loadClass(cn);
            thecl = ClassLoader.getSystemClassLoader();
            if (logExtra && log.isDebugEnabled())
            {
                log.debug("... getting the system classloader: "+thecl+", class of class part: "+cn);
            }
        }
        if (array)
        {
            if (logExtra && log.isDebugEnabled())
            {
                log.debug("... trying thecl - load class: "+className);
            }
            // arrays are special, we need to request in the case with the non-stripped classname
            // and let the local JVM create the array class from the original class
            return thecl.loadClass(className);
        }
        return cl;
    }

    protected Object unmarshal (ClassBytes resultBytes, ClassBytes neededClass)
        throws ClassNotFoundException, StreamCorruptedException, IOException
    {
        try
        {
            if (logExtra && log.isDebugEnabled())
            {
                log.debug("unmarshal ("+resultBytes+","+neededClass+")");
            }
            if (neededClass!=null)
            {
                addClass(neededClass);
            }
            String cn = resultBytes.getClassName();
            // attempt to load the class
            loadClass(cn);
            // deserialize now that we should have the class available
            ByteArrayInputStream bis=new ByteArrayInputStream(resultBytes.getClassBytes());
            ObjectInputStream ois=new org.jboss.remoting.loading.ObjectInputStreamWithClassLoader(bis,classbyteloader);
            return ois.readObject();
        }
        catch (ClassNotFoundException cnf)
        {
            //NOTE: make sure the class you return is from the exception message instead of
            //the result passed in the method - since classloading is recursive and the class not found
            //may not be the one passed in
            if (logExtra && log.isDebugEnabled())
            {
                log.debug("during unmarshalling, class: "+cnf.getMessage()+" not found, telling the other side as such ...");
            }
            throw new ClassNotFoundException(cnf.getMessage());
        }
    }


}
