/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
package org.jboss.remoting.transport.async;

import java.util.Map;
import java.util.WeakHashMap;

import EDU.oswego.cs.dl.util.concurrent.Slot;
import org.jboss.logging.Logger;

/**
 * Allows you to create a request to which
 * you can at a later time wait for the response to 
 * arrive asynchrously.
 * 
 * @author <a href="mailto:hiram@coredevelopers.net">Hiram Chirino</a>
 */
public class Correlator {
	
	static public class FutureResult extends Slot {
		private final HashKey key;	
		
		FutureResult(HashKey key){
			this.key=key;	
		}
		
		public int getID() {
			return key.value;
		}		
	}
	
	Logger log = Logger.getLogger(Correlator.class);

	/**
	 * Since a incrementing int is used as a key to a HashMap,
	 * this class is used to calculate a hashCode that is more 
	 * spred to get better hashing. 
	 */
	private static class HashKey {
		final int value;
		final int hashCode;
		
		public HashKey(int value) {
			this.value=value;			
			long rc=value;
			if( (value%2)==1 ) 
				rc *= -1;
			rc *= Integer.MAX_VALUE/7;			
			hashCode = (int) (rc%Integer.MAX_VALUE);			
		}
		/**
		 * Not a very proper equals since it takes
		 * shortcuts, but since this class is not used
		 * in a general case, it's ok. 
		 */
		public boolean equals(Object obj) {
			return ((HashKey)obj).value == value;
		}
		public int hashCode() {
			return hashCode;
		}
	}

	private Map slots = new WeakHashMap(100);
	private int lastFutureResultID=0;
	
	synchronized private int getNextFutureResultID() {
		return ++lastFutureResultID;
	}
	
	public FutureResult getNextFutureResult() {
		HashKey key = new HashKey(getNextFutureResultID());
		FutureResult s = new FutureResult(key);				
		synchronized(slots) {
			slots.put(key,s);
		}		
		if( log.isTraceEnabled() )
			log.trace("Created Request: "+key.value);
		return s;	
	}

	public void dispatchResponse(int id, Object data ) throws InterruptedException {
		if( log.isTraceEnabled() )
			log.trace("Received resposne for request: "+id);
			
		FutureResult s;
		synchronized(slots) {
			s = (FutureResult) slots.get(new HashKey(id));
		}
		if( s != null ) {
			s.put(data);
		} else {
			log.trace("The request may have timed out.  Request slot was not found");			
		}
		
	}
}
