/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting;


import java.io.ByteArrayInputStream;
import java.rmi.server.UID;
import java.util.List;
import java.util.Random;
import javax.xml.parsers.DocumentBuilderFactory;
import junit.framework.*;
import org.jboss.logging.Logger;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.mock.MockClientInvoker;
import org.jboss.remoting.transport.mock.MockServerInvoker;
import org.jboss.remoting.invocation.NameBasedInvocation;
import org.w3c.dom.Document;
import java.util.ArrayList;

/**
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class InvokerClientUnitTestCase extends AbstractInvokerTest
{
    private static final Logger log = Logger.getLogger(InvokerClientUnitTestCase.class);

    private String sessionId = new UID().toString();

    private Client client;

   private Connector connector;

    public InvokerClientUnitTestCase(String name)
    {
       super(name);
    }

   public InvokerClientUnitTestCase(String name, String transport, int port)
    {
       super(name, transport, port);
    }


    protected void setUp() throws Exception
    {
       super.setUp();
       initServer(8081);
       //initServer(port);
       InvokerLocator locator = new InvokerLocator(transport + "://localhost:" + port);
       client = new Client(locator, "mock");
       client.connect();
    }

    //TODO: Same method in InvokerServerTest so could move up into AbstractInvokerTest
    private InvokerLocator initServer(int port) throws Exception
    {
        if(port < 0)
        {
            port = 1024 + Math.abs(new Random().nextInt(2000));
        }
        log.debug("port = " + port);

        InvokerRegistry.registerInvoker("mock", MockClientInvoker.class, MockServerInvoker.class);
        connector = new Connector();
        InvokerLocator locator = new InvokerLocator(transport + "://localhost:" + port);
        StringBuffer buf = new StringBuffer();
        buf.append("<?xml version=\"1.0\"?>\n");
        buf.append("<handlers>\n");
        buf.append("  <handler subsystem=\"mock\">org.jboss.remoting.transport.mock.MockServerInvocationHandler</handler>\n");
        buf.append("</handlers>\n");
        Document xml = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new ByteArrayInputStream(buf.toString().getBytes()));
        connector.setInvokerLocator(locator.getLocatorURI());
        connector.setConfiguration(xml.getDocumentElement());
        connector.start();
        return locator;
    }

   protected void tearDown() throws Exception
   {
      super.tearDown();
      connector.stop();
   }

    public void testPullCallback() throws Exception
    {
        try
        {
            // should be null by default, since don't have connector started, but setting anyway
            //client.setClientLocator(null);

            MockInvokerCallbackHandler handler = new MockInvokerCallbackHandler(sessionId);

            log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

            // simple invoke, should return foo
            Object ret = makeInvocation("bar", "foo");
            assertTrue("invocation should have returned foo rather than: " + ret, "foo".equals(ret));
            // add notification listener
            // Can now call directly on client
            //ret = makeInvocation("addListener", null);
            client.addListener(handler);
            // invoke which should cause callback on server side
            ret = makeInvocation("test", "test");
            // allow time for callback
            Thread.sleep(2000);
            List handlers = client.getCallbacks();
            assertTrue("There should be a registered callback", handlers.size() == 1);
            // can now call directly on client
            //ret = makeInvocation("removeListener", null);
            client.removeListener(handler);
            handlers = client.getCallbacks();
            assertTrue("There should be no registered callbacks instead of: " + handlers.size(), handlers.size() == 0);
        }
        catch (Exception e)
        {
           throw e;
        } // end of catch
        catch (AssertionFailedError  e)
        {
           throw (AssertionFailedError)e;
        } // end of catch
        catch (Throwable e)
        {
           e.printStackTrace();
           throw (RuntimeException)e;
        } // end of catch
        finally
        {
            if(client != null)
            {
                client.disconnect();
                client = null;
            }
        }
    }

    public void testPushCallback() throws Exception
    {
        try
        {
            sessionId = new UID().toString();
            InvokerLocator locator = initServer(-1);
            //            init();
            //client.setClientLocator(locator);

            MockInvokerCallbackHandler handler = new MockInvokerCallbackHandler(sessionId);

            log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

            // simple invoke, should return bar
            Object ret = makeInvocation("foo", "bar");
            assertTrue("invocation should have returned bar rather than: " + ret, "bar".equals(ret));
            // add notification listener
//            ret = makeInvocation("addNotificationListener", null);
            // Can now call direct on client
            //ret = makeInvocation("addListener", null);
            client.addListener(handler, locator);
            // invoke which should cause callback
            ret = makeInvocation("test", "test");
            // allow time for callback
            Thread.sleep(2000);
            log.debug("done sleeping.");
            // Can now call direct on client
            //ret = makeInvocation("removeListener", null);
            client.removeListener(handler);
            // shouldn't get callback now since removed listener
            ret = makeInvocation("test", "test");
        }
        catch (Exception e)
        {
           throw e;
        } // end of catch
        catch (AssertionFailedError  e)
        {
           throw (AssertionFailedError)e;
        } // end of catch
        catch (Throwable e)
        {
           e.printStackTrace();
           throw (RuntimeException)e;
        } // end of catch
        finally
        {
            if(client != null)
            {
                client.disconnect();
            }
        }
    }

    private Object makeInvocation(String method, String param) throws Throwable
    {
        Object ret = client.invoke(new NameBasedInvocation(method,
                                                           new Object[]{param},
                                                           new String[]{String.class.getName()}),
                                   null);

        return ret;
    }
   /*
    public static void main(String[] args)
    {
        InvokerClientTest client = null;
        if(args.length == 1)
        {
            int instances = Integer.parseInt(args[0]);
            client = new InvokerClientTest(instances);
        }
        else if(args.length == 2)
        {
            String transport = args[0];
            int port = Integer.parseInt(args[1]);
            client = new InvokerClientTest(transport, port);
        }
        else if(args.length == 3)
        {
            String transport = args[0];
            int port = Integer.parseInt(args[1]);
            int instances = Integer.parseInt(args[2]);
            client = new InvokerClientTest(transport, port, instances);
        }
        else
        {
            client = new InvokerClientTest();
            log.debug("Using default transport (" + client.getTransport() +
                               ") and default port (" + client.getPort() + ") and " +
                               "default number of instances (" + client.getNumberOfInstances() + ")" +
                               "\nCan enter transport, port, and instances via command line.");
        }

        try
        {
            client.runTest();
        }
        catch(Throwable e)
        {
            e.printStackTrace();
            System.exit(1);
        }
        System.exit(0);
    }
   */
}
