package org.jboss.security.ssl;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import javax.naming.InitialContext;
import javax.net.SocketFactory;
import javax.net.ssl.HandshakeCompletedEvent;
import javax.net.ssl.HandshakeCompletedListener;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

import com.sun.net.ssl.SSLContext;

import org.jboss.logging.Logger;
import org.jboss.security.SecurityDomain;

/**
 * An implementation of SocketFactory that creates SSL sockets using the 
 * JSSE SSLContext and a JBossSX SecurityDomain for the KeyManagerFactory 
 * and TrustManagerFactory objects.
 *
 * @see com.sun.net.ssl.SSLContext
 * @see com.sun.net.ssl.KeyManagerFactory
 * @see com.sun.net.ssl.TrustManagerFactory
 * @see org.jboss.security.SecurityDomain
 *
 * @author  Scott.Stark@jboss.org
 * @author <a href="mailto:reverbel@ime.usp.br">Francisco Reverbel</a>
 *
 * @version $Revision: 1.2.2.1 $
 */
public class DomainSocketFactory 
   extends SSLSocketFactory
   implements HandshakeCompletedListener
{
   private static Logger log = Logger.getLogger(DomainSocketFactory.class);
   private transient SecurityDomain securityDomain;
   private transient SSLContext sslCtx = null;

   /** 
    * A default constructor for use when created by Class.newInstance. The
    * factory is not usable until its SecurityDomain has been established.
    */
   public DomainSocketFactory()
   {
   }

   /** 
    * Create a socket factory instance that uses the given SecurityDomain
    * as the source for the SSL KeyManagerFactory and TrustManagerFactory.
    */
   public DomainSocketFactory(SecurityDomain securityDomain) 
      throws IOException
   {
      if( securityDomain == null )
         throw new IOException("The securityDomain may not be null");
      this.securityDomain = securityDomain;
   }

   public SecurityDomain getSecurityDomain()
   {
      return securityDomain;
   }

   public void setSecurityDomain(SecurityDomain securityDomain)
   {
      this.securityDomain = securityDomain;
   }

   // SSLSocketFactory methods --------------------------------------

   /** 
    * Create a client socket connected to the specified host and port.
    * 
    * @param serverHost - the host name
    * @param serverPort - the port number
    * @return a socket connected to the specified host and port.
    * @exception IOException if an I/O error occurs during socket creation.
    */
   public Socket createSocket(String serverHost, int serverPort)
      throws IOException, UnknownHostException
   {
      InetAddress serverAddr = InetAddress.getByName(serverHost);
      return this.createSocket(serverAddr, serverPort);
   }

   public Socket createSocket(String serverHost, int serverPort,
                              InetAddress clientAddr, int clientPort)
      throws IOException, UnknownHostException
   {
      InetAddress serverAddr = InetAddress.getByName(serverHost);
      return this.createSocket(serverAddr, serverPort, clientAddr, clientPort);
   }

   public Socket createSocket(InetAddress serverAddr, int serverPort)
      throws IOException
   {
      return this.createSocket(serverAddr, serverPort, null, 0);
   }

   public Socket createSocket(InetAddress serverAddr, int serverPort,
                              InetAddress clientAddr, int clientPort)
      throws IOException
   {
      initSSLContext();
      SSLSocketFactory factory = (SSLSocketFactory)sslCtx.getSocketFactory();
      SSLSocket socket = 
         (SSLSocket)factory.createSocket(serverAddr, serverPort, 
                                         clientAddr, clientPort);
      socket.addHandshakeCompletedListener(this);
      return socket;
   }

   public Socket createSocket(Socket s, String host, 
                              int port, boolean autoClose) 
      throws IOException
   {
      initSSLContext();
      SSLSocketFactory factory = (SSLSocketFactory)sslCtx.getSocketFactory();
      SSLSocket socket = 
         (SSLSocket)factory.createSocket(s, host, port, autoClose);
      socket.addHandshakeCompletedListener(this);
      return socket;
   }

   public String[] getDefaultCipherSuites()
   {
      String[] cipherSuites = {};
      try
      {
         initSSLContext();
         SSLSocketFactory factory = (SSLSocketFactory)sslCtx.getSocketFactory();
         cipherSuites = factory.getDefaultCipherSuites();
      }
      catch(IOException e)
      {
         log.error("Failed to get default SSLSocketFactory", e);
      }      
      return cipherSuites;
   }
   
   public String[] getSupportedCipherSuites()
   {
      String[] cipherSuites = {};
      try
      {
         initSSLContext();
         SSLSocketFactory factory = (SSLSocketFactory)sslCtx.getSocketFactory();
         cipherSuites = factory.getSupportedCipherSuites();
      }
      catch(IOException e)
      {
         log.error("Failed to get default SSLSocketFactory", e);
      }      
      return cipherSuites;
   }
   
   /** 
    * The default SocketFactory which looks to the java:/jaas/other
    * security domain configuration.
    */
   public static SocketFactory getDefault()
   {
      DomainSocketFactory ssf = null;
      try
      {
         InitialContext iniCtx = new InitialContext();
         SecurityDomain sd = (SecurityDomain)iniCtx.lookup("java:/jaas/other");
         ssf = new DomainSocketFactory(sd);
      }
      catch(Exception e)
      {
         log.error("Failed to create default SocketFactory", e);
      }
      return ssf;
   }
   
   // HandshakeCompletedListener method -----------------------------

   public void handshakeCompleted(HandshakeCompletedEvent handshakeCompletedEvent)
   {
      Logger log = Logger.getLogger(ClientSocketFactory.class);
      if( log.isTraceEnabled() )
      {
         String cipher = handshakeCompletedEvent.getCipherSuite();
         SSLSession session = handshakeCompletedEvent.getSession();
         String peerHost = session.getPeerHost();
         log.debug("SSL handshakeCompleted, cipher="+cipher
            +", peerHost="+peerHost);
      }
   }

   // Private method ------------------------------------------------

   private void initSSLContext()
      throws IOException
   {
      if( sslCtx != null )
         return;
      sslCtx = Context.forDomain(securityDomain);
   }

}
