/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.deployment;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.ObjectName;

import org.jboss.mx.util.JMXExceptionDecoder;
import org.jboss.system.ServiceContext;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;
import org.jboss.util.file.Files;
import org.jboss.util.file.JarUtils;
import org.jboss.util.stream.Streams;

/**
 * The main/primary component for deployment.
 *
 * <p>MainDeployerMBean is generated by xdoclet.
 *
 * @author <a href="mailto:marc.fleury@jboss.org">Marc Fleury</a>
 * @author <a href="mailto:scott.stark@jboss.org">Scott Stark</a>
 * @author <a href="mailto:d_jencks@users.sourceforge.net">David Jencks</a>
 * @version $Revision: 1.43.2.23 $
 *
 * @jmx:mbean name="jboss.system:service=MainDeployer"
 *  extends="org.jboss.system.ServiceMBean, org.jboss.deployment.DeployerMBean, org.jboss.deployment.MainDeployerConstants"
 */
public class MainDeployer
   extends ServiceMBeanSupport
   implements Deployer, MainDeployerMBean
{

   /**
    * The variable <code>serviceController</code> is used by the
    * checkIncompleteDeployments method to ask for lists of mbeans
    * with deployment problems.
    */
   private ObjectName serviceController;

   /** Deployers **/
   private final LinkedList deployers = new LinkedList();

   /** A Map of URL -> DeploymentInfo */
   private final Map deploymentMap = Collections.synchronizedMap(new HashMap());

   /** A list of all deployments that have deployers. */
   private final List deploymentList = new ArrayList();

   /** A list of all deployments that do not have deployers. */
   private final List waitingDeployments = new ArrayList();

   /** A helper for sorting deployment URLs. */
   private final Comparator sorter = new DeploymentSorter();

   /** A helper for sorting deploymentInfos */
   private final Comparator infoSorter = new DeploymentInfoComparator(sorter);

   /** Should local copies be made of resources on the local file system */
   private boolean copyFiles = true;

   /** The temporary directory for deployments. */
   private File tempDir;

   /** The string naming the tempDir **/
   private String tempDirString;

   /**
    * Explict no-args contsructor for JMX.
    */
   public MainDeployer()
   {
      // Is there a better place to obtain startup information?
      String localCopy = System.getProperty("jboss.deploy.localcopy");
      if (localCopy != null && (
          localCopy.equalsIgnoreCase("false") ||
          localCopy.equalsIgnoreCase("no") ||
          localCopy.equalsIgnoreCase("off")))
      {
         log.debug("Disabling local copies of file: urls");
         copyFiles = false;
      }
   }

   /**
    * Describe <code>setServiceController</code> method here.
    *
    * @param serviceController an <code>ObjectName</code> value
    * @jmx.managed-attribute
    */
   public void setServiceController(final ObjectName serviceController)
   {
      this.serviceController = serviceController;
   }

   /**
    * The <code>listDeployed</code> method returns a collection of DeploymemtInfo
    * objects for the currently deployed packages.
    *
    * @return a <code>Collection</code> value
    * @jmx:managed-operation
    */
   public Collection listDeployed()
   {
      synchronized (deploymentList)
      {
         log.info("deployment list string: " + deploymentList);
         return new ArrayList(deploymentList);
      }
   }

   /**
    * Describe <code>listDeployedAsString</code> method here.
    *
    * @return a <code>String</code> value
    * @jmx:managed-operation
    */
   public String listDeployedAsString()
   {
      return "<pre>" + listDeployed() + "</pre>";
   }

   /**
    * The <code>listIncompletelyDeployed</code> method returns a list of packages that have
    * not deployed completely. The toString method will include any exception in the status
    * field.
    *
    * @return a <code>Collection</code> value
    * @jmx:managed-operation
    */
   public Collection listIncompletelyDeployed()
   {
      List id = new ArrayList();
      List copy;
      synchronized (deploymentList)
      {
         copy = new ArrayList(deploymentList);
      }
      for (Iterator i = copy.iterator(); i.hasNext();)
      {
         DeploymentInfo di = (DeploymentInfo)i.next();
         if (!"Deployed".equals(di.status) && !"Starting".equals(di.status))
         {
            id.add(di);
         } // end of if ()

      } // end of for ()
      return id;
   }

   /**
    * The <code>listWaitingForDeployer</code> method returns a collection
    * of the packages that currently have no identified deployer.
    *
    * @return a <code>Collection</code> value
    * @jmx:managed-operation
    */
   public Collection listWaitingForDeployer()
   {
      synchronized (waitingDeployments)
      {
         return new ArrayList(waitingDeployments);
      }
   }

   /**
    * The <code>addDeployer</code> method registers a deployer with the main deployer.
    * Any waiting packages are tested to see if the new deployer will deploy them.
    *
    * @param deployer a <code>SubDeployer</code> value
    * @jmx:managed-operation
    */
   public void addDeployer(final SubDeployer deployer)
   {
      log.debug("Adding deployer: " + deployer);
      synchronized(deployers)
      {
         deployers.addFirst(deployer);
      }
      // Send a notification about the deployer addition
      Notification msg = new Notification(ADD_DEPLOYER, this, getNextNotificationSequenceNumber());
      msg.setUserData(deployer.getServiceName());
      sendNotification(msg);

      synchronized (waitingDeployments)
      {
         List copy = new ArrayList(waitingDeployments);
         waitingDeployments.clear();
         for (Iterator i = copy.iterator(); i.hasNext();)
         {
            DeploymentInfo di = (DeploymentInfo)i.next();
            log.debug("trying to deploy with new deployer: " + di.shortName);
            try
            {
               di.setServer(server);
               deploy(di);
            }
            catch (DeploymentException e)
            {
               log.error("DeploymentException while trying to deploy a package with a new deployer", e);
            } // end of try-catch

         } // end of for ()
      }
   }

   /**
    * The <code>removeDeployer</code> method unregisters a deployer with the MainDeployer.
    * Deployed packages deployed with this deployer are undeployed.
    *
    * @param deployer a <code>SubDeployer</code> value
    * @jmx:managed-operation
    */
   public void removeDeployer(final SubDeployer deployer)
   {
      log.debug("Removing deployer: " + deployer);
      boolean removed = false;
      synchronized(deployers)
      {
         removed = deployers.remove(deployer);
      }
      // Send a notification about the deployer removal
      if( removed )
      {
         Notification msg = new Notification(REMOVE_DEPLOYER, this,
            getNextNotificationSequenceNumber());
         msg.setUserData(deployer.getServiceName());
         sendNotification(msg);
      }

      List copy;
      synchronized (deploymentList) {
         copy = new ArrayList(deploymentList);
      }
      for (Iterator i = copy.iterator(); i.hasNext(); )
      {
         DeploymentInfo di = (DeploymentInfo)i.next();
         if (di.deployer == deployer)
         {
            undeploy(di);
            di.deployer = null;
            synchronized (waitingDeployments)
            {
               waitingDeployments.add(di);
            }
         }
      }
   }

   /**
    * The <code>listDeployers</code> method returns a collection of ObjectNames of
    * deployers registered with the MainDeployer.
    *
    * @return a <code>Collection<ObjectName></code> value
    * @jmx:managed-operation
    */
   public Collection listDeployers()
   {
      ArrayList deployerNames = new ArrayList();
      synchronized (deployers)
      {
         for (Iterator i = deployers.iterator(); i.hasNext(); )
         {
            SubDeployer deployer = (SubDeployer) i.next();
            ObjectName name = deployer.getServiceName();
            deployerNames.add(name);
         }
      }
      return deployerNames;
   }

   // ServiceMBeanSupport overrides ---------------------------------

   protected ObjectName getObjectName(MBeanServer server, ObjectName name)
      throws MalformedObjectNameException
   {
      return name == null ? OBJECT_NAME : name;
   }

   /**
    * The <code>createService</code> method is one of the ServiceMBean lifecyle operations.
    * (no jmx tag needed from superinterface)
    * @exception Exception if an error occurs
    */
   protected void createService() throws Exception
   {
      ServerConfig config = ServerConfigLocator.locate();
      // Get the temp directory location
      File basedir = config.getServerTempDir();
      // Set the local copy temp dir to tmp/deploy
      tempDir = new File(basedir, "deploy");
      // Delete any existing content
      Files.delete(tempDir);
      // Make sure the directory exists
      tempDir.mkdirs();

      // used in inLocalCopyDir
      tempDirString = tempDir.toURL().toString();
   }

   /**
    * The <code>shutdown</code> method undeploys all deployed packages in
    * reverse order of their deployement.
    *
    * @jmx:managed-operation
    */
   public void shutdown()
   {
      // if we shutdown in the middle of a scan, it still might be possible that we try to redeploy
      // things we are busy killing...
      int deployCounter = 0;

      // undeploy everything in sight
      List copy;
      synchronized (deploymentList)
      {
         copy = new ArrayList(deploymentList);
      }
      for (ListIterator i = copy.listIterator(copy.size()); i.hasPrevious(); )
      {
         try
         {
            undeploy((DeploymentInfo)i.previous(), true);
            deployCounter++;
         }
         catch (Exception e)
         {
            log.info("exception trying to undeploy during shutdown", e);
         }

      }

      log.debug("Undeployed " + deployCounter + " deployed packages");
   }


   /**
    * Describe <code>redeploy</code> method here.
    *
    * @param urlSpec a <code>String</code> value
    * @exception DeploymentException if an error occurs
    * @exception MalformedURLException if an error occurs
    * @jmx:managed-operation
    */
   public void redeploy(String urlspec)
      throws DeploymentException, MalformedURLException
   {
      redeploy(new URL(urlspec));
   }

   /**
    * Describe <code>redeploy</code> method here.
    *
    * @param url an <code>URL</code> value
    * @exception DeploymentException if an error occurs
    * @jmx:managed-operation
    */
   public void redeploy(URL url) throws DeploymentException
   {
      DeploymentInfo sdi = getDeployment(url);
      if (sdi!= null)
      {
         redeploy(sdi);
      }
      else
      {
         deploy(url);
      } // end of else
   }

   /**
    * Describe <code>redeploy</code> method here.
    *
    * @param sdi a <code>DeploymentInfo</code> value
    * @exception DeploymentException if an error occurs
    * @jmx:managed-operation
    */
   public void redeploy(DeploymentInfo sdi) throws DeploymentException
   {
      try
      {
         undeploy(sdi);
      }
      catch (Throwable t)
      {
         log.info("Throwable from undeployment attempt: ", t);
      } // end of try-catch
      sdi.setServer(server);
      deploy(sdi);
   }


   /**
    * The <code>undeploy</code> method  undeploys a package identified by a URL
    *
    * @param url an <code>URL</code> value
    * @jmx:managed-operation
    */
   public void undeploy(URL url) throws DeploymentException
   {
      DeploymentInfo sdi = getDeployment(url);
      if (sdi!= null)
      {
         undeploy(sdi);
      }
   }

   /**
    * The <code>undeploy</code> method undeploys a package identified by a string
    * representation of a URL.
    *
    * @param urlspec a <code>String</code> value
    * @exception MalformedURLException if an error occurs
    * @jmx:managed-operation
    */
   public void undeploy(String urlspec)
      throws DeploymentException, MalformedURLException
   {
      undeploy(new URL(urlspec));
   }

   /**
    * The <code>undeploy</code> method undeploys a package represented by a
    * DeploymentInfo object.
    *
    * @param di a <code>DeploymentInfo</code> value
    * @jmx:managed-operation
    */
   public void undeploy(DeploymentInfo di)
   {
      undeploy(di, false);
   }
   protected void undeploy(DeploymentInfo di, boolean isShutdown)
   {
      log.debug("Undeploying "+di.url);
      stop(di);
      destroy(di);
   }

   /**
    * The <code>stop</code> method  is the first internal step of undeployment
    *
    * @param di a <code>DeploymentInfo</code> value
    */
   private void stop(DeploymentInfo di)
   {
      // Stop all sub-deployments
      ArrayList reverseSortedSubs = new ArrayList(di.subDeployments);
      Collections.sort(reverseSortedSubs, infoSorter);
      Collections.reverse(reverseSortedSubs);
      for (Iterator subs = reverseSortedSubs.iterator(); subs.hasNext();)
      {
         DeploymentInfo sub = (DeploymentInfo) subs.next();
         log.debug("Stopping sub deployment: "+sub.url);
         stop(sub);
      }
      // Lastly stop this deployment itself
      try
      {
         // Tell the respective deployer to undeploy this one
         if (di.deployer != null)
         {
            di.deployer.stop(di);
         }
      }
      catch (Throwable t)
      {
         log.error("Undeployment failed: " + di.url, t);
      }

   }

   /**
    * The <code>destroy</code> method is the second and final internal undeployment step.
    *
    * @param di a <code>DeploymentInfo</code> value
    */
   private void destroy(DeploymentInfo di)
   {
      // Destroy all sub-deployments
      ArrayList reverseSortedSubs = new ArrayList(di.subDeployments);
      Collections.sort(reverseSortedSubs, infoSorter);
      Collections.reverse(reverseSortedSubs);
      for (Iterator subs = reverseSortedSubs.iterator(); subs.hasNext();)
      {
         DeploymentInfo sub = (DeploymentInfo) subs.next();
         log.debug("Destroying sub deployment: "+sub.url);
         destroy(sub);
      }
      // Lastly destroy the deployment itself
      try
      {
         // Tell the respective deployer to undeploy this one
         if (di.deployer != null)
         {
            di.deployer.destroy(di);
         }
      }
      catch (Throwable t)
      {
         log.error("Undeployment failed: " + di.url, t);
      }


      try
      {
         // remove from local maps
         deploymentMap.remove(di.url);
         synchronized (deploymentList)
         {
            if (deploymentList.lastIndexOf(di) != -1)
            {
               deploymentList.remove(deploymentList.lastIndexOf(di));
            }
         }
         synchronized (waitingDeployments)
         {
            waitingDeployments.remove(di);
         }
         // Nuke my stuff, this includes the class loader
         di.cleanup();

         log.debug("Undeployed "+di.url);
      }
      catch (Throwable t)
      {
         log.error("Undeployment cleanup failed: " + di.url, t);
      }
   }

   /**
    * The <code>deploy</code> method deploys a package identified by a
    * string representation of a URL.
    *
    * @param urlspec a <code>String</code> value
    * @exception MalformedURLException if an error occurs
    * @jmx:managed-operation
    */
   public void deploy(String urlspec)
      throws DeploymentException, MalformedURLException
   {
      URL url;
      try
      {
         url = new URL(urlspec);
      }
      catch (MalformedURLException e)
      {
         File file = new File(urlspec);
         url = file.toURL();
      }

      deploy(url);
   }

   /**
    * The <code>deploy</code> method deploys a package identified by a URL
    *
    * @param url an <code>URL</code> value
    * @jmx:managed-operation
    */
   public void deploy(URL url) throws DeploymentException
   {
      DeploymentInfo sdi = getDeployment(url);
      // if it does not exist create a new deployment
      if (sdi == null)
      {
         sdi = new DeploymentInfo(url, null, getServer());
         deploy(sdi);
      }
      if( sdi.state != DeploymentState.STARTED )
         checkIncompleteDeployments();
   }

   /**
    * The <code>deploy</code> method deploys a package represented by a DeploymentInfo object.
    *
    * @param deployment a <code>DeploymentInfo</code> value
    * @exception DeploymentException if an error occurs
    * @jmx:managed-operation
    */
   public void deploy(DeploymentInfo deployment)
      throws DeploymentException
   {
      // If we are already deployed return
      if (isDeployed(deployment.url))
      {
         log.info("Package: " + deployment.url + " is already deployed");
         return;
      }
      log.debug("Starting deployment of package: " + deployment.url);

      boolean inited = false;
      try
      {
         inited = init(deployment);
      }
      catch (Throwable t)
      {
         log.error("Could not initialise deployment: " + deployment.url, t);
         DeploymentException.rethrowAsDeploymentException("Could not initialise deployment: " + deployment.url, t);
      }
      if ( inited )
      {
         create(deployment);
         start(deployment);
         log.debug("Deployed package: " + deployment.url);
      } // end of if ()
      else
      {
         log.debug("Deployment of package: " + deployment.url + " is waiting for an appropriate deployer.");
      } // end of else
   }


   /**
    * The <code>init</code> method is the first internal deployment step.
    * The tasks are to copy the code if necessary,
    * set up classloaders, and identify the deployer for the package.
    *
    * @param deployment a <code>DeploymentInfo</code> value
    * @throws DeploymentException if an error occurs
    */
   private boolean init(DeploymentInfo deployment) throws DeploymentException
   {
      // If we are already deployed return
      if (isDeployed(deployment.url))
      {
         log.info("Package: " + deployment.url + " is already deployed");
         return false;
      }
      log.debug("Starting deployment (init step) of package at: " + deployment.url);
      try
      {
         // Create a local copy of that File, the sdi keeps track of the copy directory
         if (deployment.localUrl == null)
         {
            makeLocalCopy(deployment);
            // initialize the unified classloaders for this deployment
//            deployment.createClassLoaders();
            URL[] localCP = {deployment.localUrl};
            deployment.localCl = new URLClassLoader(localCP);
         } // end of if ()

         // What deployer is able to deploy this file
         findDeployer(deployment);

         if(deployment.deployer == null)
         {
            deployment.state = DeploymentState.INIT_WAITING_DEPLOYER;
            log.debug("deployment waiting for deployer: " + deployment.url);
            synchronized (waitingDeployments)
            {
               if (waitingDeployments.contains(deployment) == false)
                  waitingDeployments.add(deployment);
            }
            return false;
         }
         deployment.state = DeploymentState.INIT_DEPLOYER;
         //we have the deployer, continue deployment.
         deployment.deployer.init(deployment);
         // initialize the unified classloaders for this deployment
         deployment.createClassLoaders();
         deployment.state = DeploymentState.INITIALIZED;

         // Add the deployment to the map so we can detect circular deployments
         deploymentMap.put(deployment.url, deployment);

         // create subdeployments as needed
         parseManifestLibraries(deployment);

         log.debug("found " + deployment.subDeployments.size() + " subpackages of " + deployment.url);
         // get sorted subDeployments
         ArrayList sortedSubs = new ArrayList(deployment.subDeployments);
         Collections.sort(sortedSubs, infoSorter);
         for (Iterator lt = sortedSubs.listIterator(); lt.hasNext();)
         {
            init((DeploymentInfo) lt.next());
         }
      }
      catch (Exception e)
      {
         deployment.state = DeploymentState.FAILED;
         DeploymentException.rethrowAsDeploymentException("exception in init of " + deployment.url, e);
      }
      finally
      {
         // whether you do it or not, for the autodeployer
         try
         {
            URL url = deployment.localUrl == null ? deployment.url : deployment.localUrl;

            long lastModified = -1;

            if (url.getProtocol().equals("file"))
               lastModified = new File(url.getFile()).lastModified();
            else
               lastModified = url.openConnection().getLastModified();

            deployment.lastModified=lastModified;
            deployment.lastDeployed=System.currentTimeMillis();
         }
         catch (IOException ignore)
         {
            deployment.lastModified=System.currentTimeMillis();
            deployment.lastDeployed=System.currentTimeMillis();
         }

         synchronized (deploymentList)
         {
            // Do we watch it? Watch only urls outside our copy directory.
            if (!inLocalCopyDir(deployment.url) && deploymentList.contains(deployment) == false)
            {
               deploymentList.add(deployment);
               log.debug("Watching new file: " + deployment.url);
            }
         }
      }
      return true;
   }

   /**
    * The <code>create</code> method is the second internal deployment step.
    * It should set up all information not
    * requiring other components.  for instance, the ejb Container is created,
    * and the proxy bound into jndi.
    *
    * @param deployment a <code>DeploymentInfo</code> value
    * @throws DeploymentException if an error occurs
    */
   private void create(DeploymentInfo deployment) throws DeploymentException
   {
      log.debug("create step for deployment " + deployment.url);
      try
      {
         ArrayList sortedSubs = new ArrayList(deployment.subDeployments);
         Collections.sort(sortedSubs, infoSorter);
         for (Iterator lt = sortedSubs.listIterator(); lt.hasNext();)
         {
            create((DeploymentInfo) lt.next());
         }
         deployment.state = DeploymentState.CREATE_SUBDEPLOYMENTS;

         // Deploy this SDI, if it is a deployable type
         if (deployment.deployer != null)
         {
            deployment.state = DeploymentState.CREATE_DEPLOYER;
            deployment.deployer.create(deployment);
            // See if all mbeans are created...
            deployment.state = DeploymentState.CREATED;
            deployment.status="Created";
            log.debug("Done with create step of deploying " + deployment.shortName);
         }
         else
         {
            log.debug("Still no deployer for package in create step: "  + deployment.shortName);
         } // end of else
      }
      catch (Throwable t)
      {
         log.error("could not create deployment: " + deployment.url, t);
         deployment.status = "Deployment FAILED reason: " + t.getMessage();
         deployment.state = DeploymentState.FAILED;
         DeploymentException.rethrowAsDeploymentException("Could not create deployment: " + deployment.url, t);
      }
   }

   /**
    * The <code>start</code> method is the third and final internal deployment step.
    * The purpose is to set up relationships between components.
    * for instance, ejb links are set up here.
    *
    * @param deployment a <code>DeploymentInfo</code> value
    * @throws DeploymentException if an error occurs
    */
   private void start(DeploymentInfo deployment) throws DeploymentException
   {
      deployment.status = "Starting";
      log.debug("Begin deployment start " + deployment.url);
      try
      {
         ArrayList sortedSubs = new ArrayList(deployment.subDeployments);
         Collections.sort(sortedSubs, infoSorter);
         for (Iterator lt = sortedSubs.listIterator(); lt.hasNext();)
         {
            start((DeploymentInfo) lt.next());
         }
         deployment.state = DeploymentState.START_SUBDEPLOYMENTS;

         // Deploy this SDI, if it is a deployable type
         if (deployment.deployer != null)
         {
            deployment.state = DeploymentState.START_DEPLOYER;
            deployment.deployer.start(deployment);
            // See if all mbeans are started...
            Object[] args = {deployment, DeploymentState.STARTED};
            String[] sig = {"org.jboss.deployment.DeploymentInfo",
               "org.jboss.deployment.DeploymentState"};
            server.invoke(serviceController, "validateDeploymentState",args, sig);
            deployment.status="Deployed";
            log.debug("End deployment start on package: "+ deployment.shortName);
         }
         else
         {
            log.debug("Still no deployer for package in start step: "  + deployment.shortName);
         } // end of else
      }
      catch (Throwable t)
      {
         log.error("could not start deployment: " + deployment.url, t);
         deployment.state = DeploymentState.FAILED;
         deployment.status = "Deployment FAILED reason: " + t.getMessage();
         DeploymentException.rethrowAsDeploymentException("Could not create deployment: " + deployment.url, t);
      }
   }

   /**
    * The <code>findDeployer</code> method attempts to find a deployer for the DeploymentInfo
    * supplied as a parameter.
    *
    * @param sdi a <code>DeploymentInfo</code> value
    */
   private void findDeployer(DeploymentInfo sdi)
   {
      // If there is already a deployer use it
      if( sdi.deployer != null )
      {
         log.debug("using existing deployer "+sdi.deployer);
         return;
      }

      //
      // To deploy directories of beans one should just name the directory
      // mybean.ear/bla...bla, so that the directory gets picked up by the right deployer
      //
      synchronized(deployers)
      {
         for (Iterator iterator = deployers.iterator(); iterator.hasNext(); )
         {
            SubDeployer deployer = (SubDeployer) iterator.next();
            if (deployer.accepts(sdi))
            {
               sdi.deployer = deployer;
               log.debug("using deployer "+deployer);
               return;
            }
         }
      }
      log.debug("No deployer found for url: " + sdi.url);
   }

   /**
    * The <code>parseManifestLibraries</code> method looks into the manifest for classpath
    * goo, and tries to deploy referenced packages.
    *
    * @param sdi a <code>DeploymentInfo</code> value
    * @exception DeploymentException if an error occurs
    */
   private void parseManifestLibraries(DeploymentInfo sdi)
   {
      String classPath = null;

      Manifest mf = sdi.getManifest();

      if( mf != null )
      {
         Attributes mainAttributes = mf.getMainAttributes();
         classPath = mainAttributes.getValue(Attributes.Name.CLASS_PATH);
      }

      if (classPath != null)
      {
         StringTokenizer st = new StringTokenizer(classPath);
         log.debug("resolveLibraries: "+classPath);

         while (st.hasMoreTokens())
         {
            URL lib = null;
            String tk = st.nextToken();
            DeploymentInfo sub = null;
            log.debug("new manifest entry for sdi at "+sdi.shortName+" entry is "+tk);

            try
            {
               if (sdi.isDirectory)
               {
                  File parentDir = new File(sdi.url.getPath()).getParentFile();
                  lib = new File(parentDir, tk).toURL();
               }
               else
               {
                  lib = new URL(sdi.url, tk);
               }

               // Only deploy this if it is not already being deployed
               if ( deploymentMap.containsKey(lib) == false )
               {
                  /* Test that the only deployer for this is the JARDeployer.
                   Any other type of deployment cannot be initiated through
                   a manifest reference.
                  */
                  DeploymentInfo mfRef = new DeploymentInfo(lib, null, getServer());
                  makeLocalCopy(mfRef);
                  URL[] localURL = {mfRef.localUrl};
                  mfRef.localCl = new java.net.URLClassLoader(localURL);
                  findDeployer(mfRef);
                  SubDeployer deployer = mfRef.deployer;
                  if( ( deployer instanceof JARDeployer) == true || deployer == null )
                  {
                     sdi.addLibraryJar(lib);
                  }
                  else
                  {
                     // Its a non-jar deployment that must be deployed seperately
                     log.debug("Found non-jar deployer("+sub.deployer+
                           "), delaying deployment of: "+sub);
                  }
               }
            }
            catch (Exception ignore)
            {
               log.debug("The manifest entry in "+sdi.url+" references URL "+lib+
                  " which could not be opened, entry ignored");
            }
         }
      }
   }

   /**
    * Downloads the jar file or directory the src URL points to.
    * In case of directory it becomes packed to a jar file.
    *
    * @return a File object representing the downloaded module
    * @throws DeploymentException
    */
   private void makeLocalCopy(DeploymentInfo sdi)
   {
      try
      {
         if (sdi.url.getProtocol().equals("file") && (!copyFiles || sdi.isDirectory))
         {
            // If local copies have been disabled, do nothing
            sdi.localUrl = sdi.url;
            return;
         }
         // Are we already in the localCopyDir?
         else if (inLocalCopyDir(sdi.url))
         {
            sdi.localUrl = sdi.url;
            return;
         }
         else
         {
            String shortName = sdi.shortName;
            File localFile = File.createTempFile("tmp", shortName, tempDir);
            sdi.localUrl = localFile.toURL();
            copy(sdi.url, localFile);
         }
      }
      catch (Exception e)
      {
         log.error("Could not make local copy for " + sdi.url, e);
      }
   }

   private boolean inLocalCopyDir(URL url)
   {
      int i = 0;
      String urlTest = url.toString();
      if( urlTest.startsWith("jar:") )
         i = 4;

      return urlTest.startsWith(tempDirString, i);
   }

   protected void copy(URL src, File dest) throws IOException
   {
      log.debug("Copying " + src + " -> " + dest);
      
      // Validate that the dest parent directory structure exists
      File dir = dest.getParentFile();
      if (!dir.exists())
      {
         boolean created = dir.mkdirs();
         if( created == false )
            throw new IOException("mkdirs failed for: "+dir.getAbsolutePath());
      }

      // Remove any existing dest content
      if( dest.exists() == true )
      {
         boolean deleted = Files.delete(dest);
         if( deleted == false )
            throw new IOException("delete of previous content failed for: "+dest.getAbsolutePath());
      }

      if (src.getProtocol().equals("file"))
      {
         File srcFile = new File(src.getFile());
         if (srcFile.isDirectory())
         {
            log.debug("Making zip copy of: " + srcFile);
            // make a jar archive of the directory
            OutputStream out = new BufferedOutputStream(new FileOutputStream(dest));
            JarUtils.jar(out, srcFile.listFiles());
            out.close();
            return;
         }
      }

      InputStream in = new BufferedInputStream(src.openStream());
      OutputStream out = new BufferedOutputStream(new FileOutputStream(dest));
      Streams.copy(in, out);
      out.flush();
      out.close();
      in.close();
   }

   /**
    * The <code>isDeployed</code> method tells you if a package identified by a string
    * representation of a URL is currently deployed.
    *
    * @param url a <code>String</code> value
    * @return a <code>boolean</code> value
    * @exception MalformedURLException if an error occurs
    * @jmx:managed-operation
    */
   public boolean isDeployed(String url)
      throws MalformedURLException
   {
      return isDeployed(new URL(url));
   }

   /**
    * The <code>isDeployed</code> method tells you if a packaged identified by
    * a URL is deployed.
    * @param url an <code>URL</code> value
    * @return a <code>boolean</code> value
    * @jmx:managed-operation
    */
   public boolean isDeployed(URL url)
   {
      DeploymentInfo di = getDeployment(url);
      if (di == null)
      {
         return false;
      } // end of if ()
      return di.state == DeploymentState.STARTED;
   }

   /**
    * The <code>getDeployment</code> method returns the DeploymentInfo
    * object for the URL supplied.
    *
    * @param url an <code>URL</code> value
    * @return a <code>DeploymentInfo</code> value
    * @jmx:managed-operation
    */
   public DeploymentInfo getDeployment(URL url)
   {
      return (DeploymentInfo) deploymentMap.get(url);
   }

   /**
    * The <code>getWatchUrl</code> method returns the URL that, when modified,
    * indicates that a redeploy is needed.
    *
    * @param url an <code>URL</code> value
    * @return a <code>URL</code> value
    * @jmx:managed-operation
    */
   public URL getWatchUrl(URL url)
   {
      DeploymentInfo info = getDeployment(url);
      return info == null ? null : info.watch;
   }

   /** Check the current deployment states and generate a IncompleteDeploymentException
    * if there are mbeans waiting for depedencies.
    * @exception IncompleteDeploymentException
    * @jmx:managed-operation
    */
   public void checkIncompleteDeployments() throws DeploymentException
   {
      try
      {
         Collection waitingForClasses = new HashSet();
         Collection waitingForDepends = (Collection)server.invoke(serviceController,
                                                      "listIncompletelyDeployed",
                                                      new Object[] {},
                                                      new String[] {});
         Collection allServices = (Collection) server.invoke(serviceController,
               "listDeployed",
               new Object[] {},
               new String[] {});
         
         // Weed services that are waiting for other deployments
         Collection rootCause = new HashSet(waitingForDepends);
         Collection missing = new HashSet();
         for (Iterator i = rootCause.iterator(); i.hasNext();)
         {
            ServiceContext ctx = (ServiceContext) i.next();
            for (Iterator j = ctx.iDependOn.iterator(); j.hasNext(); )
            {
               ServiceContext dependee = (ServiceContext) j.next();
               if (dependee.state != ServiceContext.RUNNING)
               {
                  // Add missing mbean
                  if (allServices.contains(dependee) == false)
                     missing.add(dependee);
                  // We are not a root cause
                  i.remove();
                  break;
               }
            }
         }
         // Add missing mbeans to the root cause
         rootCause.addAll(missing);

         IncompleteDeploymentException ide = new IncompleteDeploymentException(
            waitingForClasses,
            waitingForDepends,
            rootCause,
            listIncompletelyDeployed(),
            listWaitingForDeployer());
         if (!ide.isEmpty())
         {
            throw ide;
         } // end of if ()
      }
      catch (JMException jme)
      {
         throw new DeploymentException(JMXExceptionDecoder.decode(jme));
      } // end of try-catch
   }

}
