/*
 *
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.test.cache.test.local;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.TreeCache;
import org.jboss.cache.Fqn;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.transaction.DummyTransactionManager;
import org.jboss.logging.Logger;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.transaction.UserTransaction;
import java.util.Properties;
import java.util.Set;

/**
 * Verifies that there are no read locks held when a transaction ends.
 *
 * @author Bela Ban
 * @version $Id: LockReleaseUnitTestCase.java,v 1.3.2.3 2004/12/30 17:11:57 starksm Exp $
 */
public class LockReleaseUnitTestCase extends TestCase {
   TreeCache cache=null;
   UserTransaction tx=null;
   Logger log;
   Properties p=null;
   String old_factory=null;
   final String FACTORY="org.jboss.cache.transaction.DummyContextFactory";
   final Fqn    NODE1=Fqn.fromString("/test");
   final Fqn    NODE2=Fqn.fromString("/my/test");
   final String KEY="key";
   final String VAL1="val1";
   final String VAL2="val2";


   public LockReleaseUnitTestCase(String name) {
      super(name);
   }

   public void setUp() throws Exception {
      super.setUp();
      old_factory=System.getProperty(Context.INITIAL_CONTEXT_FACTORY);
      System.setProperty(Context.INITIAL_CONTEXT_FACTORY, FACTORY);
      DummyTransactionManager.getInstance();
      if(p == null) {
         p=new Properties();
         p.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.cache.transaction.DummyContextFactory");
      }
      tx=(UserTransaction)new InitialContext(p).lookup("UserTransaction");
   }

   public void tearDown() throws Exception {
      if(cache != null)
         cache.stopService();

      // BW. kind of a hack to destroy jndi binding and thread local tx before next run.
      DummyTransactionManager.destroy();
      if(old_factory != null) {
         System.setProperty(Context.INITIAL_CONTEXT_FACTORY, old_factory);
         old_factory=null;
      }

      if(tx != null) {
         try {
            tx.rollback();
         }
         catch(Throwable t) {
         }
         tx=null;
      }
   }

   TreeCache createCache(IsolationLevel level) throws Exception {
      TreeCache cache=new TreeCache("test", null, 10000);
      cache.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache.setLockAcquisitionTimeout(500);
      cache.setIsolationLevel(level);
      cache.createService();
      cache.startService();
      return cache;
   }



   public void testReadWithReadUncommitted() throws Exception {
      testReadLockRelease(IsolationLevel.READ_UNCOMMITTED);
   }

   public void testWriteWithReadUncommitted() throws Exception {
      testWriteLockRelease(IsolationLevel.READ_UNCOMMITTED);
   }


   public void testReadWithReadCommitted() throws Exception {
      testReadLockRelease(IsolationLevel.READ_COMMITTED);
   }

   public void testWriteWithReadCommitted() throws Exception {
      testWriteLockRelease(IsolationLevel.READ_COMMITTED);
   }


   public void testReadWithRepeatableRead() throws Exception {
      testReadLockRelease(IsolationLevel.REPEATABLE_READ);
   }

   public void testWriteWithRepeatableRead() throws Exception {
      testWriteLockRelease(IsolationLevel.REPEATABLE_READ);
   }


   public void testReadWithSerialzable() throws Exception {
      testReadLockRelease(IsolationLevel.SERIALIZABLE);
   }

   public void testWriteWithSerializable() throws Exception {
      testWriteLockRelease(IsolationLevel.SERIALIZABLE);
   }


   public void testGetKeys() throws Exception {
      cache=createCache(IsolationLevel.REPEATABLE_READ);
      // add initial values outside of TX
      cache.put(NODE1, KEY, VAL1);
      cache.put(NODE2, KEY, VAL1);
      assertEquals("we ran outside of a TX, locks should have been released: ", 0, cache.getNumberOfLocksHeld());

      Set keys=cache.getKeys(NODE1);
      System.out.println("keys of " + NODE1 + " are " + keys);
      assertEquals("getKeys() called outside the TX should have released all locks", 0, cache.getNumberOfLocksHeld());

      tx.begin();
      keys=cache.getKeys(NODE1);
      assertEquals("we should hold 1 read locks now: ", 1, cache.getNumberOfLocksHeld());
      keys=cache.getKeys(NODE2);
      assertEquals("we should hold 3 read locks now: ", 3, cache.getNumberOfLocksHeld());
      tx.commit();
      assertEquals("we should have released all 3 read locks: ", 0, cache.getNumberOfLocksHeld());
   }


   public void testGetChildrenNames() throws Exception {
      cache=createCache(IsolationLevel.REPEATABLE_READ);
      // add initial values outside of TX
      cache.put(NODE1, KEY, VAL1);
      cache.put(NODE2, KEY, VAL1);
      assertEquals("we ran outside of a TX, locks should have been released: ", 0, cache.getNumberOfLocksHeld());

      Set keys=cache.getChildrenNames(NODE2);
      System.out.println("keys of " + NODE2 + " are " + keys);
      assertEquals("getChildrenNames() called outside the TX should have released all locks", 0,
                   cache.getNumberOfLocksHeld());

      tx.begin();
      keys=cache.getChildrenNames(NODE1);
      assertEquals("we should hold 1 read locks now: ", 1, cache.getNumberOfLocksHeld());
      keys=cache.getChildrenNames(NODE2);
      assertEquals("we should hold 3 read locks now: ", 3, cache.getNumberOfLocksHeld());
      tx.commit();
      assertEquals("we should have released all 3 read locks: ", 0, cache.getNumberOfLocksHeld());
   }

   public void testPrint() throws Exception {
      cache=createCache(IsolationLevel.REPEATABLE_READ);
      // add initial values outside of TX
      cache.put(NODE1, KEY, VAL1);
      cache.put(NODE2, KEY, VAL1);
      assertEquals("we ran outside of a TX, locks should have been released: ", 0, cache.getNumberOfLocksHeld());

      cache.print(NODE1);
      assertEquals("print() called outside the TX should have released all locks", 0, cache.getNumberOfLocksHeld());

      tx.begin();
      cache.print(NODE1);
      assertEquals("we should hold 1 read locks now (for print()): ", 1, cache.getNumberOfLocksHeld());
      cache.print(NODE2);
      assertEquals("we should hold 3 read locks now (for print()): ", 3, cache.getNumberOfLocksHeld());
      tx.commit();
      assertEquals("we should have released all 3 read locks: ", 0, cache.getNumberOfLocksHeld());
   }


   void testReadLockRelease(IsolationLevel level) throws Exception {
      cache=createCache(level);
      // add initial values outside of TX
      cache.put(NODE1, KEY, VAL1);
      cache.put(NODE2, KEY, VAL1);

      assertEquals("we ran outside of a TX, locks should have been released: ", 0, cache.getNumberOfLocksHeld());

      tx.begin();
      assertEquals(VAL1, cache.get(NODE1, KEY));
      assertEquals(VAL1, cache.get(NODE2, KEY));
      assertEquals("we should hold 3 read locks now: ", 3, cache.getNumberOfLocksHeld());
      tx.commit();
      assertEquals("we should have released all 3 read locks: ", 0, cache.getNumberOfLocksHeld());
   }

   void testWriteLockRelease(IsolationLevel level) throws Exception {
      cache=createCache(level);
      // add initial values outside of TX
      cache.put(NODE1, KEY, VAL1);
      cache.put(NODE2, KEY, VAL1);

      assertEquals("we ran outside of a TX, locks should have been released: ", 0, cache.getNumberOfLocksHeld());

      tx.begin();
      cache.put(NODE1, KEY, VAL1);
      cache.put(NODE2, KEY, VAL1);
      assertEquals("we should hold 3 write locks now: ", 3, cache.getNumberOfLocksHeld());
      tx.commit();
      assertEquals("we should have released all 3 write locks: ", 0, cache.getNumberOfLocksHeld());
   }

   void log(String msg) {
      log.info("-- [" + Thread.currentThread() + "]: " + msg);
   }


   public static Test suite() throws Exception {
      return new TestSuite(LockReleaseUnitTestCase.class);
   }

   public static void main(String[] args) throws Exception {
      junit.textui.TestRunner.run(suite());
   }


}
