/*
 *
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.test.cache.test.local;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.TreeCache;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.transaction.DummyTransactionManager;
import org.jboss.logging.Logger;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.transaction.UserTransaction;
import java.util.Properties;

/**
 * Tests upgrade locks from read -> write
 *
 * @author Bela Ban
 * @version $Id: UpgradeLockUnitTestCase.java,v 1.3.2.3 2004/12/30 17:11:57 starksm Exp $
 */
public class UpgradeLockUnitTestCase extends TestCase {
   TreeCache cache=null;
   UserTransaction tx=null;
   Logger log;
   Properties p=null;
   String old_factory=null;
   final String FACTORY="org.jboss.cache.transaction.DummyContextFactory";
   final String NODE1="/test";
   final String NODE2="/my/test";
   final String KEY="key";
   final String VAL1="val1";
   final String VAL2="val2";


   public UpgradeLockUnitTestCase(String name) {
      super(name);
   }

   public void setUp() throws Exception {
      super.setUp();
      old_factory=System.getProperty(Context.INITIAL_CONTEXT_FACTORY);
      System.setProperty(Context.INITIAL_CONTEXT_FACTORY, FACTORY);
      DummyTransactionManager.getInstance();
      if(p == null) {
         p=new Properties();
         p.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.cache.transaction.DummyContextFactory");
      }
      tx=(UserTransaction)new InitialContext(p).lookup("UserTransaction");
   }

   public void tearDown() throws Exception {
      if(cache != null)
         cache.stopService();

      // BW. kind of a hack to destroy jndi binding and thread local tx before next run.
      DummyTransactionManager.destroy();
      if(old_factory != null) {
         System.setProperty(Context.INITIAL_CONTEXT_FACTORY, old_factory);
         old_factory=null;
      }

      if(tx != null) {
         try {
            tx.rollback();
         }
         catch(Throwable t) {
         }
         tx=null;
      }
   }

   TreeCache createCache(IsolationLevel level) throws Exception {
      TreeCache cache=new TreeCache("test", null, 10000);
      cache.setTransactionManagerLookupClass("org.jboss.cache.JBossTransactionManagerLookup");
      cache.setLockAcquisitionTimeout(500);
      cache.setIsolationLevel(level);
      cache.createService();
      cache.startService();
      return cache;
   }


   public void testUpgradeWithNone() throws Exception {
      runTestWithIsolationLevel(IsolationLevel.NONE);
   }


   public void testUpgradeWithReadUncommitted() throws Exception {
      runTestWithIsolationLevel(IsolationLevel.READ_UNCOMMITTED);
   }

   public void testUpgradeWithReadCommitted() throws Exception {
      runTestWithIsolationLevel(IsolationLevel.READ_COMMITTED);
   }

   public void testUpgradeWithRepeatableRead() throws Exception {
      runTestWithIsolationLevel(IsolationLevel.REPEATABLE_READ);
   }

   public void testUpgradeWithSerializable() throws Exception {
      runTestWithIsolationLevel(IsolationLevel.SERIALIZABLE);
   }


   void runTestWithIsolationLevel(IsolationLevel level) throws Exception {
      cache=createCache(level);
      // add initial values outside of TX
      cache.put(NODE1, KEY, VAL1);
      cache.put(NODE2, KEY, VAL1);

      tx.begin();
      try {
         assertEquals(VAL1, cache.get(NODE1, KEY));
         assertEquals(VAL1, cache.get(NODE2, KEY));

         cache.put(NODE1, KEY, VAL2);  // causes read lock to upgrade to r/w lock
         cache.put(NODE2, KEY, VAL2);  // causes read lock to upgrade to r/w lock
         assertEquals(VAL2, cache.get(NODE1, KEY));
         assertEquals(VAL2, cache.get(NODE2, KEY));
         tx.commit();
      }
      catch(Throwable t) {
         if(tx != null)
            tx.rollback();
      }
      assertEquals(VAL2, cache.get(NODE1, KEY));
      assertEquals(VAL2, cache.get(NODE2, KEY));
   }


   void sleep(long timeout) {
      try {
         Thread.sleep(timeout);
      }
      catch(InterruptedException e) {
      }
   }

   void log(String msg) {
      log.info("-- [" + Thread.currentThread() + "]: " + msg);
   }


   public static Test suite() throws Exception {
      // return getDeploySetup(TxUnitTestCase.class, "cachetest.jar");
      return new TestSuite(UpgradeLockUnitTestCase.class);
   }

   public static void main(String[] args) throws Exception {
      junit.textui.TestRunner.run(suite());
   }


}
