/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.test.cluster.test;

import java.net.HttpURLConnection;

import junit.framework.Test;

import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpState;
import org.apache.commons.httpclient.methods.GetMethod;
import org.jboss.test.JBossClusteredTestCase;

/** Tests of http session replication
 *
 * @author  Scott.Stark@jboss.org
 * @version $Revision: 1.1.2.4 $
 */
public class WebSessionTestCase extends JBossClusteredTestCase
{
   public WebSessionTestCase(String name)
   {
      super(name);
   }

   public static Test suite() throws Exception
   {
      Test t1 = getDeploySetup(WebSessionTestCase.class, "dist-ss.war");
      return t1;
   }

   /** This makes 2 requests to the jbosstest.cluster.node0 /dist-ss/StatefulSessionServlet
    * followed by 2 requests to the jbosstest.cluster.node1 /dist-ss/StatefulSessionServlet
    * using the same session ID to validate that the session is replicated
    * with the current value and updated correctly. The session AccessCount
    * value is returned via the X-AccessCount header which should be 4 after
    * the last request.
    * 
    * @throws Exception
    */ 
   public void testServletSessionFailover()
      throws Exception
   {
      getLog().debug("+++ testServletSessionFailover");

      String[] servers = super.getServers();
      String[] httpURLs  = super.getHttpURLs();
      // Access the StatefulSessionServlet of dist-ss.war@server0 twice
      String baseURL0 = httpURLs[0];
      HttpClient httpConn = new HttpClient();
      GetMethod servletGet0 = new GetMethod(baseURL0+"/dist-ss/StatefulSessionServlet/");
      int responseCode = httpConn.executeMethod(servletGet0);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      Header accessCount = servletGet0.getResponseHeader("X-AccessCount");
      int count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 1", count == 1);
      // Get the state for the JSESSIONID
      HttpState state = httpConn.getState();
      servletGet0 = new GetMethod(baseURL0+"/dist-ss/StatefulSessionServlet/");
      responseCode = httpConn.executeMethod(servletGet0.getHostConfiguration(),
         servletGet0, state);
      accessCount = servletGet0.getResponseHeader("X-AccessCount");
      count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 2", count == 2);
      // Get the JSESSIONID so we can reset the host
      Cookie[] cookies = state.getCookies();
      Cookie sessionID = null;
      for(int c = 0; c < cookies.length; c ++)
      {
         Cookie k = cookies[c];
         if( k.getName().equalsIgnoreCase("JSESSIONID") )
            sessionID = k;
      }
      log.info("Saw JSESSIONID="+sessionID);
      // Reset the domain so that the cookie will be sent to server1
      sessionID.setDomain(servers[1]);
      state.addCookie(sessionID);

      // Access the StatefulSessionServlet of dist-ss.war@server1 twice
      String baseURL1 = httpURLs[1];
      GetMethod servletGet1 = new GetMethod(baseURL1+"/dist-ss/StatefulSessionServlet/");
      responseCode = httpConn.executeMethod(servletGet1.getHostConfiguration(),
         servletGet1, state);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      accessCount = servletGet1.getResponseHeader("X-AccessCount");
      count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 3", count == 3);
      servletGet1 = new GetMethod(baseURL1+"/dist-ss/StatefulSessionServlet/");
      responseCode = httpConn.executeMethod(servletGet1.getHostConfiguration(),
         servletGet1, state);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      accessCount = servletGet1.getResponseHeader("X-AccessCount");
      count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 4", count == 4);
   }

   /** This makes 4 requests alternating between the jbosstest.cluster.node0
    * /dist-ss/StatefulSessionServlet and jbosstest.cluster.node1 /dist-ss/StatefulSessionServlet
    * using the same session ID to validate that the session is replicated
    * with the current value and updated correctly. The session AccessCount
    * value is returned via the X-AccessCount header which should be 4 after
    * the last request.
    * 
    * This test currently does not work because the JBossCache http session
    * layer does not invalidate the local session when a remote session is
    * updated.
    * @todo fix the clustering to update the sessions for loadbalancing
    * @throws Exception
    */ 
   public void badtestServletSessionLoadBalancing()
      throws Exception
   {
      getLog().debug("+++ testServletSessionLoadBalancing");

      String[] servers = getServers();
      String baseURL0 = System.getProperty("jbosstest.cluster.node0.http.url")+ "/";
      String baseURL1 = baseURL0;
      if( servers.length > 1 )
      {
        baseURL1 = System.getProperty("jbosstest.cluster.node1.http.url") + "/";
      }
      // Access the StatefulSessionServlet of dist-ss.war@server0 twice
      HttpClient httpConn = new HttpClient();
      GetMethod servletGet0 = new GetMethod(baseURL0+"/dist-ss/StatefulSessionServlet/");
      int responseCode = httpConn.executeMethod(servletGet0);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      Header accessCount = servletGet0.getResponseHeader("X-AccessCount");
      int count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 1", count == 1);
      // Get the state for the JSESSIONID
      HttpState state = httpConn.getState();
      // Get the JSESSIONID so we can reset the host
      Cookie[] cookies = state.getCookies();
      Cookie sessionID = null;
      for(int c = 0; c < cookies.length; c ++)
      {
         Cookie k = cookies[c];
         if( k.getName().equalsIgnoreCase("JSESSIONID") )
            sessionID = k;
      }
      log.info("Saw JSESSIONID="+sessionID);

      // Reset the domain so that the cookie will be sent to server1
      sessionID.setDomain(servers[1]);
      state.addCookie(sessionID);      
      // Access the StatefulSessionServlet of dist-ss.war@server1 twice
      GetMethod servletGet1 = new GetMethod(baseURL1+"/dist-ss/StatefulSessionServlet/");
      responseCode = httpConn.executeMethod(servletGet1.getHostConfiguration(),
         servletGet1, state);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      accessCount = servletGet1.getResponseHeader("X-AccessCount");
      count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 2", count == 2);

      // Reset the domain so that the cookie will be sent to server0
      sessionID.setDomain(servers[0]);
      state.addCookie(sessionID);
      servletGet0 = new GetMethod(baseURL0+"/dist-ss/StatefulSessionServlet/");
      responseCode = httpConn.executeMethod(servletGet0.getHostConfiguration(),
         servletGet0, state);
      accessCount = servletGet0.getResponseHeader("X-AccessCount");
      count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 3", count == 3);

      // Reset the domain so that the cookie will be sent to server1
      sessionID.setDomain(servers[1]);
      state.addCookie(sessionID);
      servletGet1 = new GetMethod(baseURL1+"/dist-ss/StatefulSessionServlet/");
      responseCode = httpConn.executeMethod(servletGet1.getHostConfiguration(),
         servletGet1, state);
      assertTrue("Get OK", responseCode == HttpURLConnection.HTTP_OK);
      accessCount = servletGet1.getResponseHeader("X-AccessCount");
      count = Integer.parseInt(accessCount.getValue());
      assertTrue("X-AccessCount == 4", count == 4);
   }

}
