/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
 
package org.jboss.monitor.alarm;

import javax.management.ObjectName;
import javax.management.Notification;

/**
 * AlarmTableNotification
 *
 * userData field, holds a reference to the source Notification
 *
 * @author  <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 *
 * @version $Revision: 1.1.2.2 $
**/
public class AlarmTableNotification
   extends AlarmNotification
{
   // Constants -----------------------------------------------------
   
   /** the type of AlarmTableNotification */
   public static final String ALARM_TABLE_UPDATE = "jboss.alarm.table.update";
   
   // Private Data --------------------------------------------------

   /** the id of the (MBean) server this alarm originated */
   private String serverId;
   
   // AckStuff
   /** the acked/unacked status of the alarm */
   private boolean ackState;

   /** the time the ack/unack happened */
   private long ackTime;
   
   /** the user that performed the ack/unack */
   private String ackUser;
   
   /** the system ack/unack came from */
   private String ackSystem;   
   
   // CTORS ---------------------------------------------------------
   
   /**
    * CTOR, creates an AlarmTableNotification object
    *
    * Same restrictions with AlarmNotification apply
   **/
   public AlarmTableNotification(
      String type, Object source,
      long sequenceNumber, long timeStamp, String message,
      int alarmState, int severity,
      String serverId)
   {
      super(type, source, sequenceNumber, timeStamp, message, alarmState, severity);
      
      this.serverId = serverId;
   }
   
   /**
    * Copy Constructor.
    *
    * Note, userData is not deep copied!
   **/
   public AlarmTableNotification(AlarmTableNotification atn)
   {
      super(
         atn.getType(), atn.getSource(),
         atn.getSequenceNumber(), atn.getTimeStamp(), atn.getMessage(),
         atn.getAlarmState(), atn.getSeverity()
         );
      
      // this is not a deep copy!
      this.setUserData(atn.getUserData());
      
      this.serverId = atn.serverId;
      this.ackState = atn.ackState;
      this.ackTime = atn.ackTime;
      this.ackUser = atn.ackUser;
      this.ackSystem = atn.ackSystem;
   }
   
   // Accessors/Mutators --------------------------------------------
   
   /**
    * Gets serverId
   **/
   public String getServerId()
   {
      return this.serverId;
   }

   /**
    * Gets the acked/unacked status of the alarm
   **/
   public boolean getAckState()
   {
      return this.ackState;
   }
   
   /**
    * Gets the last time the alarm was acked/unacked
   **/
   public long getAckTime()
   {
      return this.ackTime;
   }
   
   /**
    * Gets the user that performed the ack/unack
   **/
   public String getAckUser()
   {
      return this.ackUser;
   }
   
   /**
    * Gets the system that performed the ack/unack
   **/
   public String getAckSystem()
   {
      return this.ackSystem;
   }
   
   /**
    * Sets all ack parameters
   **/
   public void setAckParams(boolean ackState, long ackTime, String ackUser, String ackSystem)
   {
      this.ackState = ackState;
      this.ackTime = ackTime;
      this.ackUser = ackUser;
      this.ackSystem = ackSystem;
   }
   
   /**
    * Returns a key that can be used in AlarmTables (maps)
   **/
   public Object createKey()
   {
      // Must be a Notification!
      Notification n = (Notification)this.getUserData();
      
      return AlarmKey.createKey(this.serverId, n.getSource(), n.getType());
   }
   
   // Object stuff --------------------------------------------------
   
   /**
    * toString()
   **/
   public String toString()
   {
      StringBuffer sbuf = new StringBuffer(256);
      
      sbuf.append(AlarmTableNotification.class.getName());
      sbuf.append(" [ source=").append(this.getSource());
      sbuf.append(", type=").append(this.getType());
      sbuf.append(", sequenceNumber=").append(this.getSequenceNumber());
      sbuf.append(", timeStamp=").append(this.getTimeStamp());
      sbuf.append(", message=").append(this.getMessage());
      sbuf.append(", userData={").append(this.getUserData());
      sbuf.append("}, alarmState=").append(Alarm.STATE_STRINGS[this.getAlarmState()]);
      sbuf.append(", severity=").append(Alarm.SEVERITY_STRINGS[this.getSeverity()]);
      sbuf.append(", serverId=").append(this.serverId);
      sbuf.append(", ackState=").append(this.ackState);
      sbuf.append(", ackTime=").append(this.ackTime);
      sbuf.append(", ackUser=").append(this.ackUser);
      sbuf.append(", ackSystem=").append(this.ackSystem);
      sbuf.append(" ]");
      
      return sbuf.toString();
   }
}
