/***************************************************************************
                          kbearcopyjob.cpp  -  description
                             -------------------
    begin                : fre maj 10 2002
    copyright            : (C) 2003 by Bjrn Sahlstrm
    email                : kbjorn@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//////////////////////////////////////////////////////////////////////
// Qt specific include files
#include <qtimer.h>
#include <qfile.h>
//////////////////////////////////////////////////////////////////////
// KDE specific include files
#include <kapplication.h>
#include <dcopclient.h>
#include <klocale.h>
#include <kio/slave.h>
#include <kio/job.h>
#include <kio/scheduler.h>
#include <kmimemagic.h>
#include <kprotocolinfo.h>
#include <kprotocolmanager.h>
#include <kio/renamedlg.h>
#include <kio/observer.h>
#include <kio/global.h>
#include <ktempfile.h>
#include <kdeversion.h>
#include <kdebug.h>
#include <ksimpleconfig.h>
#include <kmessagebox.h>
//////////////////////////////////////////////////////////////////////
// application specific include files
#include "kbearcopyjob.h"
#include "kbearfilecopyjob.h"
#include "kbeardeletejob.h"
#include "kbearlistjob.h"
#include "transfer.h"
#include "transfergroup.h"
#include "connectionmanager.h"
#include <sys/types.h>
#include <assert.h>

#define REPORT_TIMEOUT 200

#define KIO_ARGS QByteArray packedArgs; QDataStream stream( packedArgs, IO_WriteOnly ); stream


using namespace KBear;

#include "kbearcopyjob.moc"

KBearCopyJob::KBearCopyJob( Transfer* transfer, CopyMode mode, bool asMethod )
  : KIO::Job(false), m_transfer( transfer ), m_mode(mode), m_asMethod(asMethod),
    destinationState(DEST_NOT_STATED), state(STATE_STATING),
    m_totalSize(0), m_processedSize(0), m_fileProcessedSize(0),
    m_processedFiles(0), m_processedDirs(0),
	 m_bCurrentOperationIsLink(false), m_bSingleFileCopy(false), m_bOnlyRenames(mode==Move),
    m_bAutoSkip( false ), m_bOverwriteAll( false ),
    m_conflictError(0), m_reportTimer(0),
	 m_destID( -1 ), m_sourceID( -1 )
{
/* called by the connectionManager
    QTimer::singleShot(0, this, SLOT(slotStart()));
*/
}

KBearCopyJob::KBearCopyJob( const KURL::List& src, const KURL& dest, CopyMode mode, bool asMethod )
  : KIO::Job(false), m_transfer( 0L ), m_mode(mode), m_asMethod(asMethod),
    destinationState(DEST_NOT_STATED), state(STATE_STATING),
    m_totalSize(0), m_processedSize(0), m_fileProcessedSize(0),
    m_processedFiles(0), m_processedDirs(0),
	 m_bCurrentOperationIsLink(false), m_bSingleFileCopy(false), m_bOnlyRenames(mode==Move),
    m_bAutoSkip( false ), m_bOverwriteAll( false ),
    m_conflictError(0), m_reportTimer(0),
	 m_destID( -1 ), m_sourceID( -1 )
{
	m_srcList = src;
	m_currentStatSrc = m_srcList.begin();
	m_dest = dest;
}
void KBearCopyJob::setOverwriteAll( bool overwrite ) {
	m_bOverwriteAll = overwrite;
}
void KBearCopyJob::setSkipAll( bool skip ) {
	m_bAutoSkip = skip;
}
void KBearCopyJob::slotStart( int sourceID, int destID )
{
   m_destID = destID;
   m_sourceID = sourceID;
	if( m_transfer ) {
		m_srcList = m_transfer->sourceURLs();
		m_currentStatSrc = m_srcList.begin();
		m_dest = m_transfer->destURL();
	}
	/**
       We call the functions directly instead of using signals.
       Calling a function via a signal takes approx. 65 times the time
       compared to calling it directly (at least on my machine). aleXXX
    */
    m_reportTimer = new QTimer(this);

    connect(m_reportTimer,SIGNAL(timeout()),this,SLOT(slotReport()));
    m_reportTimer->start(REPORT_TIMEOUT,false);

	 // Stat the dest
    KIO::SimpleJob * job = KIO::stat( m_dest, false, 2, false );
	 ConnectionManager::getInstance()->attachJob( m_destID, job );
    kdDebug(7007) << "KBearCopyJob:stating the dest " << m_dest.prettyURL() << endl;
    addSubjob(job, false);
}
void KBearCopyJob::slotResultStating( Job *job )
{
    kdDebug(7007) << "KBearCopyJob::slotResultStating" << endl;
    // Was there an error while stating the src ?
    if (job->error() && destinationState != DEST_NOT_STATED )
    {
        KURL srcurl = ((KIO::SimpleJob*)job)->url();
        if ( !srcurl.isLocalFile() )
        {
            // Probably : src doesn't exist. Well, over some protocols (e.g. FTP)
            // this info isn't really reliable (thanks to MS FTP servers).
            // We'll assume a file, and try to download anyway.
            kdDebug(7007) << "Error while stating source. Activating hack" << endl;
            subjobs.remove( job );
            assert ( subjobs.isEmpty() ); // We should have only one job at a time ...
            struct KIO::CopyInfo info;
            info.permissions = (mode_t) -1;
            info.mtime = (time_t) -1;
            info.ctime = (time_t) -1;
            info.size = (KIO::filesize_t)-1;
            info.uSource = srcurl;
            info.uDest = m_dest;
            // Append filename or dirname to destination URL, if allowed
            if ( destinationState == DEST_IS_DIR && !m_asMethod )
                info.uDest.addPath( srcurl.fileName() );
            files.append( info );
            ++m_currentStatSrc;
            statNextSrc();
            return;
        }
        // Local file. If stat fails, the file definitely doesn't exist.
        KIO::Job::slotResult( job ); // will set the error and emit result(this)
        return;
    }

    // Is it a file or a dir ?
    KIO::UDSEntry entry = ((KIO::StatJob*)job)->statResult();
    bool bDir = false;
    bool bLink = false;
    KIO::UDSEntry::ConstIterator it2 = entry.begin();
    for( ; it2 != entry.end(); it2++ ) {
        if ( ((*it2).m_uds) == KIO::UDS_FILE_TYPE )
            bDir = S_ISDIR( (mode_t)(*it2).m_long );
        else if ( ((*it2).m_uds) == KIO::UDS_LINK_DEST )
            bLink = !((*it2).m_str.isEmpty());
    }

    if ( destinationState == DEST_NOT_STATED )
        // we were stating the dest
    {
        if (job->error())
            destinationState = DEST_DOESNT_EXIST;
        else {
            // Treat symlinks to dirs as dirs here, so no test on bLink
            destinationState = bDir ? DEST_IS_DIR : DEST_IS_FILE;
            kdDebug(7007) << "KBearCopyJob::slotResultStating dest is dir:" << bDir << endl;
        }
        subjobs.remove( job );
        assert ( subjobs.isEmpty() );

        // After knowing what the dest is, we can start stat'ing the first src.
        statNextSrc();
        return;
    }
    // We were stating the current source URL
    m_currentDest = m_dest; // used by slotEntries
    // Create a dummy list with it, for slotEntries
    KIO::UDSEntryList lst;
    lst.append(entry);

    // There 6 cases, and all end up calling slotEntries(job, lst) first :
    // 1 - src is a dir, destination is a directory,
    // slotEntries will append the source-dir-name to the destination
    // 2 - src is a dir, destination is a file, ERROR (done later on)
    // 3 - src is a dir, destination doesn't exist, then it's the destination dirname,
    // so slotEntries will use it as destination.

    // 4 - src is a file, destination is a directory,
    // slotEntries will append the filename to the destination.
    // 5 - src is a file, destination is a file, m_dest is the exact destination name
    // 6 - src is a file, destination doesn't exist, m_dest is the exact destination name
    // Tell slotEntries not to alter the src url
    m_bCurrentSrcIsDir = false;
    slotEntries(job, lst);

    KURL srcurl = ((KIO::SimpleJob*)job)->url();

    subjobs.remove( job );
    assert ( subjobs.isEmpty() ); // We should have only one job at a time ...

    if ( bDir
         && !bLink // treat symlinks as files (no recursion)
         && m_mode != Link ) // No recursion in Link mode either.
    {
        kdDebug(7007) << " Source is a directory " << endl;

        m_bCurrentSrcIsDir = true; // used by slotEntries
        if ( destinationState == DEST_IS_DIR ) // (case 1)
            // Use <desturl>/<directory_copied> as destination, from now on
            m_currentDest.addPath( srcurl.fileName() );
        else if ( destinationState == DEST_IS_FILE ) // (case 2)
        {
            m_error = KIO::ERR_IS_FILE;
            m_errorText = m_dest.prettyURL();
            emitResult();
            return;
        }
        else // (case 3)
        {
            // otherwise dest is new name for toplevel dir
            // so the destination exists, in fact, from now on.
            // (This even works with other src urls in the list, since the
            //  dir has effectively been created)
            destinationState = DEST_IS_DIR;
        }

        startListing( srcurl );
    }
    else
    {
        kdDebug(7007) << " Source is a file (or a symlink), or we are linking -> no recursive listing " << endl;
        ++m_currentStatSrc;
        statNextSrc();
    }
}

void KBearCopyJob::slotReport()
{
	 switch (state) {
        case STATE_COPYING_FILES:
            emit processedFiles( this, m_processedFiles );
            if (m_mode==Move) {
                emit moving( this, m_currentSrcURL, m_currentDestURL);
            }
            else if (m_mode==Link) {
                emit linking( this, m_currentSrcURL.path(), m_currentDestURL );
            }
            else {
                emit copying( this, m_currentSrcURL, m_currentDestURL );
            }
            break;

        case STATE_CREATING_DIRS:
            emit processedDirs( this, m_processedDirs );
            break;

        case STATE_STATING:
        case STATE_LISTING:
            emit totalSize( this, m_totalSize );
            emit totalFiles( this, files.count() );
            emit totalDirs( this, dirs.count() );
            if (!dirs.isEmpty())
               emit aboutToCreate( this, dirs );
            if (!files.isEmpty())
               emit aboutToCreate( this, files );
            break;

        default:
            break;
    }
};

void KBearCopyJob::slotEntries(KIO::Job* job, const KIO::UDSEntryList& list)
{
    KIO::UDSEntryListConstIterator it = list.begin();
    KIO::UDSEntryListConstIterator end = list.end();
    for (; it != end; ++it) {
        KIO::UDSEntry::ConstIterator it2 = (*it).begin();
        struct KIO::CopyInfo info;
        info.permissions = -1;
        info.mtime = (time_t) -1;
        info.ctime = (time_t) -1;
        info.size = (KIO::filesize_t)-1;
        QString relName;
        bool isDir = false;
        for( ; it2 != (*it).end(); it2++ ) {
            switch ((*it2).m_uds) {
                case KIO::UDS_FILE_TYPE:
                    //info.type = (mode_t)((*it2).m_long);
                    isDir = S_ISDIR( (mode_t)((*it2).m_long) );
                    break;
                case KIO::UDS_NAME:
                    relName = (*it2).m_str;
                    break;
                case KIO::UDS_LINK_DEST:
                    info.linkDest = (*it2).m_str;
                    break;
                case KIO::UDS_ACCESS:
                    info.permissions = ((*it2).m_long);
                    break;
                case KIO::UDS_SIZE:
                    info.size = (KIO::filesize_t)((*it2).m_long);
                    m_totalSize += info.size;
                    break;
                case KIO::UDS_MODIFICATION_TIME:
                    info.mtime = (time_t)((*it2).m_long);
                    break;
                case KIO::UDS_CREATION_TIME:
                    info.ctime = (time_t)((*it2).m_long);
                default:
                    break;
            }
        }
        if (relName != ".." && relName != ".")
        {
            kdDebug(7007) << "KBearCopyJob::slotEntries '" << relName << "'" << endl;
            info.uSource = ((KIO::SimpleJob *)job)->url();
            if ( m_bCurrentSrcIsDir ) // Only if src is a directory. Otherwise uSource is fine as is
                info.uSource.addPath( relName );
            info.uDest = m_currentDest;
            kdDebug(7007) << "uDest(1)=" << info.uDest.prettyURL() << endl;
            // Append filename or dirname to destination URL, if allowed
            if ( destinationState == DEST_IS_DIR && !m_asMethod )
            {
                // Here we _really_ have to add some filename to the dest.
                // Otherwise, we end up with e.g. dest=..../Desktop/ itself.
                // (This can happen when dropping a link to a webpage with no path)
                if ( relName.isEmpty() )
                    info.uDest.addPath( KIO::encodeFileName( info.uSource.prettyURL() ) );
                else
                    info.uDest.addPath( relName );
            }
            kdDebug(7007) << "uDest(2)=" << info.uDest.prettyURL() << endl;
            if ( info.linkDest.isEmpty() && (isDir /*S_ISDIR(info.type)*/) && m_mode != Link ) // Dir
            {
                dirs.append( info ); // Directories
                emit totalFiles( this, dirs.count() );
					 if (m_mode == Move)
                    dirsToRemove.append( info.uSource );
            }
            else {
                files.append( info ); // Files and any symlinks
					 emit totalFiles( this, files.count() );
            }
        }
    }
}

void KBearCopyJob::statNextSrc()
{
    if ( m_currentStatSrc != m_srcList.end() )
    {
        m_currentSrcURL = (*m_currentStatSrc);
        if ( m_mode == Link )
        {
            // Skip the "stating the source" stage, we don't need it for linking
            m_currentDest = m_dest;
            struct KIO::CopyInfo info;
            info.permissions = -1;
            info.mtime = (time_t) -1;
            info.ctime = (time_t) -1;
            info.size = (KIO::filesize_t)-1;
            info.uSource = m_currentSrcURL;
            info.uDest = m_currentDest;
            // Append filename or dirname to destination URL, if allowed
            if ( destinationState == DEST_IS_DIR && !m_asMethod )
            {
                if (
                    (m_currentSrcURL.protocol() == info.uDest.protocol()) &&
                    (m_currentSrcURL.host() == info.uDest.host()) &&
                    (m_currentSrcURL.port() == info.uDest.port()) &&
                    (m_currentSrcURL.user() == info.uDest.user()) &&
                    (m_currentSrcURL.pass() == info.uDest.pass()) )
                {
                    // This is the case of creating a real symlink
                    info.uDest.addPath( m_currentSrcURL.fileName() );
                }
                else
                {
                    // Different protocols, we'll create a .desktop file
                    // We have to change the extension anyway, so while we're at it,
                    // name the file like the URL
                    info.uDest.addPath( KIO::encodeFileName( m_currentSrcURL.prettyURL() )+".desktop" );
                }
            }
            files.append( info ); // Files and any symlinks
            ++m_currentStatSrc;
            statNextSrc(); // we could use a loop instead of a recursive call :)
        }
        // If moving, before going for the full stat+[list+]copy+del thing, try to rename
        else if ( m_mode == Move &&
                  (m_currentSrcURL.protocol() == m_dest.protocol()) &&
                  (m_currentSrcURL.host() == m_dest.host()) &&
                  (m_currentSrcURL.port() == m_dest.port()) &&
                  (m_currentSrcURL.user() == m_dest.user()) &&
                  (m_currentSrcURL.pass() == m_dest.pass()) )
        {
            KURL dest = m_dest;
            // Append filename or dirname to destination URL, if allowed
            if ( destinationState == DEST_IS_DIR && !m_asMethod )
                dest.addPath( m_currentSrcURL.fileName() );
            kdDebug(7007) << "This seems to be a suitable case for trying to rename before stat+[list+]copy+del" << endl;
            state = STATE_RENAMING;

            struct KIO::CopyInfo info;
            info.permissions = -1;
            info.mtime = (time_t) -1;
            info.ctime = (time_t) -1;
            info.size = (KIO::filesize_t)-1;
            info.uSource = m_currentSrcURL;
            info.uDest = dest;
            QValueList<KIO::CopyInfo> files;
            files.append(info);
            emit aboutToCreate( this, files );

            KIO::SimpleJob * newJob = KIO::rename( m_currentSrcURL, dest, false /*no overwrite */);
            ConnectionManager::getInstance()->attachJob( m_sourceID, newJob );
            addSubjob( newJob );
            if ( m_currentSrcURL.directory() != dest.directory() ) // For the user, moving isn't renaming. Only renaming is.
                m_bOnlyRenames = false;
        }
        else
        {
            // if the file system doesn't support deleting, we do not even stat
            if (m_mode == Move && !KProtocolInfo::supportsDeleting(m_currentSrcURL)) {
                KMessageBox::information( 0, KIO::buildErrorString(KIO::ERR_CANNOT_DELETE, m_currentSrcURL.prettyURL()));
			++m_currentStatSrc;
                statNextSrc(); // we could use a loop instead of a recursive call :)
                return;
            }
            // Stat the next src url
            KIO::SimpleJob * job = KIO::stat( m_currentSrcURL, true, 2, false );
				ConnectionManager::getInstance()->attachJob( m_sourceID, job );
            kdDebug(7007) << "KIO::stat on " << m_currentSrcURL.prettyURL() << endl;
            state = STATE_STATING;
            addSubjob(job, false );
            m_currentDestURL=m_dest;
            m_bOnlyRenames = false;
        }
    } else
    {
        // Finished the stat'ing phase
        // First make sure that the totals were correctly emitted
        state = STATE_STATING;
        slotReport();
        // Check if we are copying a single file
        m_bSingleFileCopy = ( files.count() == 1 && dirs.isEmpty() );
        // Then start copying things
        state = STATE_CREATING_DIRS;
        createNextDir();
    }
}


void KBearCopyJob::startListing( const KURL & src )
{
    state = STATE_LISTING;
    KBearListJob * newjob = KBearListJob::listRecursive( m_sourceID, src, false, true );
	 ConnectionManager::getInstance()->attachJob( m_sourceID, newjob );
    connect(newjob, SIGNAL(entries( KIO::Job *,
                                    const KIO::UDSEntryList& )),
            SLOT( slotEntries( KIO::Job*,
                               const KIO::UDSEntryList& )));
    addSubjob( newjob, false );
}

void KBearCopyJob::skip( const KURL & sourceUrl )
{
    // Check if this is one if toplevel sources
    // IF yes, remove it from m_srcList, for a correct FilesRemoved() signal
    kdDebug(7007) << "KBearCopyJob::skip: looking for " << sourceUrl.prettyURL() << endl;
    KURL::List::Iterator sit = m_srcList.find( sourceUrl );
    if ( sit != m_srcList.end() )
    {
        kdDebug(7007) << "KBearCopyJob::skip: removing " << sourceUrl.prettyURL() << " from list" << endl;
        m_srcList.remove( sit );
    }
    dirsToRemove.remove( sourceUrl );
}

void KBearCopyJob::slotResultCreatingDirs( KIO::Job * job )
{
    // The dir we are trying to create:
    QValueList<KIO::CopyInfo>::Iterator it = dirs.begin();
    // Was there an error creating a dir ?
    if ( job->error() )
    {
        m_conflictError = job->error();
        if ( (m_conflictError == KIO::ERR_DIR_ALREADY_EXIST)
             || (m_conflictError == KIO::ERR_FILE_ALREADY_EXIST) )
        {
            KURL oldURL = ((KIO::SimpleJob*)job)->url();
            // Should we skip automatically ?
            if ( m_bAutoSkip ) {
                // We dont want to copy files in this directory, so we put it on the skip list
                m_skipList.append( oldURL.path( 1 ) );
                skip( oldURL );
                dirs.remove( it ); // Move on to next dir
            } else if ( m_bOverwriteAll ) { // overwrite all => just skip
                emit copyingDone( this, ( *it ).uSource, ( *it ).uDest, true /* directory */, false /* renamed */ );
                dirs.remove( it ); // Move on to next dir
            } else
            {
                assert( ((KIO::SimpleJob*)job)->url().url() == (*it).uDest.url() );
                subjobs.remove( job );
                assert ( subjobs.isEmpty() ); // We should have only one job at a time ...

                // We need to stat the existing dir, to get its last-modification time
                KURL existingDest( (*it).uDest );
                KIO::SimpleJob * newJob = KIO::stat( existingDest, false, 2, false );
					 ConnectionManager::getInstance()->attachJob( m_destID, newJob );
                kdDebug(7007) << "KIO::stat for resolving conflict on " << existingDest.prettyURL() << endl;
                state = STATE_CONFLICT_CREATING_DIRS;
                addSubjob(newJob, false);
                return; // Don't move to next dir yet !
            }
        }
        else
        {
            // Severe error, abort
            KIO::Job::slotResult( job ); // will set the error and emit result(this)
            return;
        }
    }
    else // no error : remove from list, to move on to next dir
    {
       //this is required for the undo feature
        emit copyingDone( this, (*it).uSource, (*it).uDest, true, false );
        dirs.remove( it );
    }

    m_processedDirs++;
    emit processedDirs( this, m_processedDirs );
    subjobs.remove( job );
    assert ( subjobs.isEmpty() ); // We should have only one job at a time ...
    createNextDir();
}

void KBearCopyJob::slotResultConflictCreatingDirs( KIO::Job * job )
{
    // We come here after a conflict has been detected and we've stated the existing dir

    // The dir we were trying to create:
    QValueList<KIO::CopyInfo>::Iterator it = dirs.begin();
    // Its modification time:
    time_t destmtime = (time_t)-1;
    time_t destctime = (time_t)-1;
    KIO::filesize_t destsize = 0;
    KIO::UDSEntry entry = ((KIO::StatJob*)job)->statResult();
    KIO::UDSEntry::ConstIterator it2 = entry.begin();
    for( ; it2 != entry.end(); it2++ ) {
        switch ((*it2).m_uds) {
            case KIO::UDS_MODIFICATION_TIME:
                destmtime = (time_t)((*it2).m_long);
                break;
            case KIO::UDS_CREATION_TIME:
                destctime = (time_t)((*it2).m_long);
                break;
            case KIO::UDS_SIZE:
                destsize = (*it2).m_long;
                break;
        }
    }
    subjobs.remove( job );
    assert ( subjobs.isEmpty() ); // We should have only one job at a time ...

    // Always multi and skip (since there are files after that)
    KIO::RenameDlg_Mode mode = (KIO::RenameDlg_Mode)( KIO::M_MULTI | KIO::M_SKIP );
    // Overwrite only if the existing thing is a dir (no chance with a file)
    if ( m_conflictError == KIO::ERR_DIR_ALREADY_EXIST )
        mode = (KIO::RenameDlg_Mode)( mode | KIO::M_OVERWRITE );

    QString existingDest = (*it).uDest.path();
    QString newPath;
    if (m_reportTimer)
        m_reportTimer->stop();

	 KIO::RenameDlg_Result r = Observer::self()->open_RenameDlg( this, i18n("Directory Already Exists"),
                                         (*it).uSource.prettyURL(0, KURL::StripFileProtocol),
                                         (*it).uDest.prettyURL(0, KURL::StripFileProtocol),
                                         mode, newPath,
                                         (*it).size, destsize,
                                         (*it).ctime, destctime,
                                         (*it).mtime, destmtime );
    if (m_reportTimer)
        m_reportTimer->start(REPORT_TIMEOUT,false);

    switch ( r ) {
        case KIO::R_CANCEL:
            m_error = KIO::ERR_USER_CANCELED;
            emitResult();
            return;
        case KIO::R_RENAME:
        {
            QString oldPath = (*it).uDest.path( 1 );
            KURL newUrl( (*it).uDest );
            newUrl.setPath( newPath );
            emit renamed( this, (*it).uDest, newUrl ); // for e.g. kpropsdlg

            // Change the current one and strip the trailing '/'
            (*it).uDest = newUrl.path( -1 );
            newPath = newUrl.path( 1 ); // With trailing slash
            QValueList<KIO::CopyInfo>::Iterator renamedirit = it;
            renamedirit++;
            // Change the name of subdirectories inside the directory
            for( ; renamedirit != dirs.end() ; ++renamedirit )
            {
                QString path = (*renamedirit).uDest.path();
                if ( path.left(oldPath.length()) == oldPath )
                    (*renamedirit).uDest.setPath( path.replace( 0, oldPath.length(), newPath ) );
            }
            // Change filenames inside the directory
            QValueList<KIO::CopyInfo>::Iterator renamefileit = files.begin();
            for( ; renamefileit != files.end() ; ++renamefileit )
            {
                QString path = (*renamefileit).uDest.path();
                if ( path.left(oldPath.length()) == oldPath )
                    (*renamefileit).uDest.setPath( path.replace( 0, oldPath.length(), newPath ) );
            }
            if (!dirs.isEmpty())
                emit aboutToCreate( this, dirs );
            if (!files.isEmpty())
                emit aboutToCreate( this, files );
        }
        break;
        case KIO::R_AUTO_SKIP:
           m_bAutoSkip = true;
           if( m_transfer )
			  	m_transfer->group()->setSkip( m_transfer );
            // fall through
        case KIO::R_SKIP:
            m_skipList.append( existingDest );
            skip( (*it).uSource );
            // Move on to next dir
            dirs.remove( it );
            break;
        case KIO::R_OVERWRITE:
            m_overwriteList.append( existingDest );
            emit copyingDone( this, ( *it ).uSource, ( *it ).uDest, true /* directory */, false /* renamed */ );
            // Move on to next dir
            dirs.remove( it );
            break;
        case KIO::R_OVERWRITE_ALL:
            m_bOverwriteAll = true;
            if( m_transfer )
					m_transfer->group()->setOverWrite( m_transfer );
            // Move on to next dir
            emit copyingDone( this, ( *it ).uSource, ( *it ).uDest, true /* directory */, false /* renamed */ );
				dirs.remove( it );
            break;
        default:
            assert( 0 );
    }
    state = STATE_CREATING_DIRS;
    m_processedDirs++;
    emit processedDirs( this, m_processedDirs );
    createNextDir();
}

void KBearCopyJob::createNextDir()
{
    KURL udir;
    if ( !dirs.isEmpty() )
    {
        // Take first dir to create out of list
        QValueList<KIO::CopyInfo>::Iterator it = dirs.begin();
        // Is this URL on the skip list or the overwrite list ?
        while( it != dirs.end() && udir.isEmpty() )
        {
            QString dir = (*it).uDest.path();
            bool bCreateDir = true; // we'll create it if it's not in any list

            QStringList::Iterator sit = m_skipList.begin();
            for( ; sit != m_skipList.end() && bCreateDir; sit++ )
                // Is dir a subdirectory of *sit ?
                if ( *sit == dir.left( (*sit).length() ) )
                    bCreateDir = false; // skip this dir

            if ( !bCreateDir ) {
                dirs.remove( it );
                it = dirs.begin();
            } else
                udir = (*it).uDest;
        }
    }
    if ( !udir.isEmpty() ) // any dir to create, finally ?
    {
        // Create the directory - with default permissions so that we can put files into it
        #warning TODO : change permissions once all is finished
        KIO::SimpleJob *newJob = KIO::mkdir( udir, -1 );
		  ConnectionManager::getInstance()->attachJob( m_destID, newJob );

        m_currentDestURL = udir;

        addSubjob(newJob);
        return;
    }
    else // we have finished creating dirs
    {
        state = STATE_COPYING_FILES;
        m_processedFiles++; // Ralf wants it to start a 1, not 0
        copyNextFile();
    }
}

void KBearCopyJob::slotResultCopyingFiles( Job * job )
{
    // The file we were trying to copy:
    QValueList<KIO::CopyInfo>::Iterator it = files.begin();
    if ( job->error() )
    {
        // Should we skip automatically ?
        if ( m_bAutoSkip )
        {
            skip( (*it).uSource );
            files.remove( it ); // Move on to next file
        }
        else
        {
            m_conflictError = job->error(); // save for later
            // Existing dest ?
            if ( ( m_conflictError == KIO::ERR_FILE_ALREADY_EXIST )
                 || ( m_conflictError == KIO::ERR_DIR_ALREADY_EXIST ) )
            {
                subjobs.remove( job );
                assert ( subjobs.isEmpty() );
                // We need to stat the existing file, to get its last-modification time
                KURL existingFile( (*it).uDest );
                KIO::SimpleJob * newJob = KIO::stat( existingFile, false, 2, false );
					 ConnectionManager::getInstance()->attachJob( m_destID, newJob );
                kdDebug(7007) << "KIO::stat for resolving conflict on " << existingFile.prettyURL() << endl;
                state = STATE_CONFLICT_COPYING_FILES;
                addSubjob(newJob, false);
                return; // Don't move to next file yet !
            }
            else
            {
                if ( m_bCurrentOperationIsLink && job->inherits( "KBearDeleteJob" ) )
                {
                    // Very special case, see a few lines below
                    // We are deleting the source of a symlink we successfully moved... ignore error
                    files.remove( it );
                } else {
                    // Go directly to the conflict resolution, there is nothing to stat
                    slotResultConflictCopyingFiles( job );
                    return;
                }
            }
        }
    } else // no error
    {
        // Special case for moving links. That operation needs two jobs, unlike others.
        if ( m_bCurrentOperationIsLink && m_mode == Move
             && !job->inherits( "KBearDeleteJob" ) // Deleting source not already done
             )
        {
            subjobs.remove( job );
            assert ( subjobs.isEmpty() );
            // The only problem with this trick is that the error handling for this del operation
            // is not going to be right... see 'Very special case' above.
            KIO::Job * newjob = KBearDeleteJob::del( (*it).uSource, false /*don't shred*/, false /*no GUI*/ );
            static_cast<KBearDeleteJob*>(newjob)->start( m_sourceID );
            addSubjob( newjob, false );
            return; // Don't move to next file yet !
        }

        if ( m_bCurrentOperationIsLink )
        {
            QString target = ( m_mode == Link ? (*it).uSource.path() : (*it).linkDest );
            //required for the undo feature
            emit copyingLinkDone( this, (*it).uSource, target, (*it).uDest );
        }
        else
            //required for the undo feature
            emit copyingDone( this, (*it).uSource, (*it).uDest, false, false );
        // remove from list, to move on to next file
        files.remove( it );
    }
    m_processedFiles++;

    // clear processed size for last file and add it to overall processed size
    m_processedSize += m_fileProcessedSize;
    m_fileProcessedSize = 0;

    kdDebug(7007) << files.count() << " files remaining" << endl;
    subjobs.remove( job );
    assert ( subjobs.isEmpty() ); // We should have only one job at a time ...
    copyNextFile();
}

void KBearCopyJob::slotResultConflictCopyingFiles( KIO::Job * job )
{
    // We come here after a conflict has been detected and we've stated the existing file
    // The file we were trying to create:
    QValueList<KIO::CopyInfo>::Iterator it = files.begin();

    KIO::RenameDlg_Result res;
    QString newPath;

    if (m_reportTimer)
        m_reportTimer->stop();

    if ( ( m_conflictError == KIO::ERR_FILE_ALREADY_EXIST )
      || ( m_conflictError == KIO::ERR_DIR_ALREADY_EXIST ) )
    {
        // Its modification time:
        time_t destmtime = (time_t)-1;
        time_t destctime = (time_t)-1;
        KIO::filesize_t destsize = 0;
        KIO::UDSEntry entry = ((KIO::StatJob*)job)->statResult();
        KIO::UDSEntry::ConstIterator it2 = entry.begin();
        for( ; it2 != entry.end(); it2++ ) {
            switch ((*it2).m_uds) {
                case KIO::UDS_MODIFICATION_TIME:
                    destmtime = (time_t)((*it2).m_long);
                    break;
                case KIO::UDS_CREATION_TIME:
                    destctime = (time_t)((*it2).m_long);
                    break;
                case KIO::UDS_SIZE:
                    destsize = (*it2).m_long;
                    break;
            }
        }

        // Offer overwrite only if the existing thing is a file
        // If src==dest, use "overwrite-itself"
        KIO::RenameDlg_Mode mode = (KIO::RenameDlg_Mode)
            ( ( m_conflictError == KIO::ERR_DIR_ALREADY_EXIST ? 0 :
             ( (*it).uSource == (*it).uDest ) ? KIO::M_OVERWRITE_ITSELF : KIO::M_OVERWRITE ) );
        if ( files.count() > 0 ) // Not last one
            mode = (KIO::RenameDlg_Mode) ( mode | KIO::M_MULTI | KIO::M_SKIP );
        else
            mode = (KIO::RenameDlg_Mode) ( mode | KIO::M_SINGLE );
        res = Observer::self()->open_RenameDlg( this , m_conflictError == KIO::ERR_FILE_ALREADY_EXIST ?
                                i18n("File Already Exists") : i18n("Already Exists as a Directory"),
                                (*it).uSource.prettyURL(0, KURL::StripFileProtocol),
                                (*it).uDest.prettyURL(0, KURL::StripFileProtocol),
                                mode, newPath,
                              (*it).size, destsize,
                              (*it).ctime, destctime,
                              (*it).mtime, destmtime );

    }
    else
    {
        if ( job->error() == KIO::ERR_USER_CANCELED )
            res = KIO::R_CANCEL;
        else
        {
            KIO::SkipDlg_Result skipResult = Observer::self()->open_SkipDlg( this, files.count() > 0,
                                                                        job->errorString() );

            // Convert the return code from SkipDlg into a RenameDlg code
            res = ( skipResult == KIO::S_SKIP ) ? KIO::R_SKIP :
                         ( skipResult == KIO::S_AUTO_SKIP ) ? KIO::R_AUTO_SKIP :
                                        KIO::R_CANCEL;
        }
    }

    if (m_reportTimer)
        m_reportTimer->start(REPORT_TIMEOUT,false);

    subjobs.remove( job );
    assert ( subjobs.isEmpty() );
    switch ( res ) {
        case KIO::R_CANCEL:
            m_error = KIO::ERR_USER_CANCELED;
            emitResult();
            return;
        case KIO::R_RENAME:
        {
            KURL newUrl( (*it).uDest );
            newUrl.setPath( newPath );
            emit renamed( this, (*it).uDest, newUrl ); // for e.g. kpropsdlg
            (*it).uDest = newUrl;

            QValueList<KIO::CopyInfo> files;
            files.append(*it);
            emit aboutToCreate( this, files );
        }
        break;
        case KIO::R_AUTO_SKIP:
           m_bAutoSkip = true;
           if( m_transfer )
			  		m_transfer->group()->setSkip( m_transfer );
            // fall through
        case KIO::R_SKIP:
            // Move on to next file
            skip( (*it).uSource );
            files.remove( it );
            break;
       case KIO::R_OVERWRITE_ALL:
            m_bOverwriteAll = true;
            if( m_transfer )
					m_transfer->group()->setOverWrite( m_transfer );
            break;
        case KIO::R_OVERWRITE:
            // Add to overwrite list, so that copyNextFile knows to overwrite
            m_overwriteList.append( (*it).uDest.path() );
            break;
        default:
            assert( 0 );
    }
    state = STATE_COPYING_FILES;
    m_processedFiles++;
    emit processedFiles( this, m_processedFiles );
    copyNextFile();
}

void KBearCopyJob::copyNextFile()
{
    bool bCopyFile = false;
    kdDebug(7007) << "KBearCopyJob::copyNextFile()" << endl;
    // Take the first file in the list
    QValueList<KIO::CopyInfo>::Iterator it = files.begin();
    // Is this URL on the skip list ?
    while (it != files.end() && !bCopyFile)
    {
        bCopyFile = true;
        QString destFile = (*it).uDest.path();

        QStringList::Iterator sit = m_skipList.begin();
        for( ; sit != m_skipList.end() && bCopyFile; sit++ )
            // Is destFile in *sit (or a subdirectory of *sit) ?
            if ( *sit == destFile.left( (*sit).length() ) )
                bCopyFile = false; // skip this file

        if (!bCopyFile) {
            files.remove( it );
            it = files.begin();
        }
    }

    if (bCopyFile) // any file to create, finally ?
    {
        // Do we set overwrite ?
        bool bOverwrite = m_bOverwriteAll; // yes if overwrite all
        QString destFile = (*it).uDest.path();
        if ( (*it).uDest == (*it).uSource )
            bOverwrite = false;
        else
        {
            // or if on the overwrite list
            QStringList::Iterator sit = m_overwriteList.begin();
            for( ; sit != m_overwriteList.end() && !bOverwrite; sit++ )
                if ( *sit == destFile.left( (*sit).length() ) )
                    bOverwrite = true;
        }

        m_bCurrentOperationIsLink = false;
        KIO::Job * newjob = 0L;
        if ( m_mode == Link )
        {
            kdDebug(7007) << "Linking" << endl;
            if (
                ((*it).uSource.protocol() == (*it).uDest.protocol()) &&
                ((*it).uSource.host() == (*it).uDest.host()) &&
                ((*it).uSource.port() == (*it).uDest.port()) &&
                ((*it).uSource.user() == (*it).uDest.user()) &&
                ((*it).uSource.pass() == (*it).uDest.pass()) )
            {
                // This is the case of creating a real symlink
                KIO::SimpleJob *newJob = KIO::symlink( (*it).uSource.path(), (*it).uDest, bOverwrite, false /*no GUI*/ );
					 ConnectionManager::getInstance()->attachJob( m_destID, newJob );
                kdDebug(7007) << "KBearCopyJob::copyNextFile : Linking target=" << (*it).uSource.path() << " link=" << (*it).uDest.prettyURL() << endl;
                emit linking( this, (*it).uSource.path(), (*it).uDest );
                m_bCurrentOperationIsLink = true;
                m_currentSrcURL=(*it).uSource;
                m_currentDestURL=(*it).uDest;
                //Observer::self()->slotCopying( this, (*it).uSource, (*it).uDest ); // should be slotLinking perhaps
            } else {
                kdDebug(7007) << "KBearCopyJob::copyNextFile : Linking URL=" << (*it).uSource.prettyURL() << " link=" << (*it).uDest.prettyURL() << endl;
                if ( (*it).uDest.isLocalFile() )
                {
                    bool devicesOk=false;

                    // if the source is a devices url, handle it a littlebit special
                    if ((*it).uSource.protocol()==QString::fromLatin1("devices"))
                    {
                       QByteArray data;
                       QByteArray param;
                       QCString retType;
                       QDataStream streamout(param,IO_WriteOnly);
                       streamout<<(*it).uSource;
                       streamout<<(*it).uDest;
                       if ( kapp->dcopClient()->call( "kded",
                            "mountwatcher", "createLink(KURL, KURL)", param,retType,data,false ) )
                       {
                          QDataStream streamin(data,IO_ReadOnly);
                          streamin>>devicesOk;
                       }
                       if (devicesOk)
                       {
                           files.remove( it );
                           m_processedFiles++;
                           emit processedFiles( this, m_processedFiles );
                           copyNextFile();
                           return;
                       }
                    }

                    if (!devicesOk)
                    {
                       QString path = (*it).uDest.path();
                       kdDebug(7007) << "CopyJob::copyNextFile path=" << path << endl;
                       QFile f( path );
                       if ( f.open( IO_ReadWrite ) )
                       {
                           f.close();
                           KSimpleConfig config( path );
                           config.setDesktopGroup();
                           config.writePathEntry( QString::fromLatin1("URL"), (*it).uSource.url() );
                           config.writeEntry( QString::fromLatin1("Type"), QString::fromLatin1("Link") );
                           QString protocol = (*it).uSource.protocol();
                           if ( protocol == QString::fromLatin1("ftp") )
                               config.writeEntry( QString::fromLatin1("Icon"), QString::fromLatin1("ftp") );
                           else if ( protocol == QString::fromLatin1("http") )
                               config.writeEntry( QString::fromLatin1("Icon"), QString::fromLatin1("www") );
                           else if ( protocol == QString::fromLatin1("info") )
                               config.writeEntry( QString::fromLatin1("Icon"), QString::fromLatin1("info") );
                           else if ( protocol == QString::fromLatin1("mailto") )   // sven:
                               config.writeEntry( QString::fromLatin1("Icon"), QString::fromLatin1("kmail") ); // added mailto: support
                           else
                               config.writeEntry( QString::fromLatin1("Icon"), QString::fromLatin1("unknown") );
                           config.sync();
                           files.remove( it );
                           m_processedFiles++;
                           emit processedFiles( this, m_processedFiles );
                           copyNextFile();
                           return;
                       }
                       else
                       {
                           kdDebug(7007) << "CopyJob::copyNextFile ERR_CANNOT_OPEN_FOR_WRITING" << endl;
                           m_error = KIO::ERR_CANNOT_OPEN_FOR_WRITING;
                           m_errorText = (*it).uDest.path();
                           emitResult();
                           return;
                       }
                    }
                } else {
                    // Todo: not show "link" on remote dirs if the src urls are not from the same protocol+host+...
                    m_error = KIO::ERR_CANNOT_SYMLINK;
                    m_errorText = (*it).uDest.prettyURL();
                    emitResult();
                    return;
                }
            }
        }
        else if ( !(*it).linkDest.isEmpty() &&
                  ((*it).uSource.protocol() == (*it).uDest.protocol()) &&
                  ((*it).uSource.host() == (*it).uDest.host()) &&
                  ((*it).uSource.port() == (*it).uDest.port()) &&
                  ((*it).uSource.user() == (*it).uDest.user()) &&
                  ((*it).uSource.pass() == (*it).uDest.pass()))
            // Copying a symlink - only on the same protocol/host/etc. (#5601, downloading an FTP file through its link),
        {
            KIO::SimpleJob *newJob = KIO::symlink( (*it).linkDest, (*it).uDest, bOverwrite, false /*no GUI*/ );
				ConnectionManager::getInstance()->attachJob( m_destID, newJob );
            newjob = newJob;
            kdDebug(7007) << "KBearCopyJob::copyNextFile : Linking target=" << (*it).linkDest << " link=" << (*it).uDest.prettyURL() << endl;
            emit linking( this, (*it).linkDest, (*it).uDest );
            m_currentSrcURL=(*it).linkDest;
            m_currentDestURL=(*it).uDest;
            //Observer::self()->slotCopying( this, (*it).linkDest, (*it).uDest ); // should be slotLinking perhaps
            m_bCurrentOperationIsLink = true;
            // NOTE: if we are moving stuff, the deletion of the source will be done in slotResultCopyingFiles
        } else if (m_mode == Move) // Moving a file
        {
            KBearFileCopyJob * moveJob = KBearFileCopyJob::file_move( m_destID, m_sourceID, (*it).uSource, (*it).uDest, (*it).permissions, bOverwrite, false, false/*no GUI*/ );
 #if KDE_VERSION_MINOR < 2
            moveJob->setSourceSize( (*it).size );
  #else
           moveJob->setSourceSize64( (*it).size );
 #endif
            newjob = moveJob;
            kdDebug(7007) << "KBearCopyJob::copyNextFile : Moving " << (*it).uSource.prettyURL() << " to " << (*it).uDest.prettyURL() << endl;
            emit moving( this, (*it).uSource, (*it).uDest );
            m_currentSrcURL=(*it).uSource;
            m_currentDestURL=(*it).uDest;
            //Observer::self()->slotMoving( this, (*it).uSource, (*it).uDest );
        }
        else // Copying a file
        {
            // If source isn't local and target is local, we ignore the original permissions
            // Otherwise, files downloaded from HTTP end up with -r--r--r--
            // But for files coming from TAR, we want to preserve permissions -> we use default perms only if from remote
            // The real fix would be KProtocolInfo::inputType(protocol) == T_FILESYSTEM, but we can't access ksycoca from here !
//  changed by kbjorn I want to keep permission
//            bool remoteSource = !(*it).uSource.isLocalFile() && ((*it).uSource.protocol() != "tar"); // HACK
//            int permissions = ( remoteSource && (*it).uDest.isLocalFile() ) ? -1 : (*it).permissions;
            int permissions = (*it).permissions;
            KBearFileCopyJob * copyJob = KBearFileCopyJob::file_copy( m_destID, m_sourceID,(*it).uSource, (*it).uDest, permissions, bOverwrite, false, false/*no GUI*/ );
             copyJob->setParentJob( this );
				 copyJob->setSourceSize( (*it).size );
            newjob = copyJob;
            kdDebug(7007) << "KBearCopyJob::copyNextFile : Copying " << (*it).uSource.prettyURL() << " to " << (*it).uDest.prettyURL() << endl;
            emit copying( this, (*it).uSource, (*it).uDest );
				m_currentSrcURL=(*it).uSource;
            m_currentDestURL=(*it).uDest;
        }
        addSubjob(newjob, false);
        connect( newjob, SIGNAL( processedSize( KIO::Job*, KIO::filesize_t ) ),
                 this, SLOT( slotProcessedSize( KIO::Job*, KIO::filesize_t ) ) );
        connect( newjob, SIGNAL( totalSize( KIO::Job*, KIO::filesize_t ) ),
                 this, SLOT( slotTotalSize( KIO::Job*, KIO::filesize_t ) ) );
    }
    else
    {
        // We're done
        kdDebug(7007) << "copyNextFile finished" << endl;
        deleteNextDir();
    }
}

void KBearCopyJob::deleteNextDir()
{
    if ( m_mode == Move && !dirsToRemove.isEmpty() ) // some dirs to delete ?
    {
        state = STATE_DELETING_DIRS;
        // Take first dir to delete out of list - last ones first !
        KURL::List::Iterator it = dirsToRemove.fromLast();
        KIO::SimpleJob *job = KIO::rmdir( *it );
		  ConnectionManager::getInstance()->attachJob( m_sourceID, job );
        dirsToRemove.remove(it);
        addSubjob( job, false );
    }
    else
    {
        // Finished - tell the world
        if ( !m_bOnlyRenames )
        {
            KURL url( m_dest );
            if ( destinationState != DEST_IS_DIR || m_asMethod )
                url.setPath( url.directory() );
            kdDebug(7007) << "KDirNotify'ing FilesAdded " << url.prettyURL() << endl;
			QByteArray data;
			QDataStream arg( data, IO_WriteOnly );
			arg << url;
			kapp->dcopClient()->send( "*", "KDirNotify", "FilesAdded(const KURL&)", data );

            if ( m_mode == Move && !m_srcList.isEmpty() ) {
			QByteArray data2;
			QDataStream arg2( data2, IO_WriteOnly );
			arg2 << m_srcList;
			kapp->dcopClient()->send( "*", "KDirNotify", "FilesRemoved(const KURL::List&)", data2 );
		  }
        }
        if (m_reportTimer!=0)
            m_reportTimer->stop();

        emitResult();
    }
}

void KBearCopyJob::slotProcessedSize( KIO::Job*, KIO::filesize_t data_size )
{
  kdDebug(7007) << "KBearCopyJob::slotProcessedSize " << (unsigned long)data_size << endl;
  m_fileProcessedSize = data_size;
#if KDE_VERSION_MINOR > 2
  setProcessedSize(m_processedSize + m_fileProcessedSize);
#endif
  if ( m_processedSize + m_fileProcessedSize > m_totalSize )
  {
    m_totalSize = m_processedSize + m_fileProcessedSize;
    kdDebug(7007) << "Adjusting m_totalSize to " << (unsigned int) m_totalSize << endl;
    emit totalSize( this, m_totalSize ); // safety
  }
  kdDebug(7007) << "emit processedSize " << (unsigned int) (m_processedSize + m_fileProcessedSize) << endl;
  emit processedSize( this, m_processedSize + m_fileProcessedSize );
  emitPercent( m_processedSize + m_fileProcessedSize, m_totalSize );
}

void KBearCopyJob::slotTotalSize( KIO::Job*, KIO::filesize_t size )
{
  // Special case for copying a single file
  // This is because some protocols don't implement stat properly
  // (e.g. HTTP), and don't give us a size in some cases (redirection)
  // so we'd rather rely on the size given for the m_sourceID
  if ( m_bSingleFileCopy )
  {
    kdDebug(7007) << "Single file -> updating totalsize to " << (unsigned long)size << endl;
    m_totalSize = size;
    emit totalSize( this, size );
  }
}

void KBearCopyJob::slotResultDeletingDirs( Job * job )
{
    if (job->error())
    {
        // Couldn't remove directory. Well, perhaps it's not empty
        // because the user pressed Skip for a given file in it.
        // Let's not display "Could not remove dir ..." for each of those dir !
    }
    subjobs.remove( job );
    assert ( subjobs.isEmpty() );
    deleteNextDir();
}

void KBearCopyJob::slotResult( Job *job )
{
    kdDebug(7007) << "KBearCopyJob::slotResult() state=" << (int) state << endl;
    // In each case, what we have to do is :
    // 1 - check for errors and treat them
    // 2 - subjobs.remove(job);
    // 3 - decide what to do next

    switch ( state ) {
        case STATE_STATING: // We were trying to stat a src url or the dest
            slotResultStating( job );
            break;
        case STATE_RENAMING: // We were trying to do a direct renaming, before even stat'ing
        {
            int err = job->error();
            subjobs.remove( job );
            assert ( subjobs.isEmpty() );
            // Determine dest again
            KURL dest = m_dest;
            if ( destinationState == DEST_IS_DIR && !m_asMethod )
                dest.addPath( m_currentSrcURL.fileName() );
            if ( err )
            {
                // Direct renaming didn't work. Try renaming to a temp name,
                // this can help e.g. when renaming 'a' to 'A' on a VFAT partition.
                // In that case it's the _same_ dir, we don't want to copy+del (data loss!)
                if ( m_currentSrcURL.isLocalFile() &&
                     m_currentSrcURL.url(-1).lower() == dest.url(-1).lower() &&
                     ( job->error() == KIO::ERR_FILE_ALREADY_EXIST || job->error() == KIO::ERR_DIR_ALREADY_EXIST ) )
                {
                    kdDebug(7007) << "Couldn't rename directly, dest already exists. Detected special case of lower/uppercase renaming in same dir, try with 2 rename calls" << endl;
                    QCString _src( QFile::encodeName(m_currentSrcURL.path()) );
                    QCString _dest( QFile::encodeName(dest.path()) );
                    KTempFile tmpFile( m_currentSrcURL.directory() );
                    QCString _tmp( QFile::encodeName(tmpFile.name()) );
                    kdDebug() << "KBearCopyJob::slotResult KTempFile status:" << tmpFile.status() << " using " << _tmp << " as intermediary" << endl;
                    tmpFile.unlink();
                    if ( ::rename( _src, _tmp ) == 0 )
                    {
                        if ( ::rename( _tmp, _dest ) == 0 )
                        {
                            kdDebug(7007) << "Success." << endl;
                            err = 0;
                        }
                        else
                        {
                            // Revert back to original name!
                            bool b = ::rename( QFile::encodeName(tmpFile.name()), _src );
                            if (!b) {
                                kdError(7007) << "Couldn't rename " << tmpFile.name() << " back to " << _src << " !" << endl;
                                // Severe error, abort
                                KIO::Job::slotResult( job ); // will set the error and emit result(this)
                                return;
                            }
                        }
                    }
                }
            }
            if ( err )
            {
                m_currentSrcURL=*m_currentStatSrc;
                m_currentDestURL=m_dest;
                kdDebug(7007) << "Couldn't rename, reverting to normal way, starting with stat" << endl;
                KIO::SimpleJob * job = KIO::stat( m_currentSrcURL, true, 2, false );
					 ConnectionManager::getInstance()->attachJob( m_sourceID, job );
                kdDebug(7007) << "KIO::stat on " << m_currentSrcURL.prettyURL() << endl;
                state = STATE_STATING;
                addSubjob(job, false);
                m_bOnlyRenames = false;
            }
            else
            {
                kdDebug(7007) << "Renaming succeeded, move on" << endl;
                emit copyingDone( this, *m_currentStatSrc, m_currentDest, true, true );
                ++m_currentStatSrc;
                statNextSrc();
            }
        }
        break;
        case STATE_LISTING: // recursive listing finished
            kdDebug(7007) << "totalSize: " << (unsigned int) m_totalSize << " files: " << files.count() << " dirs: " << dirs.count() << endl;
            // Was there an error ?
            if (job->error())
            {
                KIO::Job::slotResult( job ); // will set the error and emit result(this)
                return;
            }

            subjobs.remove( job );
            assert ( subjobs.isEmpty() );

            ++m_currentStatSrc;
            statNextSrc();
            break;
        case STATE_CREATING_DIRS:
            slotResultCreatingDirs( job );
            break;
        case STATE_CONFLICT_CREATING_DIRS:
            slotResultConflictCreatingDirs( job );
            break;
        case STATE_COPYING_FILES:
            slotResultCopyingFiles( job );
            break;
        case STATE_CONFLICT_COPYING_FILES:
            slotResultConflictCopyingFiles( job );
            break;
        case STATE_DELETING_DIRS:
            slotResultDeletingDirs( job );
            break;
        default:
            assert( 0 );
    }
}
