/***************************************************************************
                          birthdayscanner.cpp  -  description
                             -------------------
    begin                : Tue Feb 26 2002
    copyright           : (C) 2002,2003 by Jan Hambrecht
    email                : jaham@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "birthdayscanner.h"
#include <klocale.h>
#include <kdebug.h>

BirthdayScanner::BirthdayScanner()
{
	m_options = NULL;
	
	// get standard addressbook
	m_ab =  StdAddressBook::self();
	
	connect( m_ab, SIGNAL( addressBookChanged(AddressBook*) ), this, SLOT( OnAddressbookChange(AddressBook*) ) );
}

BirthdayScanner::~BirthdayScanner()
{
}

void BirthdayScanner::setScanOptions( BirthdayOptions *options )
{
	if( ! options ) return;

	m_options = options;
}

bool BirthdayScanner::scan()
{
	Addressee entry;
	AddressBook::Iterator it;

	// set actual date
	m_today = QDate().currentDate();

	// clear buffers
	m_birthdays.clear();
	m_anniversaries.clear();

	// scan all addressbook entries
	for( it = m_ab->begin(); it != m_ab->end(); ++it )
	{
		entry = *it;

		checkEntry( entry, true );
		checkEntry( entry, false );
	}
	
	// returns true, if at least one entry was found, else false
	return ( m_birthdays.count() + m_anniversaries.count() > 0 );
}

uint BirthdayScanner::getEventsCount( bool isBirthday )
{
	if( isBirthday )
		return m_birthdays.count();
	else
		return m_anniversaries.count();
}

bool BirthdayScanner::getEvent( uint index, BirthdayEventData &event, bool isBirthday )
{
	if( isBirthday )
	{
		if( index >= m_birthdays.count() ) return false;
			event = m_birthdays[ index ];
	}
	else
	{
		if( index >= m_anniversaries.count() ) return false;
			event = m_anniversaries[ index ];
	}
	return true;
}

void BirthdayScanner::checkEntry( Addressee &entry, bool isBirthday )
{
	QDate checkDate;
	QDate beforeDate;
	QDate afterDate;

	if( isBirthday )
		checkDate =  entry.birthday().date();
	else
		checkDate = getAnniversary( entry );

	// test if checkDate is valid
	if( ! checkDate.isValid() ) return;

	// transform event into actual year
	checkDate.setYMD(  m_today.year(), checkDate.month(), checkDate.day() );

	// get dates of interval boundaries
	beforeDate = m_today.addDays(  - m_options->DaysPassed() );
	afterDate = m_today.addDays( m_options->DaysComing() );

	// event is near to come and should be selected
	if( m_today <= checkDate && checkDate <= afterDate )
	{
		fillEvent( checkDate, entry, isBirthday );
	}
	// event already passed but should be selected
	if( beforeDate <= checkDate && checkDate < m_today )
	{
		fillEvent( checkDate, entry, isBirthday );
	}
	// event has long passed but could be near for next year
	if( checkDate < beforeDate )
	{
		// move event into the future (next year)
		checkDate = checkDate.addYears( 1 );
		// event is near to come and should be selected
		if( m_today <= checkDate && checkDate <= afterDate )
		{
			fillEvent( checkDate, entry, isBirthday );
		}
	}
}

void BirthdayScanner::fillEvent( QDate &date, Addressee &entry, bool isBirthday )
{
	BirthdayEventData event;
	int days;

	event.setUid( entry.uid() );
	event.setDate( date );

	if( ! entry.formattedName().isEmpty() )
		event.setName( entry.formattedName() );
	else
	{
		// test if lastname is set
		if( ! entry.familyName().isEmpty() )
		{
			// test if firstname is set
			if( ! entry.name().isEmpty() )
				// set lastname, firstname as text
				event.setName( "  " + entry.familyName() + ", " + entry.name() );
			else
				// set lastnam as text
				event.setName( "  " + entry.familyName() );
		}
		else
		{
			// test if firstname is set
			if( ! entry.name().isEmpty() )
				// set firstname as text
				event.setName( "  " + entry.name() );
			else
				// set dummy as text
				event.setName( i18n("  no name available") );
		}
	}
	// set days to event
	days =  QDate().currentDate().daysTo( date );
	event.days() = days;
	// set color for event
	if( days < 0 )
		event.color() = QColor( 128, 128, 128 );
	else if( (uint)days > m_options->DaysHighlight() )
		event.color() = QColor( 0, 0, 0 );
	else
		event.color() = m_options->HighlightColor();
	// set years to event
	if( isBirthday )
	{
		event.years() = date.year() - entry.birthday().date().year();
		m_birthdays.append( event );
		kdDebug() << "kbirthday: [A] " << event.getName() << " " << event.days() << endl;
	}
	else
	{
		event.years() = date.year() -  getAnniversary( entry ).year();
		m_anniversaries.append( event );
		kdDebug() << "kbirthday: [B] " << event.getName() << " " << event.days() << endl;
	}
}

QDate BirthdayScanner::getAnniversary( const Addressee &entry )
{
	QString anniversary = entry.custom( "KADDRESSBOOK", "X-Anniversary" );
	
	if( anniversary.isEmpty() )
		return QDate();
	else
		return QDate::fromString( anniversary, Qt::ISODate );
}

bool BirthdayScanner::getAddressee( const QString &uid, Addressee &addressee )
{
	addressee = m_ab->findByUid( uid );
	
	return ( !addressee.isEmpty() );
}

void BirthdayScanner::OnAddressbookChange( AddressBook *changedAb )
{
	kdDebug() << "kbirthday: received addressbook changed event from ab " << changedAb->identifier() << endl;
	
	// load addressbook when it changes on disc
	if( changedAb->identifier() == m_ab->identifier() )
		m_ab = changedAb;
}

