/***************************************************************************
                          birthdayscanner.h  -  description
                             -------------------
    begin                : Tue Feb 26 2002
    copyright           : (C) 2002,2003 by Jan Hambrecht
    email                : jaham@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef BIRTHDAYSCANNER_H
#define BIRTHDAYSCANNER_H

#include <kabc/stdaddressbook.h>
#include <qvaluelist.h>
#include <qdatetime.h>

#include "birthdayeventdata.h"
#include "birthdayoptions.h"

using namespace KABC;

typedef QValueList<BirthdayEventData> EntryList;

/**
* @brief Scans all addressbook entries for occuring events like birthdays and anniversaries.
* @author Jan Hambrecht
*/
class BirthdayScanner : public QObject
{
	Q_OBJECT
public:
	/** @brief The standard constructor. */
	BirthdayScanner();
	
	/** @brief The destructor */
	~BirthdayScanner();
	
	/**
	* @brief Scans all addressbook entries.
	* @return true if any event in given interval was found, else false
	*/
	bool scan();
	
	/**
	* @brief Gets number of pending events.
	* @param isBirthday set to true to get birthdays count (default), else set to false
	* @return events count
	*/
	uint getEventsCount( bool isBirthday = true );
	
	/**
	* @brief Gets next event entry.
	* @param index index of entry to return
	* @param event event data get filled in here
	* @param isBirthday set to true to get next birthday (default), else set to false
	* @return true if next entry exists, else false
	*/
	bool getEvent( uint index, BirthdayEventData &event, bool isBirthday );
	
	/**
	* @brief Sets scan options.
	* @param options scannning options
	*/
	void setScanOptions( BirthdayOptions *options );
	
	/*
	* @brief Gets addressee from given uid.
	* @param uid uid of addressee 
	* @param addressee out parameter for found addressee
	* @return true, if addressee exists, else false
	*/
	bool getAddressee( const QString &uid, Addressee &addressee );
	
protected:
	
	/**
	* @brief Checks if addressbook entry should be selected.
	* @param entry addressbook entry to check
	* @param isBirthday checks birthday if set to true (default), else checks anniversary
	*/
	void checkEntry( Addressee &entry, bool isBirthday = true );
	
	/**
	* @brief Fills addressee data into event object.
	* @param date event date
	* @param entry addressee data
	* @param isBirthday set to true if event is birthday (default), else set to false
	*/
	void fillEvent( QDate &date, Addressee &entry, bool isBirthday = true );
	
	/**
	* @brief Gets anniversary date form addressee.
	* @param entry addressee data
	* @return date of anniversary
	*/
	QDate getAnniversary( const Addressee &entry );
	
private: // Private attributes
	
	/** @brief List of all found birthday events. */
	EntryList m_birthdays;
	
	/** @brief List of all found anniversary events. */
	EntryList m_anniversaries;
	
	/** @brief The actual date. */
	QDate m_today;
	
	/** @brief The scanning options. */
	BirthdayOptions *m_options;
	
	/** @brief A pointer to kde addressbook. */
	AddressBook *m_ab;

protected slots: 
	/** 
	* @brief Slot for receiving changed signal from addressbook.
	*
	* It reloads the addressbook to take changes into account, when
	* showing pending events.
	*
	* @param changedAb pointer to the changed addressbook
	*/
	void OnAddressbookChange( AddressBook *changedAb );
};

#endif
