/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2004 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "qhacc.h"
#include "qhaccext.h"
#include "copystore.h"
#include "qhaccview.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "qhaccgrwin.h"
#include "qhaccwidget.h"
#include "qhaccrecwin.h"
#include "guiconstants.h"
#include "qhaccdialogs.h"
#include "khaccsounder.h"
#include "qhaccacctdlg.h"
#include "qhaccprefdlg.h"
#include "qhaccacctbox.h"
#include "qhaccjrnlchsr.h"
#include "qhacclineedits.h"
#include "qhaccconstants.h"
#include "qhaccsubsetwin.h"
#include "qhaccwidget.moc"
#include "pixmaps/graph.xpm"
#include "pixmaps/report.xpm"
#include "pixmaps/recacct.xpm"
#include "pixmaps/editacct.xpm"

#include <qfile.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qmessagebox.h>
#include <qinputdialog.h>

#include <kfile.h>
#include <kconfig.h>
#include <kaction.h>
#include <klocale.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kstdaction.h>
#include <qclipboard.h>
#include <kstatusbar.h>
#include <kpopupmenu.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kapplication.h>
#include <kaboutdialog.h>
#include <kglobalsettings.h>

QHaccWidget::QHaccWidget( QHacc * qhacc, QWidget * p, const char * n )
  : KMainWindow( p, n, WDestructiveClose ){

	engine=qhacc;
	QHaccDateEdit::setDateFormat( ( QHaccDateEdit::DateFormat ) engine->getIP( "DATEFORMAT" ) );
	QHaccDateEdit::setDateSeparator( engine->getSP( "DATESEPARATOR" ) );

	engine->setSP( "CURRENCYSEPARATOR", KLocale( "khacc" ).decimalSymbol() );
	engine->setWP( "FONT", KGlobalSettings::generalFont() );

	// build the toolbar before the journal bar, so it shows up first
	tools=toolBar();

	// the account chooser is dockable
	cdock=new QDockWindow( QDockWindow::InDock, this );
	chooser=new QHaccAccountBox( qhacc, cdock );
	cdock->setResizeEnabled( true );
	cdock->setHorizontalStretchable( true );
	cdock->setVerticalStretchable( true );
	cdock->setWidget( chooser );
	addDockWindow( cdock, Qt::Left );

	// the journal bar is dockable
	jdock=new QDockWindow( QDockWindow::InDock, this );
	journaler=new QHaccJournalChooser( qhacc, jdock );
	jdock->setResizeEnabled( true );
	jdock->setHorizontalStretchable( true );
	jdock->setVerticalStretchable( true );
	jdock->setWidget( journaler );
	addDockWindow( jdock, Qt::Top );

	/*
	KDockWidget * mainDock;
	mainDock=createDockWidget( "viewer", map );
	viewer=new QHaccView( qhacc, mainDock );
	mainDock->setWidget( viewer );
	mainDock->setDockSite( KDockWidget::DockCorner );
	mainDock->setEnableDocking( KDockWidget::DockNone );
	setView( mainDock );
	setMainDockWidget( mainDock );

	// the account chooser is dockable
	vdock=createDockWidget( "chooser", map );
	chooser=new QHaccAccountChooser( qhacc, vdock );
	//vdock->setResizeEnabled( true );
	//vdock->setHorizontalStretchable( true );
	//vdock->setVerticalStretchable( true );
	vdock->setWidget( chooser );
	//addDockWindow( vdock, Qt::Left );
	vdock->manualDock( mainDock, KDockWidget::DockLeft );

	// the journal bar is dockable
	jdock=createDockWidget( "journaler", map );
	journaler=new QHaccJournalChooser( qhacc, jdock );
	//jdock->setResizeEnabled( true );
	//jdock->setHorizontalStretchable( true );
	//jdock->setVerticalStretchable( true );
	jdock->setWidget( journaler );
	//addDockWindow( jdock, Qt::Top );
	//jdock->manualDock( mainDock, KDockWidget::DockTop );
	
	// put something in the middle of this window
	*/

	viewer=new QHaccView( qhacc, this );
	setCentralWidget( viewer );
	connectView( viewer );

	sounder=new KHaccSounder();

	connect( engine, SIGNAL( updatedA( const Account&, const Account& ) ),
					 this, SLOT( updateAccount( const Account&, const Account& ) ) );

	connect( engine, SIGNAL( changedP( const QString&, QFont ) ),
					 SLOT( changePref( const QString&, QFont ) ) );
	
	connect( engine, SIGNAL( changedP( const QString&, QString ) ),
					 SLOT( changePref( const QString&, QString ) ) );

	connect( engine, SIGNAL( changedP( const QString&, bool ) ),
					 SLOT( changePref( const QString&, bool ) ) );

	connect( engine, SIGNAL( needSave( bool ) ), SLOT( setSaveFlag( bool ) ) );
	
	connect( engine, SIGNAL( overBudget( const Account&, int ) ),
					 this, SLOT( OBAccount( const Account&, int ) ) );

	connect( chooser, SIGNAL( changedAccount( const Account& ) ),
					 this, SLOT( setAccount( const Account& ) ) );

	connect( journaler, SIGNAL( changedJournal( const Journal& ) ),
					 this, SLOT( setJournal( const Journal& ) ) );

	connect( chooser, SIGNAL( changedAccount( const Account& ) ), 
					 viewer, SLOT( setAccount( const Account& ) ) );
	connect( chooser, SIGNAL( needReportOf( const Account& ) ), 
					 this, SLOT( report( const Account& ) ) );
	connect( chooser, SIGNAL( needGraphOf( const Account& ) ), 
					 this, SLOT( graph( const Account& ) ) );
	connect( chooser, SIGNAL( needSubsetOf( const Account& ) ), 
					 this, SLOT( subset( const Account& ) ) );
	connect( chooser, SIGNAL( needRecWinOf( const Account& ) ), 
					 this, SLOT( reconcile( const Account& ) ) );
	
	connect( journaler, SIGNAL( changedJournal( const Journal& ) ), 
					 viewer, SLOT( setJournal( const Journal& ) ) );





	// build the status bar
	KStatusBar * status=statusBar();
	abalance=new QLabel( status );
	saveflag=new QLabel( status );
	saveflag->setAlignment( AlignCenter );
	tcount=new QLabel( status );
	msgs=new QLabel( status );
	
	abalance->setAlignment( AlignRight );

	status->addWidget( msgs, 4, true );
	status->addWidget( saveflag, 2, true );
	status->addWidget( tcount, 8, true );
	status->addWidget( abalance, 5, true );


	// build the menubar
	KPopupMenu * file=new KPopupMenu( this );
	KPopupMenu * help=helpMenu();
	KPopupMenu * prefm=new KPopupMenu( this );
	KPopupMenu * acct=new KPopupMenu( this );
	KPopupMenu * ledg=new KPopupMenu( this );
	KPopupMenu * graphs=new KPopupMenu( this );
	KPopupMenu * ntra=new KPopupMenu( this );
	
	KMenuBar * menu=menuBar();
	menu->insertItem( i18n( "&File" ), file );
	menu->insertItem( i18n( "&Account" ), acct );
	menu->insertItem( i18n( "&Journal" ), ledg );
	menu->insertItem( i18n( "&Transaction" ), ntra );
	menu->insertItem( i18n( "&Settings" ), prefm );
	menu->insertItem( i18n( "&Graphs" ), graphs );
	menu->insertSeparator();
	menu->insertItem( i18n( "&Help" ), help );

	KAction * plugins=new KAction( i18n( "About Plugins..." ), QPixmap(), 
																 CTRL+ALT+Key_P, this, SLOT( plugins() ), 
																 actionCollection(), "1" );
	plugins->plug( help );

	// these are the actions we'll perform

	KAction * save=KStdAction::save( this, SLOT( ssave() ), actionCollection() );
	KAction * home=KStdAction::home( this, SLOT( reload() ), 
																	 actionCollection() );
	KAction * ex=KStdAction::close( this, SLOT( saveAndQuit() ),
																	actionCollection() );
	KAction * quit=KStdAction::quit( this, SLOT( quit() ), 
																	 actionCollection() );

	KAction * exprt=new KAction( i18n( "&Export To..." ), QPixmap(), 
															 CTRL+SHIFT+Key_E, this, SLOT( exporter() ), 
															 actionCollection(), "4" );
	KAction * imprt=new KAction( i18n( "&Import From...." ), QPixmap(), 
															 CTRL+SHIFT+Key_I, this, SLOT( importer() ), 
															 actionCollection(), "5" );

	save->plug( file );
	home->plug( file );
	exprt->plug( file );
	imprt->plug( file );
	ex->plug( file );
	quit->plug( file );

	KAction * cp=KStdAction::preferences( this, SLOT( changePrefs() ), 
																				actionCollection() );
	KToggleAction * rb=new KToggleAction( i18n( "Show &Reconciled Balance" ),
																				QPixmap(), CTRL+Key_L, this, 
																				SLOT( toggleRecBal() ), 
																				actionCollection(), "6" );
	rb->setChecked( engine->getBP( "SHOWRECBAL" ) );

	KToggleAction * ac=new KToggleAction( i18n( "&Use Alternate Currency" ),
																				QPixmap(), CTRL+Key_U, this, 
																				SLOT( toggleCurrency() ), 
																				actionCollection(), "17" );
	ac->setChecked( engine->getBP( "USEALTCURRENCY" ) );
	//connect( ac, SIGNAL( activated() ), this, SLOT( toggleCurrency() ) );



	cp->plug( prefm );
	rb->plug( prefm );
	ac->plug( prefm );

	KAction * rc=new KAction( i18n( "&Reconcile" ), QPixmap( recacct ),
														CTRL+Key_R, this, SLOT( reconcile() ),
														actionCollection(), "7" );
	KAction * vs=KStdAction::find( this, SLOT( subset() ), 
																	 actionCollection() );
	KAction * cr=KStdAction::openNew( this, SLOT( newAcct() ), 
																		actionCollection() );
	KAction * dl=new KAction( i18n( "&Delete Account" ), QPixmap(),
														CTRL+SHIFT+Key_X, this, SLOT( removeOpened() ), 
														actionCollection(), "8" );
	KAction * ed=new KAction( i18n( "&Edit Account..." ), QPixmap( editacct ), 
														CTRL+Key_E, this, SLOT( editAcct() ),
														actionCollection(), "9" );

	rc->plug( acct );
	vs->plug( acct );
	cr->plug( acct );
	dl->plug( acct );
	ed->plug( acct );

	KAction * lc=new KAction( i18n( "&Create Journal..." ), QPixmap(), 
														CTRL+ALT+Key_N, this, SLOT( newLed() ),
														actionCollection(), "10" );
	KAction * ll=new KAction( i18n( "&Delete Journal..." ), QPixmap(),
														CTRL+ALT+Key_X, this, SLOT( remLed() ), 
														actionCollection(), "11" );
	KAction * ld=new KAction( i18n( "&Edit Journal..." ), QPixmap(),
														KKeySequence(), this, SLOT( editLed() ),
														actionCollection(), "12" );
	lc->plug( ledg );
	ll->plug( ledg );
	ld->plug( ledg );

	KAction * nc=new KAction( i18n( "&Open Memorized Editor..." ),
														QPixmap(), KKeySequence(), this,
														SLOT( openNTrans() ), actionCollection(), "15" );
	KAction * scd=new KAction( i18n( "&Open Scheduler..." ),
														 QPixmap(), KKeySequence(), this,
														 SLOT( openSched() ), actionCollection(), "16" );
	nc->plug( ntra );
	scd->plug( ntra );


	KAction * gr=new KAction( i18n( "&Graph Account..." ), QPixmap( grapher ),
														CTRL+Key_G, this, SLOT( graph() ),
														actionCollection(), "13" );
	KAction * rp=new KAction( i18n( "&Report on Account..." ),
														QPixmap( reporter ), CTRL+Key_Y, this,
														SLOT( report() ), actionCollection(), "14" );
 
	gr->plug( graphs );
	rp->plug( graphs );

	// create the toolbar
	save->plug( tools );
	//tools->addSeparator();
	cr->plug( tools );
	rc->plug( tools );
	ed->plug( tools );
	//tools->addSeparator();
	gr->plug( tools );
	rp->plug( tools );
	//tools->addSeparator();
	cp->plug( tools );
	vs->plug( tools );

	readPrefs( true );
}

QHaccWidget::~QHaccWidget(){ delete sounder; }

void QHaccWidget::exporter(){
	auto_ptr<HomeDlg> qdd=HomeDlg::expdlg( engine, this );
	qdd->resize( 300, 300 );

	if( qdd->exec() ){
		bool okay;
		const QString str=qdd->home( &okay );
		if( okay ){
			QHaccExt( engine ).exp( str );
			readPrefs( false ); // refresh everything!
		}
	}
}

void QHaccWidget::importer(){
	auto_ptr<HomeDlg> qdd=HomeDlg::impdlg( engine, this );
	qdd->resize( 300, 300 );

	if( qdd->exec() ){
		bool okay;
		const QString str=qdd->home( &okay );
		if( okay ){
			// if the import works, refresh everything!
			if( QHaccExt( engine ).imp( str ) )	readPrefs( false ); 
		}
	}
}


void QHaccWidget::readPrefs( bool initialLoad ){
	// now that everything's built, customize based on prefs
	//setCaption( engine->getSP( "TITLEBARNAME" ) );
	autosave=engine->getBP( "AUTOSAVE" );
	QApplication::setFont( engine->getWP( "FONT" ), true );

	readdockloc( "TOOLBARLOC", tools );
	readdockloc( "CHOOSERLOC", cdock );
	readdockloc( "JOURNALLOC", jdock );

	// set up a few defaults to convert from QHacc to KHacc
	engine->setCP( "ALTCOLOR", KGlobalSettings::alternateBackgroundColor() );
	engine->setCP( "MAINCOLOR", KGlobalSettings::baseColor() );

	if( engine->cnt( ACCOUNTS )==0 ){
		// if we don't have any accounts, make some default ones
		// here, we assume we'll also be making default accounttypes
		// table (that's where the ATYPEs of 0,1,3 come from)
		uint thisaid=0;

		// checking account
		Account ac=engine->getBlankA();
		ac.set( QC::AID, TableCol( ++thisaid ) );
		ac.set( QC::ATAXED, TableCol( true ) );
		ac.set( QC::ACATEGORY, TableCol( false ) );
		ac.set( QC::ATYPE, TableCol( ( uint )0 ) );
		ac.set( QC::ADEFAULTNUM, TableCol( i18n( GUIC::ACCTINCRSTR ) ) );
		ac.set( QC::ANAME, TableCol( i18n( "Checking" ) ) );
		engine->addA( ac );

		// savings account
		ac.set( QC::AID, TableCol( ++thisaid ) );
		ac.set( QC::ADEFAULTNUM, TableCol( "" ) );
		ac.set( QC::ANAME, TableCol( i18n( "Savings" ) ) );
		engine->addA( ac );

		// credit card account
		ac.set( QC::AID, TableCol( ++thisaid ) );
		ac.set( QC::ANAME, TableCol( i18n( "Credit Card" ) ) );
		ac.set( QC::ATYPE, TableCol( ( uint )1 ) );
		engine->addA( ac );

		// miscellanious categories
		ac.set( QC::ACATEGORY, TableCol( true ) );
		ac.set( QC::ATYPE, TableCol( ( uint )3 ) );
		const uint NCATS=23;
		QString nms[NCATS]={
			I18N_NOOP( "auto" ), I18N_NOOP( "bank fees" ),
			I18N_NOOP( "bank interest" ), I18N_NOOP( "books&subscriptions" ),
			I18N_NOOP( "petty cash" ), I18N_NOOP( "charity" ),
			I18N_NOOP( "computer" ), I18N_NOOP( "entertainment" ),
			I18N_NOOP( "restaurant" ), I18N_NOOP( "gifts" ),
			I18N_NOOP( "groceries" ), I18N_NOOP( "household" ),
			I18N_NOOP( "insurance" ), I18N_NOOP( "job" ), I18N_NOOP( "medical" ),
			I18N_NOOP( "miscellaneous" ), I18N_NOOP( "rent" ), I18N_NOOP( "taxes" ),
			I18N_NOOP( "telephone" ), I18N_NOOP( "travel" ),
			I18N_NOOP( "utilities" ), I18N_NOOP( "clothing" ), I18N_NOOP( "travel" )
		};
		
		for( uint i=0; i<NCATS; i++ ){
			ac.set( QC::AID, TableCol( ++thisaid ) );
			ac.set( QC::ANAME, TableCol( i18n( nms[i] ) ) );
			engine->addA( ac );
		};
	}

	// if we don't have a pre-saved location, start with a reasonable size
	if( initialLoad ){
		QString pref=engine->getSP( "LOCATION" );	

		if( pref.isEmpty() ) resize( 800, 600 );
		else{
			QString locs[2];
			Utils::parser( pref, " ", 0, locs, 2 );
			resize( locs[0].toInt(), locs[1].toInt() );
		}

		if( engine->getBP( "PLAYSOUNDS" ) ){
			connect( engine, SIGNAL( addedT() ),   sounder, SLOT( addT() ) );
			connect( engine, SIGNAL( removedT() ), sounder, SLOT( removeT() ) );
			connect( engine, SIGNAL( updatedT() ), sounder, SLOT( updateT() ) );
		}
	}

	if( engine->getBP( "HIDEJOURNALS" ) ) journaler->hide();
	else journaler->show();

	viewer->readPrefs( initialLoad );
	chooser->readPrefs( initialLoad );
	journaler->readPrefs( initialLoad );	
	setAccount( chooser->getSelected() );
}

void QHaccWidget::setAccount( const Account& acct ){
	account=acct;
	updateAccount( acct, acct );
}

void QHaccWidget::setJournal( const Journal& j ){
	journal=j;
	updateAccount( account, account );
}
void QHaccWidget::show(){ 
	journaler->blockSignals( true );
	QMainWindow::show();
	journaler->blockSignals( false );
}

void QHaccWidget::updateAccount( const Account&, const Account& newy ){
	KLocale locale( "khacc" );

	if( !account.isNull() && account[QC::AID]!=newy[QC::AID] ) return;
	if( journal.isNull() ) return;
	setCaption( engine->getSP( "TITLEBARNAME" )+" - "+journal[QC::LNAME].gets()+
							"/"+engine->getFNameOfA( account ) );

	int bal=0;
	uint count=0;
	QString type, name, temp;
	const MonCon& conv=engine->converter();

	if( !newy.isNull() ){
		bal=conv.converti( newy[QC::ACBAL].gets(), Engine, Engine );

		vector<TableSelect> v( 1, TableSelect( QC::SACCTID, newy[QC::AID] ) );
		engine->getWhere( SPLITS, TableGet( QC::SID ), v, count );

		type=GUIC::TYPENAMES[newy[QC::ATYPE].getu()];
		name=engine->getFNameOfA( newy );
  }
	else name=I18N_NOOP( "No Account" );
	
	QString s=I18N_NOOP( "Balance " );
	//s.append( locale.formatMoney( engine->convMoney( bal ).toFloat() ) );
	s.append( locale.formatMoney( conv.convert( bal ).toDouble() ) );

	if( engine->getBP( "SHOWRECBAL" ) ){
		int rbal=conv.converti( newy[QC::ARBAL].gets(), Engine, Engine );
		s.append( " / " );
		s.append( locale.formatMoney( conv.convert( rbal ).toDouble() ) );
	}
	abalance->setText( s );
	
	// update the number of transactions
	// every time you update the balance
	tcount->setText( name+": "+temp.setNum( count )+" "+
									 type+i18n( ( count==1 ? " Transaction" :
																" Transactions" ) ) );
}

void QHaccWidget::OBAccount( const Account& acct, int obBal ){
	QMessageBox::information( this, i18n( "Account Out of Budget" ),
														i18n( "%1 has exceeded its budget by %2" ).arg( engine->getFNameOfA( acct ) ).arg( KLocale( "khacc" ).formatMoney( engine->converter().convert( obBal ).toFloat() ) ) );
}

void QHaccWidget::aboutQt(){ QMessageBox::aboutQt( this, i18n( "About QT" ) );}
void QHaccWidget::about(){
	KAboutDialog * ad=new KAboutDialog( this, 0, false );
	ad->show();
}

void QHaccWidget::plugins(){
	QString output;
	const char * titles[]={ I18N_NOOP( "Database" ), 
													I18N_NOOP( "Import" ),
													I18N_NOOP( "Export" ) };
  const int NT=3;
 	int types[]={ QHacc::PIDATABASE, QHacc::PIIMPORTER, QHacc::PIEXPORTER };
	
	for( int p=0; p<NT; p++ ){
    int curr=0;
    vector<PluginInfo> pis=engine->getPluginInfo( types[p], &curr );
		
    output.append( "\n" ).append( i18n( titles[p] ) ).append( ":" );
    int i=0;
    for ( vector<PluginInfo>::iterator it=pis.begin(); it!=pis.end(); it++ ){
      output+="\n  ";
      if( i++==curr ) output+="*";
      else output+=" ";
      output+=QString( "%1 (%2): %3" ).arg( it->stub() ).arg( it->descr() ).arg( it->fname() );
    }
  }

  QMessageBox::information( this, i18n( "KHacc Plugins" ), output );
}

void QHaccWidget::changePref( const QString& s, bool b ){
	if( s=="SHOWRECBAL" || s=="USEALTCURRENCY" ){
		Account a=chooser->getSelected();
		updateAccount( a, a );
	}
	else if( s=="PLAYSOUNDS" ){
		if( b ){
			//cout<<"connecting"<<endl;
			connect( engine, SIGNAL( addedT() ),   sounder, SLOT( addT() ) );
			connect( engine, SIGNAL( removedT() ), sounder, SLOT( removeT() ) );
			connect( engine, SIGNAL( updatedT() ), sounder, SLOT( updateT() ) );
		}
		else{
			//cout<<"disconnecting"<<endl;
			disconnect( engine, SIGNAL( addedT() ),   sounder, SLOT( addT() ) );
			disconnect( engine, SIGNAL( removedT() ), sounder, SLOT( removeT() ) );
			disconnect( engine, SIGNAL( updatedT() ), sounder, SLOT( updateT() ) );
		}
	}
	else if( s=="AUTOSAVE" ){
		autosave=b;
		if( autosave ) save();
	}
	else if( s=="HIDEJOURNALS" ){
		if( b ) journaler->hide();
		else journaler->show();
	}
}

void QHaccWidget::changePref( const QString&, QFont f ){
	setFont( f );
	QApplication::setFont( f, true );
}

void QHaccWidget::changePref( const QString& s, QString v ){
	if( s=="TITLEBARNAME" ) setCaption( v );
	else if( s=="CSYMBOL" || s=="ALTCSYMBOL" || s=="CURRENCYSEPARATOR" ){
		Account a=chooser->getSelected();
		updateAccount( a, a );
	}
}

void QHaccWidget::changePrefs(){
	( new PrefsDlg( engine, 0, this ) )->show();
}

void QHaccWidget::quit(){
	bool closeme=true;

	if( saveflag->text()==i18n( "save" ) ){
		int ans=KMessageBox::warningYesNoCancel( this, i18n( "Save then Quit?" ), i18n( "Save changes?" ), KStdGuiItem::yes(), KStdGuiItem::no(), "quitnosave" );
		if( ans==KMessageBox::Cancel ) return;
		else if( ans==KMessageBox::Yes ) closeme=save();
	}

	if( closeme ) qApp->quit();
}

void QHaccWidget::saveAndQuit(){ if( save() ) quit(); }

void QHaccWidget::saveProperties( KConfig * config ){
	config->writePathEntry( "home", engine->getHome() );
}

void QHaccWidget::readProperties( KConfig * config ){
	QString home=config->readPathEntry( "home" );
	if ( !home.isNull() ) engine->setHome( home );
}

void QHaccWidget::ssave(){ save(); }
bool QHaccWidget::save(){
	QString temp;

	QString val=temp.setNum( width() )+" "+temp.setNum( height() );
	engine->setSP( "LOCATION", val );

	savedockloc( "TOOLBARLOC", tools );
	savedockloc( "CHOOSERLOC", cdock );
	savedockloc( "JOURNALLOC", jdock );

	viewer->save();
	chooser->save();
	journaler->save();

	if( !engine->save( temp ) ){
		msgs->setText( temp );
		KMessageBox::error( this, temp, i18n( "Error During Save!" ) );
		return false;
	}
	else msgs->setText( "" );
	return true;
}

void QHaccWidget::savedockloc( const QString& pref, QDockWindow * dw ){
	Dock tbd;
	int tbl;
	bool nl;
	int extra;
	getLocation( dw, tbd, tbl, nl, extra );	
	
	QString val, temp;
	val=temp.setNum( ( int )tbd )+" "+temp.setNum( tbl )+" ";
	val+=( nl ? "Y " : "N " )+temp.setNum( extra );
	engine->setSP( pref, val );
}

void QHaccWidget::readdockloc( const QString& pref, QDockWindow * dw ){
	QString val=engine->getSP( pref );
	QString strs[4];
	Utils::parser( val, " ", 0, strs, 4 );
	Dock tbd=( Dock )( strs[0].toInt() );
	int tbl=strs[1].toInt();
	bool nl=( strs[2]=="Y" ? true : false );
	int extra=strs[3].toInt();
	moveDockWindow( dw, tbd, nl, tbl, extra );
}

void QHaccWidget::setSaveFlag( bool b ){
	if( autosave ){
		disconnect( engine, SIGNAL( needSave( bool ) ),
								this, SLOT( setSaveFlag( bool ) ) );
		if( b ){
			save();
			b=false;
		}
		connect( engine, SIGNAL( needSave( bool ) ), SLOT( setSaveFlag( bool ) ) );
	}

	saveflag->setText( b ? i18n( "save" ) : QString( "" ) );
}

void QHaccWidget::reload(){	
	auto_ptr<HomeDlg> hd=HomeDlg::homedlg( engine, this );
	if( hd->exec()==QDialog::Accepted ){
		QString home( hd->home() );
		std::ostream * str=0;
		if( Utils::debug( Utils::ERRDEFAULT, str ) )
			*str<<i18n( "Changing KHACC_HOME to %1" ).arg( home )<<endl;
		
    engine->setHome( home );
    readPrefs( false );
  }
}

void QHaccWidget::newAcct(){
	Account acct;
  ( new AccountDlg( engine, acct, this ) )->show();
}

void QHaccWidget::editAcct(){
	Account a=chooser->getSelected();
	if( a.isNull() ) newAcct();
	else ( new AccountDlg( engine, a, this ) )->show();
}

void QHaccWidget::toggleRecBal(){ 
	engine->setBP( "SHOWRECBAL", !engine->getBP( "SHOWRECBAL" ) );
}
void QHaccWidget::toggleCurrency(){ 
	engine->setBP( "USEALTCURRENCY", !engine->getBP( "USEALTCURRENCY" ) );	
}

void QHaccWidget::newLed(){
	( new JournalDlg( engine, Journal(), this ) )->show();
}
void QHaccWidget::remLed(){
	JournalDlg::qremove( engine, journaler->getSelected(), this );
}
void QHaccWidget::editLed(){
	( new JournalDlg( engine, journaler->getSelected(), this ) )->show();
}

void QHaccWidget::openNTrans(){ 
	( new NamedDlg( engine,	journaler->getSelected().getu( QC::LID ), 
									"", this ) )->show(); 
}
void QHaccWidget::openSched(){ ( new SchedDlg( engine, "", this ) )->show(); }

void QHaccWidget::subset(){ subset( chooser->getSelected() ); }
void QHaccWidget::subset( const Account& a ){
	if( !a.isNull() ){
		QHaccSubsetWin  * win=new QHaccSubsetWin( engine );
		connectView( win->getView() );
		win->show();
		win->setAL( a, journaler->getSelected() );
		
		vector<TableSelect> ts;

		const QDate now=QDate::currentDate();
		int gt=engine->getIP( "GRAPHTIME" );

		ts.push_back( TableSelect( QC::XTDATE, TableCol( now.addMonths( 0-gt ) ),
															 TableSelect::GE ) );
		ts.push_back( TableSelect( QC::XTDATE, TableCol( now ),
															 TableSelect::LE ) );
		win->setSelectors( ts );
	}	
}

void QHaccWidget::graph( const Account& a ){ igrd( a, QHaccGRDialog::GRAPH ); }
void QHaccWidget::graph(){
	igrd( chooser->getSelected(), QHaccGRDialog::GRAPH );
}

void QHaccWidget::report( const Account& a ){igrd( a, QHaccGRDialog::REPORT );}
void QHaccWidget::report(){
	igrd( chooser->getSelected(), QHaccGRDialog::REPORT );
}

void QHaccWidget::reconcile(){ reconcile( chooser->getSelected() ); }
void QHaccWidget::reconcile( const Account& a ){
	if( !a.isNull() ){
		QHaccRecWin  * win=new QHaccRecWin( engine, journaler->getSelected() );
		connectView( win->getView() );

		win->show();
		win->setAccount( a );
	}
}

void QHaccWidget::igrd( const Account& a, int t ){
	if( !a.isNull() ){
		QHaccGRDialog * grd=new QHaccGRDialog( engine,
																					 ( QHaccGRDialog::DisplayType )t,
																					 a, journaler->getSelected(), this );
		grd->readPrefs( true );
		grd->show();
	}
}

void QHaccWidget::connectView( QHaccView * v ){
	// connect this view's copy and cut operations to the global
	// signal for copy and cut transactions AND connect this view
	// to that global signal as well
	connect( v, SIGNAL( transToClipboard( bool ) ),
					 this, SLOT( transOnClipboard( bool ) ) );
	connect( this, SIGNAL( transToClipboard( bool ) ), 
					 v, SLOT( transCopied( bool ) ) );

	Transaction t;
	QHaccTable s;
	uint a=0;
	v->transCopied( QHaccTDrag::decode( QApplication::clipboard()->data(), 
																			t, s, a ) );
}

void QHaccWidget::removeOpened(){
	AccountDlg::qremove( engine, chooser->getSelected() );
}

void QHaccWidget::transOnClipboard( bool b ){ emit transToClipboard( b ); }


