/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2004 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "qhaccacctchsr.h"
#include "qhacc.h"
#include "copystore.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "qhaccdialogs.h"
#include "qhaccacctdlg.h"
#include "qhaccconstants.h"
#include "qhaccacctchsr.moc"

#include <qpainter.h>
#include <qpopupmenu.h>

#include <klocale.h>

QColor QHaccAViewItem::maincolor;
QColor QHaccAViewItem::altcolor;
bool QHaccAViewItem::showanums;
bool QHaccAViewItem::useanums;

QHaccAViewItem::QHaccAViewItem( QHaccListView * p ) : QHaccListViewItem( p ){
	iinit();
}
QHaccAViewItem::QHaccAViewItem( QHaccListViewItem * p ) 
	: QHaccListViewItem( p ){
	iinit();
}

void QHaccAViewItem::iinit(){
	idcol=QC::AID;
	setRenameEnabled( 0, true );
	setRenameEnabled( 1, false );
	setDragEnabled( true );
	setDropEnabled( true );
	curr=false;
}

QHaccAViewItem::~QHaccAViewItem(){}

void QHaccAViewItem::setRow( const TableRow& t ){
	QHaccListViewItem::setRow( t );
	if( !t.isNull() ){
		QString str=t[QC::ANAME].gets();
		if( useanums ) setText( 0, t[QC::ANUM].gets()+"-"+str );
		else setText( 0, str );
	}
}

void QHaccAViewItem::setCurr( bool b ){ curr=b; }
int QHaccAViewItem::compare( QListViewItem * i, int col, bool asc ) const {
	const TableRow& irow=( ( QHaccAViewItem * )i )->row();

	// always make null rows sort to the bottom
	if( myrow.isNull() ){
		if( asc ) return 1;
		else return -1;
	}		
	else{
		if( irow.isNull() ){
			if( asc ) return -1;
			else return 1;
		}
	}
	
	if( col==0 ) return irow[QC::ANAME].compareTo( myrow[QC::ANAME], CTSTRING );
	else return irow[QC::ACBAL].compareTo( myrow[QC::ACBAL], CTFLOAT );
}

bool QHaccAViewItem::acceptDrop( QMimeSource * e ){
	return( QHaccTDrag::canDecode( e ) || QHaccADrag::canDecode( e ) );
}

void QHaccAViewItem::dropper( QDropEvent * e ){
	//cout<<"avi item dropped"<<endl;
	TableRow t;
	QHaccTable s( 0 );
	uint a=0;
	Account acct=row();
	if( QHaccTDrag::decode( e, t, s, a ) ){
		e->accept();
		s.updateWhere( TableSelect( QC::SACCTID, TableCol( a ) ),
									 TableUpdate( QC::SACCTID, acct[QC::AID] ) );
		engine->updateT( t, s );
	}
	else if( QHaccADrag::decode( e, t ) ){
		e->accept();
		// check to make sure we're not trying to reparent one of our
		// parent account's on to this account
		TableCol taid( t[QC::AID] );
		TableCol itempid=acct[QC::APID];

		bool good=( taid!=acct[QC::AID] );
		while( itempid.getu()>0 && good ){
			if( taid==itempid )	good=false;
			itempid=engine->getA( itempid.getu() )[QC::APID];
		}
		
		if( good ){ // the dropped account isn't a parent of this item or itself
			TableRow newy( t );
			newy.set( QC::APID, acct[QC::AID] );
			engine->updateA( t, newy );
		}	
	}
}

void QHaccAViewItem::paintCell( QPainter * p, const QColorGroup& cg, int c, 
																int w, int aln ){
	QColorGroup _cg( cg );
	_cg.setColor( QColorGroup::Highlight, ( curr ? maincolor : altcolor ) );
	_cg.setColor( QColorGroup::HighlightedText, black );
	p->save();
	if( this==listView()->currentItem() ){
		QFont f=p->font();
		f.setBold( true );
		p->setFont( f );
	}
	if( myrow.isNull() ) QHaccListViewItem::paintCell( p, _cg, c, w, aln );
	else{
		const MonCon& mconv=engine->converter();

		QString txt;
		if( c==0 ){ // account names
			txt=myrow[QC::ANAME].gets();
			if( useanums && !myrow[QC::ANUM].gets().isEmpty() ) txt=myrow[QC::ANUM].gets()+"-"+txt;
			else if( showanums && !myrow[QC::ANUM].gets().isEmpty() )
				txt+=" ("+myrow[QC::ANUM].gets()+")";
		}
		else txt=mconv.convert( myrow[QC::ACBAL].gets() );

		int marg=listView()->itemMargin();
		int h=height();
		p->fillRect( 0, 0, w, h, curr ? maincolor : altcolor );
		p->drawText( marg, 0, w-1, h-1-marg, aln, txt );
	}
	p->restore();
}

void QHaccAViewItem::paintBranches( QPainter * p, const QColorGroup& cg,
																		int w, int y, int h ){
	QColorGroup mycg( cg );
	mycg.setColor( QColorGroup::Base, ( isSelected() ? maincolor : altcolor ) );
	QHaccListViewItem::paintBranches( p, mycg, w, y, h );
}

void QHaccAViewItem::resetE( QHacc * e ){
	QHaccListViewItem::resetE( e );
	maincolor=engine->getCP( "MAINCOLOR" );
	altcolor=engine->getCP( "ALTCOLOR" );
	showanums=engine->getBP( "CHOOSERSHOWANUMS" );
	useanums=engine->getBP( "USEANUMSFORNAMES" );
}











QHaccAccountChooser::QHaccAccountChooser( QHacc * eng, QWidget * p,
																					const char * c )
	: QHaccListView( eng, p, c ){

	lasti=0;
	setPrefBase( "CHSR" );
	
	addColumn( i18n( "Account" ) );
	addColumn( i18n( "Balance" ) );
	setColumnAlignment( 1, AlignRight );
  
	connect( engine, SIGNAL( addedA( const Account& ) ), SLOT( makeData() ) );
	connect( engine, SIGNAL( removedA( const Account& ) ), SLOT( makeData() ) );
	connect( engine, SIGNAL( updatedA( const Account&, const Account& ) ), 
					 SLOT( updA( const Account&, const Account& ) ) );

	connect( engine, SIGNAL( changedP( const QString&, bool ) ),
					 SLOT( changeP( const QString&, bool ) ) );
	connect( engine, SIGNAL( changedP( const QString&, float ) ),
					 SLOT( changeP( const QString&, float ) ) );
	connect( engine, SIGNAL( changedP( const QString&, QString ) ),
					 SLOT( changeP( const QString&, QString ) ) );


	connect( this, SIGNAL( itemRenamed( QListViewItem *, int, const QString& ) ),
					 this, SLOT( renamer( QListViewItem *, int, const QString& ) ) );
	connect( this, SIGNAL( clicked( QListViewItem * ) ),
					 this, SLOT( selchange( QListViewItem * ) ) );
	connect( this, SIGNAL( spacePressed( QListViewItem * ) ),
					 this, SLOT( selchange( QListViewItem * ) ) );

#if QT_VERSION>=320
	connect( this, SIGNAL( doubleClicked( QListViewItem *, const QPoint&,
																				int ) ),
					 this, SLOT( editAcct( QListViewItem *, const QPoint&, int ) ) );
#endif

  QPopupMenu * popup=new QPopupMenu( this );
  popup->insertItem( i18n( "reconcile" ), this, SLOT( recoAcct() ) );
  popup->insertItem( i18n( "graph" ), this, SLOT( chart() ) );
  popup->insertItem( i18n( "report" ), this, SLOT( report() ) );

  popup->insertItem( i18n( "edit" ), this, SLOT( editAcct() ) );
  popup->insertItem( i18n( "new" ), this, SLOT( newAcct() ) );
  popup->insertItem( i18n( "delete" ), this, SLOT( remAcct() ) );  
	
  popup->insertSeparator();
  popup->insertItem( i18n( "toggle &view" ), this,
										 SLOT( toggleView() ), CTRL+Key_V );
  popup->insertItem( i18n( "view su&bset" ), this,
										 SLOT( viewSubset() ) );

	setPopup( popup );

	setAllColumnsShowFocus( true );
	setShowSortIndicator( true );
	setRootIsDecorated( true );
	setSelectionMode( Single );
	setAcceptDrops( true );
	viewport()->setAcceptDrops( true );
}

QHaccAccountChooser::~QHaccAccountChooser(){}

void QHaccAccountChooser::renamer( QListViewItem * i, int, const QString& s ){
	Account a=getRow( i );
	a.set( QC::ANAME, s );
	engine->updateA( a, a );
}

void QHaccAccountChooser::isave() const {
	TableRow sel=getSelected();
	if( !sel.isNull() )	engine->setIP( "SHOWINDEX", sel[QC::AID].geti() );
}

void QHaccAccountChooser::ireadPrefs( bool initial ){
	QHaccAViewItem::resetE( engine );
	color=engine->getCP( "ALTCOLOR" );

	if( initial ){
		connect( engine, SIGNAL( changedP( const QString&, QColor ) ),
						 SLOT( changeP( const QString&, QColor ) ) );
	}

	makeData();

	uint aid=engine->getSP( "SHOWINDEX" ).toUInt();
	QHaccListViewItem * avi=findRowItem( aid );
	if( avi )	selchange( avi );
}

void QHaccAccountChooser::keyPressEvent( QKeyEvent * qke ){
	if( qke->key()==Key_Delete ){
		qke->accept();
		remAcct();
	}
	else QHaccListView::keyPressEvent( qke );
}

void QHaccAccountChooser::startDrag(){
	const Account& a=getMoused();

	if( !a.isNull() ){
		QHaccADrag * dragger=new QHaccADrag( a, this );
		dragger->drag();
	}
}

void QHaccAccountChooser::contentsDragEnterEvent( QDragEnterEvent * e ){
	//cout<<"chsr dragenter"<<endl;
	if( QHaccADrag::canDecode( e ) || QHaccTDrag::canDecode( e ) ) e->accept();
}

void QHaccAccountChooser::contentsDropEvent( QDropEvent * e ){
	//cout<<"chsr dropevent"<<endl;
	QHaccAViewItem * avi=( QHaccAViewItem * )itemAt( contentsToViewport( e->pos() ) );
	if( avi ) avi->dropper( e );
	else{
		Account t;
		if( QHaccADrag::decode( e, t ) ){
			e->accept();
			Account newy( t );
			newy.set( QC::APID, 0 );
			engine->updateA( t, newy );
		}	
	}
}

void QHaccAccountChooser::paintEmptyArea( QPainter * p, const QRect& rect ){
	p->fillRect( rect, color );
}

void QHaccAccountChooser::grabData(){
	std::ostream * str=0;
	if( Utils::debug( Utils::CURIOSITY, str ) ) 
		*str<<"acctchsr grabbing data!"<<endl;
	// we really only care about the account's name and it's balance...
	// just make a small table to hold the important info

	setUpdatesEnabled( false );
	clear();
	auto_ptr<QHaccResultSet> table1=engine->getAs( TableGet() );
	uint r1=table1->rows();

	// add accounts ordered as PIDs so that when we come to a child account,
	// we know the parent has already been added
	QHaccTableIndex pidx( table1.get(), QC::APID, CTUINT );

	for( uint i=0; i<r1; i++ ){
		const TableRow& row=table1->at( pidx[i] );
		uint apid=row[QC::APID].getu();
		QHaccAViewItem * avi=0;

		if( apid ){
			QHaccListViewItem * par=findRowItem( row[QC::APID].getu() );
			if( par )	avi=new QHaccAViewItem( par );
		}
		if( avi==0 ) avi=new QHaccAViewItem( this );
		
		avi->setRow( row );
	}	
	setUpdatesEnabled( true );
	triggerUpdate();
}

void QHaccAccountChooser::editAcct( QListViewItem * i, const QPoint&, int ){
	( new AccountDlg( engine, getRow( i ), this ) )->show();
}

void QHaccAccountChooser::editAcct(){
	const Account& acct=getMoused();
  if ( acct.isNull() ) newAcct();
  else ( new AccountDlg( engine, acct, this ) )->show();
}

void QHaccAccountChooser::newAcct(){
	Account acct;
	( new AccountDlg( engine, acct, this ) )->show();
}

void QHaccAccountChooser::remAcct(){
	AccountDlg::qremove( engine, getMoused(), this );
}

void QHaccAccountChooser::toggleView(){
	engine->setBP( "HIDECATEGORIES", !engine->getBP( "HIDECATEGORIES" ) );
}
void QHaccAccountChooser::recoAcct(){ emit needRecWinOf( getMoused() ); }
void QHaccAccountChooser::viewSubset(){	emit needSubsetOf( getMoused() ); }
void QHaccAccountChooser::chart(){ emit needGraphOf( getMoused() ); }
void QHaccAccountChooser::report(){ emit needReportOf( getMoused() ); }

void QHaccAccountChooser::selchange( QListViewItem * i ){
	if( !i ) return;
	if( lasti ){
		lasti->setCurr( false );
		repaintItem( lasti );
	}

	lasti=( QHaccAViewItem * )i;
	lasti->setCurr( true );
	repaintItem( lasti );
	setSelected( i, true );
	save();
	emit changedAccount( getRow( i ) );
}

void QHaccAccountChooser::updA( const Account& oldy, const Account& newy ){

	// first of all, find the item displaying the account, and update it
	QHaccListViewItem * avi=findRowItem( oldy[QC::AID].getu() );
	if( avi ){
#if QT_VERSION>=320
		const int sortcol=sortColumn();
#endif

		const QString str=avi->text( sortcol );
		avi->setRow( newy );
		if( avi->text( sortcol )!=str ) sort();
	}
	
	// if the parent account changed, we need to reorder the list
	TableCol apid( newy[QC::APID] );

	//cout<<oldy.toString()<<endl<<newy.toString()<<endl;
	if( oldy[QC::APID]!=apid ){
		//cout<<"need to reparent!"<<endl;

		TableCol aid( newy[QC::AID] );
		QHaccListViewItem * newparent=findRowItem( apid.getu() );
		QHaccListViewItem * accti=findRowItem( aid.getu() );

		QListViewItem * apar=( ( QListViewItem * )accti )->parent();
		if( apar ) apar->takeItem( accti );
		else takeItem( accti );
		
		if( newparent )	newparent->insertItem( accti );
		else insertItem( accti );
	}
}

void QHaccAccountChooser::changeP( const QString& p, QColor c ){
	if( p=="ALTCOLOR" ) color=c;
	QHaccAViewItem::resetE( engine );
	triggerUpdate();
}

void QHaccAccountChooser::changeP( const QString& p, bool ){
	bool doupdate=false;
	if( p=="USEALTCURRENCY" ) doupdate=true;
	else if( p=="CHOOSERSHOWANUMS" || "USEANUMSFORNAMES"==p ){
		QHaccAViewItem::resetE( engine );
		doupdate=true;
	}

	if( doupdate ) triggerUpdate();
}

void QHaccAccountChooser::changeP( const QString& p, float ){
	if( p=="ALTCURRENCYFACTOR" ) triggerUpdate();
}

void QHaccAccountChooser::changeP( const QString& p, QString ){
	if( p=="ALTCURRENCYSYMBOL" ) triggerUpdate();
}

void QHaccAccountChooser::setAccount( const Account& a ){
	QHaccListViewItem * ri=findRowItem( a[QC::AID].getu() );
	if( ri ) selchange( ri );
}

