/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2004 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "qhaccacctdlg.h"
#include "qhacc.h"
#include "config.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "splitdialog.h"
#include "guiconstants.h"
#include "qhaccconstants.h"
#include "qhacclineedits.h"
#include "qhaccacctdlg.moc"

#include <qlabel.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qsplitter.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qfontdialog.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qcolordialog.h> 
#include <qmultilineedit.h>

#include <klocale.h>

void AccountDlg::qremove( QHacc * eng, const Account& a, QWidget * p ){
	if( !a.isNull() ){
		if ( QMessageBox::warning( p, i18n( "Delete " )+a.gets( QC::ANAME ),
															 i18n( "Delete %1, its Subaccounts, and Transactions.\nAre you sure?" ).arg( a.gets( QC::ANAME ) ), i18n( "Delete Account" ),
															 i18n( "Reconsider" ) )==0 ) eng->removeA( a );
	}
}

AccountDlg::AccountDlg( QHacc * eng, const Account& acct, QWidget * p,
												const char * n, bool modal ) 
	: QTabDialog( p, n, modal ){

	engine=eng;

	bool forDefault=false;
	if( acct.isNull() ){
		edit=false;
		account=engine->getBlankA();
		setCaption( i18n( "Create New Account" ) );
	}
	else{
		edit=true;
		account=acct;
		if( account[QC::AID]==0 ){
			setCaption( i18n( "Set Account Defaults" ) );
			forDefault=true;
		}
		else setCaption( i18n( "Edit" )+" "+account.gets( QC::ANAME ) );
	}

	tab1=new AccountTab1( eng, account, forDefault, this );
	tab2=new AccountTab2( eng, account, this );
	tab3=new AccountTab3( eng, account, this );
	
  addTab( tab1, i18n( "Account" ) );
  addTab( tab2, i18n( "Institution" ) );
  addTab( tab3, i18n( "Loan Information" ) );
	QButtonGroup * bg=tab1->getTypeGroup();
  tab3->setEnabled( bg->id( bg->selected() )==( int )QC::LIABILITY );

  setCancelButton( i18n( "Cancel" ) );

	connect( bg, SIGNAL( clicked( int ) ), this, SLOT( typer( int ) ) );
}

AccountDlg::~AccountDlg(){}
Account AccountDlg::getAccount() const { return account; }
void AccountDlg::close(){ QTabDialog::close( true ); }
void AccountDlg::accept(){
	Account newy( account );
	tab1->modify( newy );
	tab2->modify( newy );
	tab3->modify( newy );

	if( edit ) engine->updateA( account, newy ); 
	else newy.set( QC::AID, engine->addA( newy ) );

	account=newy;
	QTabDialog::accept();
}

void AccountDlg::typer( int id ){	
	tab3->setEnabled( id==( int )QC::LIABILITY ); 
}

/* * * * * * * * * * * * * * * * */
/* * * * TAB 1 INFO* * * * * * * */
/* * * * * * * * * * * * * * * * */

AccountTab1::AccountTab1 ( QHacc * eng, const Account& a, bool forDefault, 
													 QWidget * w, const char * n ) : QWidget( w, n ){
			
	engine=eng;
	const MonCon& conv=engine->converter();
	QGridLayout * layout=new QGridLayout( this, 12, 5, 5 );
	name=new QLineEdit( this );
	number=new QLineEdit( this );
	bal=new QLineEdit( this );
	taxed=new QCheckBox( this );
	category=new QCheckBox( this );
  typeBox=new QButtonGroup();	
	budget=new QLineEdit( this );

	for( uint i=0; i<QC::NUMTYPES; i++ ){
		QRadioButton * qrb=new QRadioButton( i18n( GUIC::TYPENAMES[i] ), this );
		typeBox->insert( qrb, i );
	}

	bool nocase=engine->getBP( "AUTOCOMPNOCASE" );
	parent=new QHaccChoiceEdit( false, this );
	parent->ignoreCase( nocase );
	transtypes=new QLineEdit( this );
	dttype=new QHaccChoiceEdit( false, this );
	dttype->ignoreCase( nocase );


	QString stypes;
	QStringList ttypes, parents;
	
	ttypes=QStringList::split( " ", a.gets( QC::ATRANSNUMS ), false );
	QStringList::iterator it;
	bool notfirst=false;
	for( it=ttypes.begin(); it!=ttypes.end(); ++it ){
		if( notfirst ) stypes.append( " " );
		else notfirst=true;

		stypes.append( *it );
	}

	// go through strings and remove this account's name, and
	// names of any children of this account--we don't want
	// circular parentage    
	QString aname=engine->getFNameOfA( a );
	int length=aname.length();
	if( length==0 ) length=-1;


	// get a list of possible parent accounts to add to parent choice box
	auto_ptr<QHaccResultSet> accts=engine->getAs( TableGet() );
	uint naccts=accts->rows();
	for( uint i=0; i<naccts; i++ ){
		Account acct=accts->at( i );
		QString str=engine->getFNameOfA( acct );
		if( str.left( length )!=aname ) parents.append( str );
	}

	/* set all the fields */
	name->setText( a.gets( QC::ANAME ) );
	number->setText( a.gets( QC::ANUM ) );

	bal->setText( conv.convert( a.gets( QC::AOBAL ) ) );
	transtypes->setText( stypes );
	budget->setText( conv.convert( a.gets( QC::ABUDGET ) ) );


	parent->setEnabled( !forDefault );
	parent->insertStringList( parents );
	parent->setEmptyText( i18n( "<top level>" ) );


  dttype->insertItem( i18n( GUIC::ACCTINCRSTR ) );
	dttype->insertStringList( ttypes );
	dttype->setEmptyText( i18n( "<none>" ) );

	taxed->setChecked( a.getb( QC::ATAXED ) );
	category->setChecked( a.getb( QC::ACATEGORY ) );
	typeBox->setButton( a.getu( QC::ATYPE ) );

	QString ttext=a.gets( QC::ADEFAULTNUM );
	dttype->setText( ttext );

	uint pid=a.getu( QC::APID );
	if ( pid ) parent->setText( engine->getFNameOfA( engine->getA( pid ) ) );

	// add everything to the layout
  int row=0;
	const char * names[]={ I18N_NOOP( "Account Name" ),
												 I18N_NOOP( "Account Number" ),
												 I18N_NOOP( "Opening Balance" ),
												 I18N_NOOP( "Account is Taxable" ),
												 I18N_NOOP( "Account is a Category" ),
												 I18N_NOOP( "Account Type" ),
												 I18N_NOOP( "Child Of" ),
												 I18N_NOOP( "Transaction Types" ),
												 I18N_NOOP( "Default Transaction Type" ),
												 I18N_NOOP( "Monthly Budget Limit" ) };
	
	QWidget * ws[]={ name, number, bal, taxed, category, typeBox, parent,
									 transtypes, dttype, budget };
	QWidget * last=0;
  for ( int i=0; i<10; i++ ){
    layout->addMultiCellWidget( new QLabel( i18n( names[i] ), this ),
																row, row, 0, 3 );
		
		// add all the different account types
    if ( i==5 ){
      for ( uint j=0; j<QC::NUMTYPES; j++ ){
				layout->addMultiCellWidget( typeBox->find( j ), row, row, 4, 9 );
				row++;
      }
    }
    else layout->addMultiCellWidget( ws[i], row, row, 4, 9 );
    row++;

		if( last ) setTabOrder( last, ws[i] );
		last=ws[i];
  }

	connect( transtypes, SIGNAL( textChanged( const QString& ) ),
					 this, SLOT( makeTTChoices() ) );
}

AccountTab1::~AccountTab1(){ delete typeBox; }

void AccountTab1::makeTTChoices(){
	QString types=transtypes->text().simplifyWhiteSpace();
	QStringList listo=QStringList::split( " ", types, false );
	QString type=dttype->text();

	dttype->clear();
	dttype->insertItem( GUIC::ACCTINCRSTR );
	dttype->insertStringList( listo );
	dttype->setText( type );
}


void AccountTab1::modify( Account& acct ) const {
	const MonCon& conv=engine->converter();
	acct.set( QC::AOBAL, TableCol( conv.convert( bal->text() ) ) );
	acct.set( QC::ANAME, TableCol( name->text() ) );
	acct.set( QC::ANUM,  TableCol( number->text() ) );
	acct.set( QC::ATRANSNUMS,  TableCol( transtypes->text() ) );
	acct.set( QC::ADEFAULTNUM, TableCol( dttype->text() ) );
	acct.set( QC::ABUDGET, TableCol( conv.convert( budget->text() ) ) );
	acct.set( QC::ATYPE, TableCol( typeBox->id( typeBox->selected() ) ) );

	Account pacc=engine->getA( parent->text() );
	if( pacc.isNull() ) acct.set( QC::APID, TableCol( 0 ) );
	else acct.set( QC::APID, TableCol( pacc.getu( QC::AID ) ) );

	acct.set( QC::ATAXED, TableCol( taxed->isChecked() ) );
	acct.set( QC::ACATEGORY, TableCol( category->isChecked() ) );
}

QButtonGroup * AccountTab1::getTypeGroup() const { return typeBox; }

/* * * * * * * * * * * * * * * * */
/* * * * TAB 2 INFO* * * * * * * */
/* * * * * * * * * * * * * * * * */

AccountTab2::AccountTab2 ( QHacc *, const Account& a,
													 QWidget * p, const char * n ) : QWidget( p, n ){
	
  QGridLayout * layout=new QGridLayout( this, 14, 9, 5 );

	// FIXME: is there a better way than this?
	int limit=11, offset=9;
	instinfo=new QFrame * [limit];

	for( int i=0; i<limit; i++ ){
		int field=i+offset;
		QString data=a.gets( field );

		if( field==QC::AINSTNOTES ){
			data=data.replace( GUIC::LINESTOP, "\n" );
			instinfo[i]=new QMultiLineEdit( this );
			( ( QMultiLineEdit * )instinfo[i] )->setText( data );
			( ( QMultiLineEdit * )instinfo[i] )->setWordWrap( QMultiLineEdit::WidgetWidth );
		}
		else instinfo[i]=new QLineEdit( data, this );
	}
	
  int row=0;
	const char * names[]={ I18N_NOOP( "Name" ), I18N_NOOP( "Address" ), "",
												 I18N_NOOP( "City" ), I18N_NOOP( "State" ),
												 I18N_NOOP( "Zip" ), I18N_NOOP( "Contact" ),
												 I18N_NOOP( "Phone" ), I18N_NOOP( "Fax" ),
												 I18N_NOOP( "Email" ), I18N_NOOP( "Notes" ) };
	for( int i=0; i<limit; i++ ){
		layout->addMultiCellWidget( new QLabel( i18n( names[i] ), this ),
																row, row, 0, 3 );

		if( ( i+offset )==( int )QC::AINSTSTATE ){
			// put the zipcode on the same line as the state
			layout->addMultiCellWidget( instinfo[i], row, row, 4, 5 ); // state field

			// zip label
      layout->addMultiCellWidget( new QLabel( i18n( names[++i] ), this ), 
																	row, row, 7, 8 );
      layout->addMultiCellWidget( instinfo[i], row, row, 9, 13 ); // zip field
		}

		else layout->addMultiCellWidget( instinfo[i], row, row, 4, 13 );

		row++;
	}
}

AccountTab2::~AccountTab2(){ delete [] instinfo; }
void AccountTab2::modify( Account& acct ) const {
	int offset=9, limit=11;

	for( int i=0; i<limit; i++ ) {
		int field=i+offset;
		if( field==( int )QC::AINSTNOTES ){
			QString txt=( ( QMultiLineEdit * )instinfo[i] )->text();
			txt=txt.replace( "\n", GUIC::LINESTOP );
			acct.set( field, TableCol( txt ) );								 
		}
		else acct.set( field, TableCol( ( ( QLineEdit * )instinfo[i] )->text() ) );
	}
}

/* * * * * * * * * * * * * * * * */
/* * * * TAB 3 INFO* * * * * * * */
/* * * * * * * * * * * * * * * * */

AccountTab3::AccountTab3 ( QHacc * eng, const Account& a, QWidget * w,
													 const char * n ) : QWidget( w, n ){
			
	engine=eng;
	QGridLayout * lay=new QGridLayout( this, 12, 5, 5 );

	// default values: assume a 1-year loan at 10%, $1 payments
	float ai=10;
	int np=12;
	QString pay="1.00";
	
	bool isaloan=Utils::isLoan( a, &ai, &pay, &np );
	
	int row=0;
	isloan=new QCheckBox( this );
	connect( isloan, SIGNAL( toggled( bool ) ), SLOT( loaner( bool ) ) );
	lay->addMultiCellWidget( new QLabel( i18n( "This Account is a Loan" ), this ),
													 row, row, 0, 0 );
	lay->addMultiCellWidget( isloan, row, row, 1, 1 );
	
	row++;
	lay->addMultiCellWidget( new QLabel( i18n( "Annual Interest Rate" ), this ),
													 row, row, 0, 0 );
	anint=new QLineEdit( QString().setNum( ai, 'f', 3 ), this );
	lay->addMultiCellWidget( anint, row, row, 1, 1 );

	row++;
	lay->addMultiCellWidget( new QLabel( i18n( "Total Number of Payments" ),
																			 this ), row, row, 0, 0 );
	paysleft=new QLineEdit( QString().setNum( np ), this );
	lay->addMultiCellWidget( paysleft, row, row, 1, 1 );

	row++;
	lay->addMultiCellWidget( new QLabel( i18n( "Monthly Payment" ), this ),
													 row, row, 0, 0 );

	payment=new QLineEdit( engine->converter().convert( pay ), this );
	lay->addMultiCellWidget( payment, row, row, 1, 1 );

	isloan->setChecked( isaloan );
	loaner( isaloan );
}

AccountTab3::~AccountTab3(){}

void AccountTab3::modify( Account& a ) const {
	QString meta="";
	if( isloan->isChecked() && isEnabled() ){
		meta="loan("+anint->text()+","+
			engine->converter().convert( payment->text(), Preference, Engine )+","+
			paysleft->text()+")";
	}

	a.set( QC::AMETA, meta );
}

void AccountTab3::loaner( bool b ){
	QLineEdit * wds[]={ anint, payment, paysleft };
	for( int i=0; i<3; i++ ) wds[i]->setEnabled( b );
}

