/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2004 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************* * * * * * * * * * * * * ***************************/

#include "qhaccrecwin.h"
#include "qhacc.h"
#include "qhaccview.h"
#include "qhacctable.h"
#include "qhaccutils.h"
#include "qhaccconstants.h"
#include "qhacclineedits.h"
#include "qhaccrecwin.moc"

#include <qfont.h>
#include <qsize.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>

#include <klocale.h>

QHaccRecWin::QHaccRecWin( QHacc * e, const Journal& l,  QWidget * p,
													const char * n )
	: QWidget( p, n, WDestructiveClose ) {
  
	engine=e;
	journal=l;
  tView=new QHaccRecView( engine, this );

  enddate=new QHaccDateEdit( QDate(), this );
  enddate->setAlignment( AlignCenter );
  
	const QString zero=QString::number( 0.00 );
  ebal=new QHaccLineEdit( zero, this );
  ebal->setAlignment( AlignCenter );

  lblOBal=new QLabel( zero, this );
  lblDiff=new QLabel( zero, this );
  
  QPushButton * ok=new QPushButton( i18n( "Finished" ), this );
  QPushButton * cancel=new QPushButton( i18n( "Cancel" ), this );
  connect ( ok, SIGNAL( clicked() ), SLOT( accept() ) );
  connect ( cancel, SIGNAL( clicked() ), SLOT( cancel() ) );
  
  connect( enddate, SIGNAL( dateChanged( QDate ) ), SLOT( changeDate() ) );
  connect( ebal, SIGNAL( lostFocus() ), SLOT( refresh() ) );
  connect( ebal, SIGNAL( returnPressed() ), SLOT( refresh() ) );

	connect( engine, SIGNAL( addedT( const Transaction& ) ),
					 SLOT( modT( const Transaction& ) ) );
	connect( engine, SIGNAL( removedT( const Transaction& ) ),
					 SLOT( modT( const Transaction& ) ) );
	connect( engine, SIGNAL( updatedT( const Transaction& ) ),
					 SLOT( modT( const Transaction& ) ) );

	connect( engine, SIGNAL( updatedA( const Account&, const Account& ) ),
					 SLOT( modA( const Account&, const Account& ) ) );
  
  int align=AlignCenter;
  QGridLayout * layout=new QGridLayout( this, 15, 4, 5 );
  QWidget * widgs[]={ new QLabel( i18n( "closing date" ), this ), enddate,
											new QLabel( i18n( "opening balance" ), this ), lblOBal,
											new QLabel( i18n( "closing balance" ), this ), ebal,
											new QLabel( i18n( "difference" ), this ), lblDiff,
											ok, cancel };
	
  for( int i=0; i<10; i++ ){
		// the last two widgets should span the whole width
		if( i>7 ) align=0;
		layout->addMultiCellWidget( widgs[i], i, i, 0, 0, align );
  }
  layout->addMultiCellWidget( tView, 0, 14, 1, 3 );

	readPrefs( true );
}

QHaccRecWin::~QHaccRecWin(){}

void QHaccRecWin::modA( const Account& oldy, const Account& newy ){
	if( oldy[QC::AID]==account[QC::AID] ){
		account=newy;
		if( oldy[QC::ARBAL]!=newy[QC::ARBAL] ) refresh();
	}
}

void QHaccRecWin::setAccount( const Account& a ){
	account=a;
	changeDate();
	tView->setAccount( account );
	tView->setJournal( journal );
	setCaption( i18n( "Reconcile %1/%2" ).arg( journal.gets( QC::LNAME ) ).arg( account.gets( QC::ANAME ) ) );
}

void QHaccRecWin::readPrefs( bool initial ){
	tView->readPrefs( initial );
	if( initial ){
		connect( engine, SIGNAL( changedP( const QString&, QString ) ),
						 this, SLOT( changeP( const QString&, QString ) ) );
		connect( engine, SIGNAL( changedP( const QString&, int ) ),
						 this, SLOT( changeP( const QString&, int ) ) );
		connect( engine, SIGNAL( removedA( const Account& ) ),
						 this, SLOT( removeA( const Account& ) ) );
	}

	QString str=engine->getSP( "RECWINSIZE" );
	if( !str.isEmpty() ){
		QString vals[2];
		Utils::parser( str, " ", 0, vals, 2 );
		resize( vals[0].toInt(), vals[1].toInt() );
	}
}

void QHaccRecWin::save(){
	engine->setSP( "RECWINSIZE", QString::number( width() )+
								 " "+QString::number( height() ) );
	tView->save();
}

void QHaccRecWin::changeP( const QString& str, QString ){
	if( str=="DATESEPARATOR" ) enddate->refresh();
}

void QHaccRecWin::changeP( const QString& str, int ){
	if( str=="DATEFORMAT" )	enddate->refresh();
}

void QHaccRecWin::changeDate() {
	QDate end=enddate->getDate();

	// update the view with the correct date
	vector<TableSelect> criteria;
	criteria.push_back( TableSelect( QC::XSRECO, TableCol( QC::YREC ),
																	 TableSelect::NE ) );
	criteria.push_back( TableSelect( QC::XTDATE, TableCol( end.addDays( 1 ) ),
																 TableSelect::LT ) );

	tView->setSelectors( criteria );
	tView->setLimit( end );
	refresh();
}

void QHaccRecWin::refresh(){
	const QString& rbal=account.gets( QC::ARBAL ); // reconciled bal string
	const MonCon& conv=engine->converter();
	int userendbal=conv.converti( ebal->text(), Preference, Engine ); // end bal
	int arecbal=conv.converti( rbal, Engine, Engine );
	int recd=0;
	
	QDate end=enddate->getDate();
	vector<TableSelect> criteria;
	criteria.push_back( TableSelect( QC::XTDATE, end.addDays( 1 ),
																	 TableSelect::LT ) );
	criteria.push_back( TableSelect( QC::XSRECO, QC::YREC, TableSelect::NE ) );
	criteria.push_back( TableSelect( QC::XTVOID, false ) );

	uint rr=0;
	vector<int> tg;
	tg.push_back( QC::XSRECO );
	tg.push_back( QC::XSSUM );
		
	auto_ptr<QHaccResultSet> trans=engine->getXTForA( account, TableGet( tg ),
																										criteria, rr );
	for( uint i=0; i<rr; i++ ){
		Transaction t=trans->at( i );
		if( t[0]!=QC::NREC ) recd+=conv.converti( t[1].gets(), Engine, Engine );
	}

	lblOBal->setText( conv.convert( rbal ) );
	lblDiff->setText( conv.convert( userendbal-arecbal-recd ) );
}

void QHaccRecWin::accept(){
	hide();
	tView->disconnect();
	auto_ptr<QHaccTable> splits=tView->getContents();
	splits->deleteWhere( TableSelect( QC::XSRECO, TableCol( QC::MREC ),
																		TableSelect::NE ) );
	engine->setRec( splits.get(), QC::YREC );
	save();
	QWidget::close( true );
}

void QHaccRecWin::hide(){
	save();
  QWidget::hide();
}

void QHaccRecWin::cancel(){
	hide();
	save();
  QWidget::close( true );
}

void QHaccRecWin::removeA( const Account& a ){
	if( a[QC::AID]==account[QC::AID] ) close();
}
QHaccView * QHaccRecWin::getView() const { return tView; }

void QHaccRecWin::modT( const Transaction& t ){
	if( t[QC::XSACCTID]==account[QC::AID] )	refresh();
}
