/***************************************************************************
                          fractal.cpp  -  description
                             -------------------
    begin                : Fri Jun 23 2000
    copyright            : (C) 2000 by Helmut Steger
    email                : hel.steger@rolmail.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <strings.h>
#include <malloc.h>
#include "kisomandel.h"
#include "fractal.h"
#include "misc.h"
#include <math.h>


fractal::fractal(int width, int height) : QWidget()
{
	backpix=tmp_pix=(QPixmap *)NULL;
	painter=(QPainter *)NULL;
	calc=NULL;
	prefsWin=NULL;
	prefsPosX=prefsPosY=-1;
	data=isodata=NULL;
	xbuffer=NULL;
	cr_buffer=ci_buffer=NULL;
	prefsname="";
	dragMode=false;
	rangeSelectMode=false;
	juliaConstInteractiveMode=false;
	blitToWin=true;
	showWindow=true;
	paletteLoaded=false;
	showCalcStatus=false;

	old_fwidth=0;
	old_fheight=0;

	setDefaultCoords();	
	setPrecision (PRECISION_LDOUBLE);
	setFractalType ((FractType)kisomandel->getStartFractalType(),false);
	setZoomInOutFactor (3);
	setZoomInOutIterChange (25);	// 25 %
    changeIterations (DEFAULT_ITER);

	setMouseTracking (true);
	setCursor (CrossCursor);
	setFocusPolicy (QWidget::ClickFocus);
	setSmoothLevel (DEFAULT_SMOOTHLEVEL);
	setMinimumSize (FRACT_MINWIDTH,FRACT_MINHEIGHT);
	resetSize (width,height,true);
}


fractal::~fractal()
{
	closePrefsWindow();
	if (painter) painter->end();
	if (backpix) delete backpix;
	if (tmp_pix) delete tmp_pix;
	if (calc) delete calc;	
}


void fractal::resetSize (int width, int height, bool resizeWin)
{
	fwidth = (width > FRACT_MINWIDTH) ? width : FRACT_MINWIDTH;
	fheight = (height > FRACT_MINHEIGHT) ? height : FRACT_MINHEIGHT;
	if (fwidth>FRACT_MAXWIDTH) fwidth=FRACT_MAXWIDTH;
	if (fheight>FRACT_MAXHEIGHT) fheight=FRACT_MAXHEIGHT;

	int isores=mainwin->getRes();
	if (fwidth<isores)
	{
		fwidth=isores;
		resizeWin=true;
	}
	if (fheight<isores)
	{
		fheight=isores;
		resizeWin=true;
	}
	
	if (old_fwidth==fwidth && old_fheight==fheight) return;
	
	old_fwidth=fwidth;
	old_fheight=fheight;	
	
	if (xbuffer) delete xbuffer;
	xbuffer = new ITER[fwidth];
	
	if (cr_buffer) free (cr_buffer);
	cr_buffer = malloc (fwidth * MAX_PRECISION_SIZE);
	if (ci_buffer) free (ci_buffer);
	ci_buffer = malloc (fheight * MAX_PRECISION_SIZE);
	
	if (calc) calc->setInfo (cr_buffer,ci_buffer,&maxiter);
	
	createBackPixmap();
	
	if (data) free (data);	// free previous memory allocation
	data_size=sizeof(ITER)*fwidth*fheight;
	data=(ITER *)malloc (data_size);
	isodata=(ITER *)malloc (data_size);
	if ( data==NULL || isodata==NULL)
	{
		// TODO: errror handling
		puts ("*** fatal error: fractal/could not alloc data!!");
		return;
	}
	memset (data,0,data_size);	// clear memory
	
	if (resizeWin) setGeometry (x(),y(),fwidth,fheight);
	
	setFractalTitle();
}	


void fractal::closeEvent (QCloseEvent *e)
{
	if (kisomandel) kisomandel->slotFileQuit();
	//e->accept();
}


void fractal::resizeEvent (QResizeEvent *e)
{
	int w=e->size().width();
	int h=e->size().height();
	
	resetSize (w,h);
	mainwin->setIsoOffset();	// ensure that offset is not outside
	plotMandel();
}


void fractal::keyPressEvent (QKeyEvent *e)
{
	mainwin->handleKeyCommand ( e->key(),e->state() );
}


void fractal::mousePressEvent (QMouseEvent *e)
{
	int mousex=e->x();
	int mousey=e->y();
	
	switch (e->button())
	{
		case LeftButton :
			if (e->state() == ControlButton)
			{
				oldDragX=mousex;
				oldDragY=mousey;
				dragMode=true;
				finishRangeMode();
			} else
			{
				if (rangeSelectMode)
				{
					finishRangeMode();
	    			rangeX2=mousex;
	    			//rangeY2=mousey;
	    			rangeY2=rangeY1 + (short) (  (double)((rangeX2-rangeX1)) * ((double)fheight) / ((double)fwidth) );
					zoomRange (rangeX1,rangeY1,rangeX2,rangeY2);
				}
			}
			break;
						
		case MidButton :
			oldDragX=mousex;
			oldDragY=mousey;
			dragMode=true;
			finishRangeMode();
			break;
			
		case RightButton :
			if (!juliaConstInteractiveMode)
			{
    			if (rangeSelectMode)
    			{
    				finishRangeMode();
    			}
    			else
    			{
    				rangeSelectMode=true;
    				oldRangeX1=-1;
    				rangeX1=rangeX2=mousex;
    				rangeY1=rangeY2=mousey;
    			}
    		}
			break;
		
		default :
			finishRangeMode();
			break;
	}	// switch
}


void fractal::mouseReleaseEvent (QMouseEvent *e)
{
	if ( (e->button() == MidButton) || (e->state() & (LeftButton|ControlButton)) )
	{
		if (dragMode)
		{
			smoothData();
			dragMode=false;
			mainwin->showIso();
			updatePrefsWindow();
		}
	}
}


void fractal::mouseDoubleClickEvent (QMouseEvent *e)
{
	int mousex=e->x();
	int mousey=e->y();

	finishRangeMode();
	
 	switch (e->button())
 	{
 		case LeftButton :	// zoom in
 			zoom (mousex,mousey,(FLT)zoomInOutFactor,maxiter + (maxiter*zoomInOutIterChange/100) );
 			break;
 		case RightButton :	// zoom out
 			zoom (mousex,mousey,1.0/(FLT)zoomInOutFactor,maxiter - (maxiter*zoomInOutIterChange/100) );
 			break;
 		default:
 			break;
 	}
}


void fractal::mouseMoveEvent (QMouseEvent *e)
{
	int mousex=e->x();
	int mousey=e->y();
	
	switch (e->state())
	{
		case LeftButton :
    		mainwin->setIsoOffset (mousex,mousey,true);
    		mainwin->showIso();
    		break;

		case RightButton :
		{
    		if (prefsWin && juliaConstInteractiveMode)
    		{
        		if (mousex>=0 && mousex<fwidth && mousey>=0 && mousey<fheight)
        		{
    	    		long double	cr = calc->getCr (mousex);
    	    		long double ci = calc->getCi (mousey);
        			prefsWin->plotJuliaPreview (cr,ci);
        			updatePrefsWindow();
        		}
        	}
    		break;
    	}
    	
    	case (LeftButton | ControlButton) :
    	
    	case MidButton :
    	{
    		int x = oldDragX-mousex;
    		int y = oldDragY-mousey;
    		int absx = abs(x);
    		int absy = abs(y);
    		
    		if ((absx<=MIN_STEP_SMOOTH_SCROLL) && (absy<=MIN_STEP_SMOOTH_SCROLL))
    		{
    			// scroll in one step
    			scrollxy (x,y);
    		}
    		else
    		{
	    		// scroll in more steps
    			if (absx>absy)
    			{
    				int sx=0;
    				int prev_rely=0;
    				while (sx < absx)
    				{
    					int relx = min(MIN_STEP_SMOOTH_SCROLL,absx-sx);
    					sx+=relx;
    					int rely = y * sx / absx;
    					
    					scrollxy ( (x>0) ? relx : -relx , rely-prev_rely);
    					bitBlt (this,0,0,backpix,0,0,fwidth,fheight);
    					prev_rely=rely;
    				}
    			}
    			else
    			{
    				int sy=0;
    				int prev_relx=0;
    				while (sy < absy)
    				{
    					int rely = min(MIN_STEP_SMOOTH_SCROLL,absy-sy);
    					sy+=rely;
    					int relx = x * sy / absy;
    					
    					scrollxy (relx-prev_relx , (y>0) ? rely : -rely);
    					bitBlt (this,0,0,backpix,0,0,fwidth,fheight);
    					prev_relx=relx;
    				}
    			}
    		}
    		oldDragX=mousex;
    		oldDragY=mousey;
    		break;
    	}

    	case NoButton :
    	{
    		if (mousex>=0 && mousex<fwidth && mousey>=0 && mousey<fheight)
    		{
	    		QString s,cr_str,ci_str;
	    		long double	cr = calc->getCr (mousex);
	    		long double ci = calc->getCi (mousey);
	    		
	    		fnum2str (cr,cr_str,20);
	    		fnum2str (ci,ci_str,20);
	    		
	    		s.sprintf ("X=%s Y=%s iter=%d",
	    			(const char *) (cr_str.leftJustify (24,'0') ),
	    			(const char *) (ci_str.leftJustify (24,'0') ),
	    			getData (mousex,mousey)
	    			);
	    		kisomandel->statusBar()->message(s, 2000);
	    		
	    		if (rangeSelectMode)
	    		{
	    			rangeX2=mousex;
	    			//rangeY2=mousey;
	    			rangeY2=rangeY1 + (short) (  (double)((rangeX2-rangeX1)) * ((double)fheight) / ((double)fwidth) );
	    			drawRangeRect();
				}
	    	}
	    	break;
	    }
	    default:
	    	break;
	}	// switch
}


void fractal::blitRectangle (short x1, short y1, short x2, short y2, bool drawrect)
{
	if (x1>x2)
	{
		short tmp=x1;
		x1=x2;
		x2=tmp;
	}
	if (y1>y2)
	{
		short tmp=y1;
		y1=y2;
		y2=tmp;
	}
	
	if (drawrect && x2>x1 && y2>y1)
	{
		RasterOp saverop = painter->rasterOp();
	  	painter->setRasterOp (NotROP);	// inverse mode
		painter->drawRect (x1,y1,x2-x1+1,y2-y1+1);
  		painter->setRasterOp (saverop);
  	}
  	
  	// line (x1,y1,x2,y1)
 	bitBlt (this,x1,y1,backpix,x1,y1,x2-x1+1,1);
  	// line (x1,y2,x2,y2)
  	bitBlt (this,x1,y2,backpix,x1,y2,x2-x1+1,1);
  	// line (x1,y1,x1,y2)
  	bitBlt (this,x1,y1,backpix,x1,y1,1,y2-y1+1);
  	// line (x2,y1,x2,y2)
  	bitBlt (this,x2,y1,backpix,x2,y1,1,y2-y1+1);
}


void fractal::drawRangeRect()
{
  	if (oldRangeX1 != -1) blitRectangle (oldRangeX1,oldRangeY1,oldRangeX2,oldRangeY2,true);
	blitRectangle (rangeX1,rangeY1,rangeX2,rangeY2,true);
  	oldRangeX1=rangeX1;
  	oldRangeY1=rangeY1;
  	oldRangeX2=rangeX2;
  	oldRangeY2=rangeY2;
}


void fractal::finishRangeMode()
{
	if (rangeSelectMode)
	{
		blitRectangle (rangeX1,rangeY1,rangeX2,rangeY2,true);
		rangeSelectMode=false;
	}
}


void fractal::setPrecision (short p, bool plotmandel)
{
	short oldp = precision;
	
	if (calc) delete calc;
	
	switch (p)
	{
		case PRECISION_FLOAT :
			puts ("fractal::setPrecision (float)");
			precision=p;
			calc = new calculation ();
			break;
		
		case PRECISION_DOUBLE :
			puts ("fractal::setPrecision (double)");
			precision=p;
			calc = new calculation_Double ();
			break;
		
		default :
			puts ("fractal::setPrecision (long double)");
			precision=PRECISION_LDOUBLE;
			calc = new calculation_LongDouble ();
			break;
	}
	
	calc->setInfo (cr_buffer,ci_buffer,&maxiter);
	
	if ( plotmandel && (p != oldp) ) plotMandel();
	
}


void fractal::setFractalType (FractType f, bool plot)
{
	if ( (!calc) || (f != fractalType) )
	{
		switch (f)
		{
			case Fract_Julia :
				fractalType=f;
				setJulia();
				break;
			
			case Fract_Mandel :
			default :
				fractalType=Fract_Mandel;
				setPrecision (precision);
				break;
		}	// switch
		if (plot) plotMandel();		
	}
	setFractalTitle();
}
	

void fractal::setJulia()
{
	if (calc) delete calc;
	calc = new calculation_Julia ();
	calc->setInfo (cr_buffer,ci_buffer,&maxiter);
	calc->setJuliaConst (julia_rconst,julia_iconst);
	precision=PRECISION_LDOUBLE_JULIA;
	// plotMandel();
}
	

void fractal::calcCrCiBuffer (int x1, int y1, int x2, int y2)
{
	int x,y;
	long double dx=(rmax-rmin)/fwidth;
	long double dy=(imax-imin)/fheight;
	
	switch (precision)
	{
		case PRECISION_FLOAT :
		{
			for (x=x1;x<=x2;x++) ((float *)cr_buffer)[x]=(float) (rmin+dx*(FLT)x);
			for (y=y1;y<=y2;y++) ((float *)ci_buffer)[y]=(float) (imin+dy*(FLT)y);
			break;
		}
		
		case PRECISION_DOUBLE :
		{
			for (x=x1;x<=x2;x++) ((double *)cr_buffer)[x]=(double) (rmin+dx*(FLT)x);
			for (y=y1;y<=y2;y++) ((double *)ci_buffer)[y]=(double) (imin+dy*(FLT)y);
			break;
		}

		default :
		{
			for (x=x1;x<=x2;x++) ((long double *)cr_buffer)[x]=(long double) (rmin+dx*(FLT)x);
			for (y=y1;y<=y2;y++) ((long double *)ci_buffer)[y]=(long double) (imin+dy*(FLT)y);
			break;
		}
	}
}


void fractal::changeIterations (ITER newiter)
{
   	maxiter=validIterations (newiter);
   	initPalette(maxiter,paletteLoaded);
}


void fractal::setCoords (FLT r_min, FLT r_max, FLT i_min, FLT i_max)
{
	if (r_min < r_max && i_min < i_max)
	{
		rmin=r_min;
		rmax=r_max;
		imin=i_min;
		imax=i_max;
	}
}


void fractal::adjustReal()
{
	FLT isize=imax-imin;
	
	rmax=rmin+isize*4.0/3.0;
	plotMandel();
	updatePrefsWindow();
}	


void fractal::adjustImag()
{
	FLT rsize=rmax-rmin;
	
	imax=imin+rsize*3.0/4.0;
	plotMandel();
	updatePrefsWindow();
}	


void fractal::setDefaultCoords()
{
	setCoords (-2.0,2.0,-1.5,1.5);
	julia_rconst=-0.8;
	julia_iconst=-0.2;
}


void fractal::setZoomInOutFactor (short z)
{
	if (z<2) z=2;
	if (z>6) z=6;
	zoomInOutFactor=z;
}
	
	
void fractal::setZoomInOutIterChange (short i)
{
	if (i<0) i=0;
	if (i>50) i=50;
	zoomInOutIterChange=i;
}


void fractal::zoom (int x, int y, FLT fact, ITER newiter)
{
	if (fact != 0.0)
	{
		FLT rsize=rmax-rmin;
		FLT isize=imax-imin;
		
		rmin=rmin+(rsize*x/fwidth)-rsize/(fact*2);
		rmax=rmin+rsize/fact;
		imin=imin+(isize*y/fheight)-isize/(fact*2);
		imax=imin+isize/fact;
		changeIterations (newiter);
		plotMandel();
		updatePrefsWindow();
	}
}


void fractal::zoomRange (short rangeX1, short rangeY1, short rangeX2, short rangeY2)
{


	if (rangeX1>rangeX2)
	{
		short tmp=rangeX1;
		rangeX1=rangeX2;
		rangeX2=tmp;
	}
	if (rangeX1==rangeX2) rangeX2++;
	
	
	if (rangeY1>rangeY2)
	{
		short tmp=rangeY1;
		rangeY1=rangeY2;
		rangeY2=tmp;
	}
	if (rangeY1==rangeY2) rangeY2++;

	FLT rsize=rmax-rmin;
	FLT isize=imax-imin;
	FLT rleft=rmin;
	FLT itop=imin;
	
	rmin=rleft+((FLT)rangeX1)*rsize/(FLT)fwidth;
	rmax=rleft+((FLT)rangeX2)*rsize/(FLT)fwidth;
	imin=itop+((FLT)rangeY1)*isize/(FLT)fheight;
	imax=itop+((FLT)rangeY2)*isize/(FLT)fheight;
	
	changeIterations (maxiter + (maxiter*zoomInOutIterChange/100) );
	plotMandel();
	updatePrefsWindow();
}	


void fractal::createBackPixmap()
{
	if (painter)
	{
		painter->end();
		delete painter;
	}
	
	if (tmp_pix) delete tmp_pix;
	tmp_pix = new QPixmap (fwidth,fheight);		// temp pixmap for scrolling
	
	if (backpix) delete backpix;
	backpix = new QPixmap (fwidth,fheight);
	setBackgroundPixmap (*backpix);
	painter = new QPainter (backpix);
	painter->fillRect (0,0,fwidth,fheight,black);
	bitBlt (this,0,0,backpix,0,0,fwidth,fheight);
}


ITER fractal::validIterations (ITER iter)
{
	if (iter<MIN_ITER) iter=MIN_ITER;
	if (iter>MAX_ITER) iter=MAX_ITER;
	return iter;
}


void fractal::plotMandel (bool calcit)
{
	showCalcStatus=true;
	calc->setJuliaConst (julia_rconst,julia_iconst);
	rangeSelectMode=false;
	plotMandel (0,0,fwidth-1,fheight-1,calcit);
	showCalcStatus=false;
}


void fractal::plotMandel (int x1, int y1, int x2, int y2, bool calcit)
{
	mainwin->hideIsoRange();
	calcCrCiBuffer(x1,y1,x2,y2);
	plot (x1,y1,x2,y2,calcit);
	mainwin->showIsoRange();
	if (!dragMode)
	{
		smoothData();
		mainwin->showIso();
	}
}


void fractal::plot (int x1, int y1, int x2, int y2, bool calcit)
{
	int x,y;
	int startx;			// for fast draw (lines instead of points)
	ITER col,oldcol;
	short ycount=0,ystep;
	int count_points=0;	// number of total points drawn
	int count_lines=0;	// number of total lines drawn
	int msec=0,curr_msec;
	QTime *t=NULL;
	
	ystep=(y2-y1+1)/20; if (ystep<1) ystep=1;
	
	if (showCalcStatus)
	{
		t=new QTime;
		t->start();
	}
	
	for (y=y1;y<=y2;y++)
	{
		if (calcit)
		{
			for (x=x1;x<=x2;x++) xbuffer[x]=calc->calcPoint(x,y);
			if (showCalcStatus)
			{
				curr_msec=t->elapsed();
				if ( (curr_msec - msec) >= 300 )	// show calc status at least every 0.3 seconds
				{
					QString s;
					s.sprintf ("Status: %d %%",y*100/(y2-y1+1));
					setCaption ( (const char *)s);
					QApplication::flushX();
					msec = curr_msec;
				}
			}
		}
		else
		{
			for (x=x1;x<=x2;x++) xbuffer[x]=getData(x,y);
		}

		startx=x1;
		oldcol=xbuffer[x1];
		setData (x1,y,oldcol);
		painter->setPen (*colorpalette[oldcol]);
		for (x=x1+1;x<=x2;x++)
		{
			col=xbuffer[x];
			setData(x,y,col);
			
			if (col != oldcol)
			{
				if (x-startx > 1)	// same color for more than one point ?
				{
					painter->drawLine (startx,y,x-1,y);	// draw line instead of more points with same color
					count_lines++;
				}
				else
				{
					painter->drawPoint (startx,y);
					count_points++;
				}
				startx=x;
				painter->setPen (*colorpalette[col]);
				oldcol=col;
			}
		}

		if (startx != x2)
		{
			painter->drawLine (startx,y,x2,y);
			count_lines++;
		}
		else
		{
			painter->drawPoint (x2,y);
		}
		
		// blit from background pixmap to fractal window
		if ( ((ycount ++ % ystep) == 0) && blitToWin ) bitBlt (this,0,y-ystep+1,backpix,0,y-ystep+1,fwidth,ystep);
	}
	
	if (blitToWin) bitBlt (this,0,fheight-ystep,backpix,0,fheight-ystep,fwidth,ystep+1);
	
	if (showCalcStatus)
	{
		delete t;	// timer
		setFractalTitle();
	}

	/*
	printf (
		"PlotMandel (%d x %d = %d points): %d points and %d lines drawn (%d total draw calls)\n",
		x2-x1+1,
		y2-y1+1,
		(x2-x1+1)*(y2-y1+1),
		count_points,
		count_lines,
		count_points+count_lines);
	*/
}


void fractal::calcMandelRect (int x1, int y1, int x2, int y2)
{
	if (x1<0) x1=0;
	if (y1<0) y1=0;
	if (x2>=fwidth) x2=fwidth-1;
	if (y2>=fheight) y2=fheight-1;
	if ( (x1<=x2) && (y1<=y2) ) plotMandel (x1,y1,x2,y2);
}


void fractal::scrollWin (int stepx, int stepy)
{
	int sx,sy,dx,dy;
	int offs,copy_size;

	if (stepx == 0 && stepy == 0) return;
	
	offs=(stepy*fwidth+stepx)*sizeof(ITER);
	copy_size=data_size-abs(offs);		// important: abs !!!!

	if (copy_size>0)
	{
		if (offs>0)
			memmove ( (char *)data + offs,data,copy_size);
		else
			memmove (data, (char *)data - offs,copy_size);
	}

	mainwin->hideIsoRange();

	QPainter p (tmp_pix);
	
	sx = (stepx<0) ? -stepx : 0;
	sy = (stepy<0) ? -stepy : 0;
	dx = (stepx<0) ? 0 : stepx;
	dy = (stepy<0) ? 0 : stepy;
	
	bitBlt (tmp_pix,dx,dy,backpix,sx,sy,fwidth-abs(stepx),fheight-abs(stepy));
	p.end();

	painter->drawPixmap (0,0,*tmp_pix);
	
	mainwin->showIsoRange(false);
}


void fractal::scrollxy (int stepx, int stepy)
{
	int w=fwidth-1;
	int h=fheight-1;
	FLT radd=(rmax - rmin)/(FLT)fwidth*stepx;
	rmin = rmin + radd;
	rmax = rmax + radd;
	FLT iadd=(imax - imin)/(FLT)fheight*stepy;
	imin = imin + iadd;
	imax = imax + iadd;
	int sx=0;
	int ex=w;

	if (stepx > w    )	stepx = w;
	if (stepx < (-w) )	stepx = -w;
	if (stepy > h    )	stepy = h;
	if (stepy < (-h) )	stepy = -h;

	blitToWin=false;

	scrollWin (-stepx,-stepy);
   	if (stepx>0)	// left
   	{
   		calcMandelRect (fwidth-stepx,0,fwidth-1,fheight-1);
   		ex=fwidth-stepx-1;
   	}
   	else if (stepx<0)	// right
   	{
   		stepx=-stepx;
   		calcMandelRect (0,0,stepx-1,fheight-1);
   		sx=stepx;
   	}
	
   	if (stepy>0)	// up
   	{
   		calcMandelRect (sx,fheight-stepy,ex,fheight-1);
   	}
   	else if (stepy<0)	// down
   	{
   		stepy=-stepy;
   		calcMandelRect (sx,0,ex,stepy-1);
   	}
   	
   	update();
   	if (!dragMode) updatePrefsWindow();
   	blitToWin=true;
}		


void fractal::setSmoothLevel (short level)
{
	if (level<0) level=0;
	if (level>MAX_SMOOTHLEVEL) level=MAX_SMOOTHLEVEL;
	smoothLevel=level;
}


void fractal::smoothData()
{
	short x,y,c;
	short w=fwidth-1;
		
	memcpy (isodata,data,data_size);	// we start with isodata=data
	
	for (c=0;c<smoothLevel;c++)
	{
		for (y=0;y<fheight;y++)
		{
			short ym1=y-1; if (ym1<0) ym1=0;
			for (x=0;x<fwidth;x++)
			{
				ITER i1=getIsoData (x,y);
				ITER i2=getIsoData ( (x==w) ? x : x+1 , y);
				ITER i3=getIsoData (x,ym1);
				ITER i4=getIsoData ( (x==0) ? 0 : x-1 , y);
				
				setIsoData (x,y, (i1+i2+i3+i4)>>2 );	// average of the 4 points
			}
		}
	}
}


void fractal::setFractalTitle ()
{
	QString s,tmp,tmp2,ft;
	int i;

	tmp = prefsname;
	if (prefsname.isEmpty()) tmp="Unnamed";
	tmp2=tmp;
	
	i=tmp.findRev ('/');
	if (i < 0) i=tmp.findRev ('\\');	// backslash (windows)
	if (i > 0) tmp=tmp2.right(tmp2.length()-i-1);	// filename without path
	if (tmp.length()>50)
	{
		tmp.truncate (50);
		tmp=tmp+"...";
	}
	switch (fractalType)
	{
		case Fract_Julia :
		 	ft="Julia";
		 	break;
		 case Fract_Mandel :
		 	ft="Mandel";
		 	break;
		 default :
		 	ft="Fractal";
		 	break;
	}
	s.sprintf ("%s %dx%d (%s)",(const char *)ft,fwidth,fheight,(const char *)tmp);
	setCaption ((const char *)s);
}


void fractal::openPrefsWindow()
{
	if (!prefsWin)	// already open?
	{
		prefsWin = new FractPrefs ();
	}
	prefsWin->show();
	prefsWin->raise();
}


void fractal::closePrefsWindow()
{
	if (prefsWin)
	{
		// remember window position
		prefsPosX=prefsWin->x();
		prefsPosY=prefsWin->y();
		prefsSize=prefsWin->size();
		delete prefsWin;
		prefsWin=NULL;
		puts ("fractal::prefsWin closed");
	}
}


void fractal::updatePrefsWindow()
{
	if (prefsWin) prefsWin->updateValues();
}


bool fractal::saveParameters (const char *filename)
{
	FILE *f;

	printf ("fractal::saveParameters (%s)\n",filename);
	f=fopen (filename,"r");		// check if file exists
	if (f)
	{
		fclose (f);
		int overwrite=QMessageBox::warning (
					this, "Save fractal parameters",
                    "File already exists!\n" \
                    "Do you want to overwrite it ?",
                    QMessageBox::Yes, QMessageBox::No );


        if (overwrite == QMessageBox::No) return false;	// don't overwrite
	}
	
	f=fopen (filename,"w");
	if (f)
	{
		QString s;
		fputs ("#Kisomandel\n",f);
		fputs ("#fractprefs\n",f);
		s="Mandel";
		if (fractalType == Fract_Julia) s="Julia";
		fprintf (f,"FractalType %s\n",(const char *)s);
		fnum2str (rmin,s); fprintf (f,"rmin     %s\n",(const char *)s);
		fnum2str (rmax,s); fprintf (f,"rmax     %s\n",(const char *)s);
		fnum2str (imin,s); fprintf (f,"imin     %s\n",(const char *)s);
		fnum2str (imax,s); fprintf (f,"imax     %s\n",(const char *)s);
		fnum2str (julia_rconst,s); fprintf (f,"juliarconst %s\n",(const char *)s);
		fnum2str (julia_iconst,s); fprintf (f,"juliaiconst %s\n",(const char *)s);
		fprintf (f,"maxiter  %d\n",maxiter);
		fprintf (f,"width    %d\n",fwidth);
		fprintf (f,"height   %d\n",fheight);
		fclose (f);
		prefsname=filename;
		setFractalTitle ();
	}
	else
	{
		QMessageBox::critical (
			this,
			"Error",
			"Could not create file for saving the fractal parameters!");
		return false;
	}
	return true;
}


bool fractal::loadParameters (const char *filename)
{
	FILE *f;
	char s[500];	// string buffer for reading
	FractType new_fractalType=Fract_Mandel;
	char fractalType[50];
	bool kisomandel_identify=false;
	bool fractalprefs_identify=false;
	bool string_parsed;
	bool winsize_ok;
    PREFVAL tmp_rmin,tmp_imin;
    PREFVAL tmp_rmax,tmp_imax;
    PREFVAL tmp_maxiter;
    PREFVAL tmp_width,tmp_height;
    PREFVAL tmp_juliarconst;
    PREFVAL tmp_juliaiconst;

    struct Prefs fp[]=
    {
    	{"FractalType",0,fractalType,sizeof(fractalType)-1,false},
    	{"rmin",0,NULL,0,false},
    	{"imin",0,NULL,0,false},
    	{"rmax",0,NULL,0,false},
    	{"imax",0,NULL,0,false},
    	{"juliarconst",0,NULL,0,false},
    	{"juliaiconst",0,NULL,0,false},
    	{"maxiter",0,NULL,0,false},
    	{"width",0,NULL,0,false},
    	{"height",0,NULL,0,false},
    	{NULL,0,NULL,0,false}	// end prefs
    };
  	
    printf ("fractal::loadParameters (%s)\n",filename);

	try
	{
        f=fopen (filename,"r");		// check if file exists
        if (!f) throw ("Could not open file!");

         while ( ! feof(f) )
         {
         	int len;
         	fgets (s,sizeof(s)-1,f);	// read one line
         	len=strlen(s);
         	if (len>0) if (s[len-1]=='\n') s[len-1]=0;
         	lowercase (s);
         	string_parsed=false;
      	 	
         	if (!kisomandel_identify)
         	{
         		if ( strcmp (s,"#kisomandel") == 0 )
         		{
         			kisomandel_identify=true;
         			string_parsed=true;
         			puts ("#kisomandel identification found");
         		}
         	}
         	if (!fractalprefs_identify)
         	{
         		if ( strcmp (s,"#fractprefs") == 0 )
         		{
         			fractalprefs_identify=true;
         			string_parsed=true;
         			puts ("#fractalprefs identification found");
         		}
         	}
      	 	
         	if (kisomandel_identify && fractalprefs_identify && (!string_parsed))
         	{
        		char token[50];
        		char str_value[50];
        		PREFVAL value;
      			
	      		// try to get token with numeric value         		
         		if (sscanf (s,"%40s " PREFVAL_FMT, &token,&value) == 2)	// token and value scanned
         		{
         		 	prefsSetTokenValue (fp, token, value);
         		}
         	
	      		// try to get token with string value         		
         		if (sscanf (s,"%40s %40s",&token,&str_value) == 2)	// token and value scanned
         		{
         		 	prefsSetTokenValue (fp, token, str_value);
         		}
         	}
         }	// while
         fclose (f);

         if (prefsGetTokenValue (fp,"FractalType",s))
         {
              QString s2 = s;
              if (s2.lower() == "mandelbrot" || s2.lower() == "mandel") new_fractalType=Fract_Mandel;
              if (s2.lower() == "julia") new_fractalType=Fract_Julia;
         }

         bool coords_ok =
         	prefsGetTokenValue (fp,"rmin",tmp_rmin) &&
         	prefsGetTokenValue (fp,"rmax",tmp_rmax) &&
         	prefsGetTokenValue (fp,"imin",tmp_imin) &&
         	prefsGetTokenValue (fp,"imax",tmp_imax);

         bool maxiter_ok =
         	prefsGetTokenValue (fp,"maxiter",tmp_maxiter);
         	
         winsize_ok =
         	prefsGetTokenValue (fp,"width",tmp_width) &&
         	prefsGetTokenValue (fp,"height",tmp_height);

    	 	if ( ! (kisomandel_identify && fractalprefs_identify) )
    	 		throw ("Not a valid preferences file!");
    	 	
    	 	if (coords_ok)
    	 	{
    	 		if (tmp_rmin>tmp_rmax || tmp_imin>tmp_imax)
    	 			throw ("Bad values: rmax/imax must be greater than rmin/imin!");
    	 	}
    	 	else throw ("Could not get values: rmin,imin,rmax,imax!");
    	 	
    	 		
    	 	
    	 	if ( !maxiter_ok )
    	 		throw ("Could not get value: maxiter");

     }	// try
   	
   	 catch (const char *msg)
	 {
    	QMessageBox::critical (this,"Error loading fractal parameters",msg);
    	return false;
	 }

	setFractalType (new_fractalType,false);

	rmin=tmp_rmin; imin=tmp_imin;
	rmax=tmp_rmax; imax=tmp_imax;

    if (prefsGetTokenValue (fp,"juliarconst",tmp_juliarconst))
    {
    	julia_rconst=tmp_juliarconst;
    }
    if (prefsGetTokenValue (fp,"juliaiconst",tmp_juliaiconst))
    {
    	julia_iconst=tmp_juliaiconst;
    }
		
	if (winsize_ok)
	{
		fwidth = (int)tmp_width;
		fheight = (int)tmp_height;
	}

	mainwin->setIsoOffset();	// ensure that offset is not outside
	changeIterations ( (ITER)tmp_maxiter);
	
	raise();	// fractal window to top
	
	if (old_fwidth != fwidth || old_fheight != fheight)
	{
		resetSize (fwidth,fheight,true);
	} else
	{
		updatePrefsWindow();
		plotMandel();
	}
	
	prefsname=filename;
	setFractalTitle();
	return true;
}


bool fractal::loadPalette (const char *filename)
{
	FILE *f;
	char s[500];
	int reading_color=0;
	
	try
	{
        f=fopen (filename,"r");		// check if file exists
        QString err = "Could not open file (" + QString(filename) + ")";
        if (!f) throw (err);

         while ( ! feof(f) && (reading_color < MAX_ITER) )
         {
         	int len;
         	fgets (s,sizeof(s)-1,f);	// read one line
         	len=strlen(s);
         	if (len>0) if (s[len-1]=='\n') s[len-1]=0;
	
         	int r,g,b;
         	if (sscanf (s,"%d %d %d",&r,&g,&b) == 3)	// color read
         	{
         		colors_rgb[reading_color].r = r & 255;
         		colors_rgb[reading_color].g = g & 255;
         		colors_rgb[reading_color].b = b & 255;
         		reading_color++;
         	}
         }	// while
         fclose (f);

		 if (reading_color == 0)
		 	throw (QString("Bad file format!\nFormat must be: Red Green Blue (example: 100 56 230)"));

     }	// try
   	
   	 catch (QString msg)
	 {
    	QMessageBox::critical (this,"Error loading palette",msg);
    	return false;
	 }
	
	 // duplicate palette if number of read colors < MAX_ITER
	
	 int source=0;
	 for (int i=reading_color;i<MAX_ITER;i++)
	 {
	 	colors_rgb[i]=colors_rgb[source++];
	 	if (source>=reading_color) source=0;
	 }
	
	 initPalette (maxiter,paletteLoaded=true);
	 plotMandel (false);
	 if (prefsWin) prefsWin->plotJuliaPreview();
	 mainwin->showIso();
	 return true;
}



//////////// JuliaPreview ////////////

JuliaPreview::JuliaPreview (QWidget *w)
{
	const long double rmin=-2.0;
	const long double rmax=2.0;
	const long double imin=-1.5;
	const long double imax=1.5;

	widget = w;
	wi=widget->width();
	he=widget->height();

	widget->setFixedSize (QSize (wi,he));
	backpix=new QPixmap (wi,he);
	widget->setBackgroundPixmap (*backpix);
	painter = new QPainter (backpix);
	painter->fillRect (0,0,wi,he,QWidget::black);

	// Init Cr Buffer
	cr_buffer = new long double [wi];
	long double dx = (rmax-rmin)/(long double)wi;
	long double r=rmin;
	for (int x=0;x<wi;x++)
	{
		cr_buffer[x]=r;
		r=r+dx;
	}
	
	// Init Ci Buffer
	ci_buffer = new long double [he];
	long double dy = (imax-imin)/(long double)he;
	long double i=imin;
	for (int y=0;y<he;y++)
	{
		ci_buffer[y]=i;
		i=i+dy;
	}
	juliacalc = new calculation_Julia();
	juliacalc->setInfo (cr_buffer,ci_buffer,&fract->maxiter);
}


void JuliaPreview::plot()
{
	juliacalc->setJuliaConst (real_const,imag_const);
	
	for (int y=0;y<he;y++)
	{
		for (int x=0;x<wi;x++)
		{
			ITER col = juliacalc->calcPoint (x,y);
			painter->setPen (*colorpalette[col]);
			painter->drawPoint (x,y);
		}
	}
	bitBlt (widget,0,0,backpix,0,0,wi,he);
}


void JuliaPreview::setJuliaConst (long double r, long double i)
{
	real_const=r;
	imag_const=i;
}


JuliaPreview::~JuliaPreview()
{
	if (painter) painter->end();
	if (backpix) delete backpix;
	delete juliacalc;
	delete [] cr_buffer;
	delete [] ci_buffer;
	
}
