/***************************************************************************
                          isowin.cpp  -  description
                             -------------------
    begin                : Thu Jun 22 2000
    copyright            : (C) 2000 by Helmut Steger
    email                : hel.steger@rolmail.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <ctype.h>
#include "isowin.h"
#include "kisomandel.h"
#include "settings.h"



static const BarDrawInfo bdi[]=
{	// stepy,stepx,bx1,by1,bx2,by2,bx3,by3
  {1,-1,0,0,0,1,1,1,1,0},	// 0 - 89 Grad
  {-1,-1,0,1,0,0,1,0,1,1},	// 90 - 179 Grad
  {-1,1,1,1,1,0,0,0,0,1},	// 180 - 269 Grad
  {1,1,1,0,1,1,0,1,0,0}
};



isowin::isowin (QWidget * parent, const struct GlobalPrefs &gp) : QWidget (parent)
{
	backpix=(QPixmap *)NULL;
	painter=(QPainter *)NULL;
	fract=NULL;
	prefsWin=NULL;
	prefsPosX=prefsPosY=-1;
	generalPrefsWin=NULL;
	generalPrefsPosX=generalPrefsPosY=-1;
	
	old_isowidth=0;
	old_isoheight=0;
	isoRangeDrawn=false;
	startX=0;
	startY=0;
	dispMode=ISOMODE_LINES;
	showHorLines=true;
	showVerLines=true;
	areaOutline=gp.areaOutline;
	autoHeight=gp.autoHeight;
	fullScreen=false;
	printMode=false;
	heightFactor=(float)(gp.isoHeight) / 100.0;
	resX=validRes(gp.isoResolution+1);
	resY=resX;
	drawYStart=0;
	setRotate (450);
	setVertRotate (400);
	setZoom ((float)(gp.isoZoom) / 100.0);
	
	linecolor=QColor (gp.lineColor);
	outlinecolor=QColor (gp.outlineColor);
	QColor col (gp.maxiterColor);
	setMaxiterColor (&col);
	lineWidth=0;
	//setMouseTracking (true);
	setFocusPolicy (QWidget::ClickFocus);
	setMinimumSize (WIN_MINWIDTH,WIN_MINHEIGHT);
	centerRelX=0.5;
	centerRelY=0.5;
	resetSize (width(),height(),true);
}


isowin::~isowin()
{
	closePrefsWindow();
	closeGeneralSettingsWindow();
	if (painter) painter->end();
	if (backpix) delete backpix;
}


void isowin::createBackPixmap()
{
	if (painter) painter->end();
	if (backpix) delete backpix;
	
	backpix=new QPixmap (isowidth,isoheight);
	setBackgroundPixmap (*backpix);
	painter = new QPainter (backpix);
	painter->fillRect (0,0,isowidth,isoheight,black);	
}


void isowin::printIso ()
{
	int old_isowidth = isowidth;
	int old_isoheight = isoheight;
	float height_fact = (float)isoheight/(float)isowidth;
	
	QPrinter printer;
	if (printer.setup(this))
	{
	    QPainter *p = new QPainter (&printer);
        QRect r = p->viewport();
        int w = r.width();
        int h = r.height();

        printf ("Print: Viewport Size = %d x %d\n",w,h);
	    QPaintDeviceMetrics pdm(&printer);
	    printf ("Print: DpiX = %d DpiY = %d\n",pdm.logicalDpiX(),pdm.logicalDpiY());
        p->end();

        printMode=true;
        h=(int)((float)w*height_fact);
        resetSize (w,h);
		QPainter *save_painter = painter;
		painter = new QPainter (&printer);
        showIso();
	    painter->end();
        painter = save_painter;

        resetSize (old_isowidth,old_isoheight);
        printMode=false;
        showIso();
	}
}


bool isowin::savePicture (const QString filename, const QString fmt)
{
	return backpix->save (filename,fmt);
}


void isowin::mousePressEvent (QMouseEvent *e)
{
	int mouseX=e->x();
	int mouseY=e->y();

	if (e->button() == LeftButton)
	{
		if (e->state() == ShiftButton)
		{
			oldVertRotatePos=mouseY;
		}
		else
		{
			oldDragAngle=angle10(centerX,centerY,mouseX,mouseY);
			firstDragAngle=CalcDeg10(rotate);
		}
		
		if (e->state() == ControlButton)
		{
			centerFirstX=mouseX-centerX;
			centerFirstY=mouseY-centerY;
		}
	}
	
	if (e->button() == RightButton)
	{
		if (e->state() == ShiftButton)
		{
			oldZoomPos=mouseX;
		}
		else
		{
		 	kisomandel->getIsoMenu()->exec(QCursor::pos());
		}
	}
	
	if ( (e->button() == MidButton) )
	{
		centerFirstX=mouseX-centerX;
		centerFirstY=mouseY-centerY;
	}
}	


void isowin::mouseMoveEvent ( QMouseEvent *e)
{
	int mouseX=e->x();
	int mouseY=e->y();
	
	if (e->state() == LeftButton)
	{
		int a=angle10(centerX,centerY,mouseX,mouseY);
		setRotate (a-oldDragAngle+firstDragAngle);
		showIso();
		updatePrefsWin();
	}
	
	if (e->state() == (LeftButton | ShiftButton) )
	{
		setVertRotate ( (mouseY-oldVertRotatePos) * 3,true);
		oldVertRotatePos=mouseY;
		showIso();
		updatePrefsWin();
	}

	if ( (e->state() == (LeftButton | ControlButton)) || e->state() == MidButton )
	{
		centerX=mouseX-centerFirstX; centerRelX=(double)centerX/(double)isowidth;
		centerY=mouseY-centerFirstY; centerRelY=(double)centerY/(double)isoheight;
		calcDrawXY();
		showIso();
	}
		
	if (e->state() == (RightButton | ShiftButton) )
	{
		changeZoom (zoom + (float)(mouseX-oldZoomPos) / 120.0 );
		oldZoomPos=mouseX;
		showIso();
		updatePrefsWin();
	}
}


void isowin::resizeEvent (QResizeEvent *e)
{
	int w=e->size().width();
	int h=e->size().height();
	
	resetSize (w,h);
	showIso();
}
		

void isowin::activateButton (short id, bool activate, bool enable)
{
	QToolButton *b = kisomandel->getButton (id);
		
	if (b)
	{
		b->setToggleButton (true);
		b->setOn (activate & enable);
		b->setEnabled (enable);
	}
}


void isowin::updateButtons()
{
	activateButton (ID_BUTTON_LINES,dispMode==ISOMODE_LINES);
	activateButton (ID_BUTTON_COLORLINES,dispMode==ISOMODE_COLORLINES);
	activateButton (ID_BUTTON_AREAS,dispMode==ISOMODE_AREAS);
	activateButton (ID_BUTTON_BARS,dispMode==ISOMODE_BARS);

	bool f = !(dispMode==ISOMODE_LINES || dispMode==ISOMODE_COLORLINES);
	activateButton (ID_BUTTON_OUTLINE,areaOutline,f);
	activateButton (ID_BUTTON_SMOOTHING_LESS,false,fract->getSmoothLevel() > 0);
	activateButton (ID_BUTTON_SMOOTHING_MORE,false,fract->getSmoothLevel() < MAX_SMOOTHLEVEL);
	activateButton (ID_BUTTON_FULLSCREEN,fullScreen);
	
}


void isowin::openPrefsWindow()
{
	if (!prefsWin)	// already open?
	{
		// Open window at previous position
		prefsWin = new IsoPrefs ();
	}
	updateMenusAndButtons();
	prefsWin->show();
	prefsWin->raise();
}


void isowin::closePrefsWindow()
{
	if (prefsWin)
	{
		// remember window position
		prefsPosX=prefsWin->x();
		prefsPosY=prefsWin->y();
		prefsSize=prefsWin->size();
		delete prefsWin;
		prefsWin=NULL;
		puts ("isowin::prefsWin closed");
	}
}


void isowin::openGeneralSettingsWindow()
{
	if (!generalPrefsWin)	// already open?
	{
		generalPrefsWin = new GeneralPrefs();
	}
	generalPrefsWin->show();
	generalPrefsWin->raise();
}
	

void isowin::closeGeneralSettingsWindow()
{
	if (generalPrefsWin)
	{
		generalPrefsPosX=generalPrefsWin->x();
		generalPrefsPosY=generalPrefsWin->y();
		generalPrefsSize=generalPrefsWin->size();
		delete generalPrefsWin;
		generalPrefsWin=NULL;
	}
}


void isowin::setItemCheck (int id, bool flag)
{
	kisomandel->setItemCheck (id,flag);
}


void isowin::updateMenus()
{
	setItemCheck (ID_ISO_DISPLAY_LINES,dispMode==ISOMODE_LINES);
	setItemCheck (ID_ISO_DISPLAY_COLORLINES,dispMode==ISOMODE_COLORLINES);
	setItemCheck (ID_ISO_DISPLAY_AREAS,dispMode==ISOMODE_AREAS);
	setItemCheck (ID_ISO_DISPLAY_BARS,dispMode==ISOMODE_BARS);
	
	int l=fract->getSmoothLevel();
	for (int i=0;i<=10;i++) setItemCheck (i+ID_ISO_SMOOTHING_NONE,l==i);
	
	setItemCheck (ID_ISO_AREAOUTLINE,areaOutline);
	setItemCheck (ID_ISO_AUTOHEIGHT,autoHeight);
	setItemCheck (ID_ISO_HORLINES,showHorLines);
	setItemCheck (ID_ISO_VERLINES,showVerLines);
	setItemCheck (ID_ISO_FULLSCREEN,fullScreen);
	
	setItemCheck (ID_FRACT_FRACTTYPE_MANDEL,fract->getFractalType()==Fract_Mandel);
	setItemCheck (ID_FRACT_FRACTTYPE_JULIA,fract->getFractalType()==Fract_Julia);
}


void isowin::updatePrefsWin()
{
	if (prefsWin) prefsWin->updateValues();
}


void isowin::updateMenusAndButtons()
{
	updateButtons();
	updateMenus();
	updatePrefsWin();
}


void isowin::handleKeyCommand (int key, int state)
{	
	switch (tolower(key))
	{
		case Key_F1 :
			dispMode=ISOMODE_LINES;
			showIso();
			break;
		case Key_F2:
			dispMode=ISOMODE_COLORLINES;
			showIso();
			break;
		case Key_F3 :
			dispMode=ISOMODE_AREAS;
			showIso();
			break;
		case Key_F4 :
			dispMode=ISOMODE_BARS;
			showIso();
			break;
		case Key_F5 :
			setAutoHeight (!autoHeight);
			break;
		case Key_F6 :
			showHorLines=!showHorLines;
			if ( (dispMode==ISOMODE_LINES) || (dispMode==ISOMODE_COLORLINES) ) showIso();
			break;
		case Key_F7 :
			showVerLines=!showVerLines;
			if ( (dispMode==ISOMODE_LINES) || (dispMode==ISOMODE_COLORLINES) ) showIso();
			break;
		case Key_F11 :
			switchFullScreen();
			break;
		case Key_F12 :
			fract->openPrefsWindow();
			break;
		case '+' :
			changeIsoRes (resX+1,resY+1,true);
			showIso();
			break;
		case '-' :
			changeIsoRes (resX-1,resY-1,true);
			showIso();
			break;
		case 'r' :
		{
		  	int msec=0;
			QTime t;
			t.start();
			for (int i=0;i<90;i++)
			{
				setRotate (40,true);
				while ( (t.elapsed() - msec) <= 10);	// not more than 100 fps
				showIso();
				msec = t.elapsed();
			}
			break;
		}
		case '>' :
		{
			short sl=fract->getSmoothLevel();
			
			if (sl<MAX_SMOOTHLEVEL)
			{
				fract->setSmoothLevel (sl+1);
				fract->smoothData();
				showIso();
			}
			break;
		}
		case '<' :
		case '0' :
		{
			short sl=fract->getSmoothLevel();
			
			if (key=='0') sl=1;
			
			if (sl>0)
			{
				fract->setSmoothLevel (sl-1);
				fract->smoothData();
				showIso();
			}
			break;
		}
		case 'o' :
			areaOutline=!areaOutline;
			if ( (dispMode != ISOMODE_LINES) && (dispMode != ISOMODE_COLORLINES) ) showIso();
			break;
		case '4' :
			setRotate (3600/60,true);
			showIso();
			break;
		case '6' :
			setRotate (-3600/60,true);
			showIso();
			break;
		case '2' :
			setVertRotate (3600/60,true);
			showIso();
			break;
		case '8' :
			setVertRotate (-3600/60,true);
			showIso();
			break;
		case Key_Left :
			fract->scrollxy (fract->getWidth()/20,0);
			break;
		case Key_Right :
			fract->scrollxy (fract->getWidth()/-20,0);
			break;
		case Key_Up :
			fract->scrollxy (0,fract->getHeight()/20);
			break;
		case Key_Down :
			fract->scrollxy (0,fract->getHeight()/-20);
			break;
		case Key_PageUp :
			changeZoom (zoom*1.1);
			showIso();
			break;
		case Key_PageDown :
			changeZoom (zoom/1.1);
			showIso();
			break;
		case 'f' :
			fract->setPrecision (PRECISION_FLOAT,true);
			break;
		case 'd' :
			fract->setPrecision (PRECISION_DOUBLE,true);
			break;
		case 'l' :
			fract->setPrecision (PRECISION_LDOUBLE,true);
			break;
		case 'm' :
			fract->setFractalType (Fract_Mandel,true);
			break;
		case 'j' :
			fract->setFractalType (Fract_Julia,true);
			break;
	}	// switch
	updateMenusAndButtons();
}
				

void isowin::keyPressEvent ( QKeyEvent *e)
{
	handleKeyCommand ( e->key(),e->state() );
}


void isowin::resetSize (int width, int height, bool resizeWin)
{

	isowidth = (width > WIN_MINWIDTH) ? width : WIN_MINWIDTH;
	isoheight = (height > WIN_MINHEIGHT) ? height : WIN_MINHEIGHT;

	if (old_isowidth==isowidth && old_isoheight==isoheight) return;
	old_isowidth=isowidth;
	old_isoheight=isoheight;	
	
	if (!printMode)
	{
		createBackPixmap();
		if (resizeWin) setGeometry (x(),y(),isowidth,isoheight);
	}
	init();
}	


void isowin::init()
{
	centerX=(int) (((double)isowidth)*centerRelX);
	centerY=(int) (((double)isoheight)*centerRelY);
	changeIsoRes (resX,resY);
	oldBlitMinX=0;
	oldBlitMinY=0;
	oldBlitMaxX=isowidth-1;
	oldBlitMaxY=isoheight-1;
}


int isowin::validRes (int res)
{

	if (res<MIN_ISORES) res=MIN_ISORES;
	if (res>MAX_ISORES) res=MAX_ISORES;
	if (fract)
	{
		int fw=fract->getWidth();
		int fh=fract->getHeight();
	
		if (res>fw) res=fw;
		if (res>fh) res=fh;
	}
	return res;
}


void isowin::changeIsoRes (int xres, int yres, bool center)
{
	short cx = startX + resX/2;
	short cy = startY + resY/2;
	
	resX=validRes(xres);
	resY=validRes(yres);
	if (center) setIsoOffset (cx-resX/2,cy-resY/2);
	calcIsoSize();
	calcDrawXY();
	showIsoRange();
}



void isowin::calcIsoSize()	// berechnet laut IsoResX,-Y die Groesse der Quadrate
{
	sizeX=isowidth/(resX+1)*2/3;
	if (sizeX<2) sizeX=2;
	if (sizeX>100) sizeX=100;
	
	sizeY=sizeX*isoheight/isowidth;
	if (sizeY<2) sizeY=2;
	if (sizeY>100) sizeY=100;
}


void isowin::calcDrawXY()
{
	short x,y;
	double cx,cy;
	double ax=cos(rotate)*sizeX;
	double ay=sin(rotate)*sizeX;
	double ax2=ax*sin(vert_rotate);
	double ay2=ay*sin(vert_rotate);
		
	cy=-resY/2;
	for (y=0;y<resY;y++)
	{
		double cy_m_ay = cy*ay;
		double cy_m_ax2 = cy*ax2;
		
		cx=-resX/2;
		for (x=0;x<resX;x++)
		{
			drawX[y][x]=centerX + (short) ((cx*ax+cy_m_ay)*zoom);
			drawY[y][x]=centerY + (short) ((cy_m_ax2-cx*ay2)*zoom);
			cx++;
		}
		cy++;
	}
}


void isowin::setZoom (float z)
{
	if (z<MIN_ISOZOOM) z=MIN_ISOZOOM;
	if (z>MAX_ISOZOOM) z=MAX_ISOZOOM;
	zoom=z;
}


void isowin::setLineWidth (short w)
{
	if (w<0) w=0;
	if (w>4) w=4;
	lineWidth=w;
	if (dispMode==ISOMODE_LINES || dispMode==ISOMODE_COLORLINES) showIso();
}


void isowin::setLineColor (QColor *col)
{
	linecolor = *col;
	if (dispMode==ISOMODE_LINES || dispMode==ISOMODE_COLORLINES) showIso();
	updatePrefsWin();
}


void isowin::setOutlineColor (QColor *col)
{
	outlinecolor = *col;
	if (dispMode==ISOMODE_AREAS || dispMode==ISOMODE_BARS) showIso();
	updatePrefsWin();
}
	

QColor * isowin::getMaxiterColor()
{
	return colorpalette[AREA_BUTTOM];
}


void isowin::setMaxiterColor (QColor *col)
{
	createPal (AREA_BUTTOM,col->red(),col->green(),col->blue());
	if (dispMode==ISOMODE_AREAS || dispMode==ISOMODE_BARS) showIso();
	updatePrefsWin();
}


void isowin::setHeightFactor (float f)
{
	if (f<0.05) f=0.05;
	if (f>2.0) f=2.0;
	
	heightFactor=f;
	showIso();
}


void isowin::setAutoHeight (bool f)
{
	if (f != autoHeight)
	{
		autoHeight=f;
		showIso();
		updatePrefsWin();
	}
}


void isowin::changeZoom (float z)
{
	setZoom (z);
	calcDrawXY();
}


void isowin::setRotate (int grad, bool relative)
{
	int g=validRotate ( (relative) ? CalcDeg10(rotate)+grad : grad);
	rotate=PI2/3600*(float)g;
	calcDrawXY();
}


void isowin::setVertRotate (int grad, bool relative)
{
	int g=validRotate ( (relative) ? CalcDeg10(vert_rotate)+grad : grad);
	vert_rotate=PI2/3600*(float)g;
	calcDrawXY();
}


void isowin::setIsoOffset (short x, short y, bool center)
{
	if (!fract) return;
	
	short ex=fract->getWidth()-resX;
	short ey=fract->getHeight()-resY;

	if (center)
	{
		x-=resX/2;
		y-=resY/2;
	}
	if (x>ex) x=ex;
	if (x<0) x=0;
	if (y>ey) y=ey;
	if (y<0) y=0;
	startX=x;
	startY=y;
	showIsoRange();
}


void isowin::setIsoOffset (bool center)
{
	if (center)
	{
		startX = (fract->getWidth() - resX) / 2;
		startY = (fract->getHeight() - resY) / 2;
	}
	setIsoOffset (startX,startY);
}


void isowin::clearIso()
{
	if (!printMode)
	{
		painter->fillRect (
			max(oldBlitMinX,0),
			max(oldBlitMinY,0),
			min(oldBlitMaxX-oldBlitMinX+1,width()),
			min(oldBlitMaxY-oldBlitMinY+1,height()),
			black);
	}
	
}


void isowin::blitIso()
{
	short x1,y1,x2,y2;

	if (!printMode)
	{
    	blitMinX-=lineWidth;
    	blitMaxX+=lineWidth;
    	blitMinY-=lineWidth;
    	blitMaxY+=lineWidth;
    	
    	x1=min ( max(0,blitMinX), oldBlitMinX );
    	y1=min ( max(0,blitMinY), oldBlitMinY );
    	x2=max ( min(isowidth-1,blitMaxX), oldBlitMaxX );
    	y2=max ( min(isoheight-1,blitMaxY), oldBlitMaxY );

    	bitBlt (this,x1,y1,backpix,x1,y1,x2-x1+1,y2-y1+1,CopyROP,true);

    	oldBlitMinX=blitMinX;
    	oldBlitMinY=blitMinY;
    	oldBlitMaxX=blitMaxX;
    	oldBlitMaxY=blitMaxY;
    }
}


void isowin::initBlitRange()
{
	blitMinX=isowidth-1;
	blitMinY=isoheight-1;
	blitMaxX=0;
	blitMaxY=0;
}


void inline isowin::adjustBlitRange (short x, short y)
{
	if (x<blitMinX) blitMinX=x;
	if (x>blitMaxX) blitMaxX=x;
	if (y<blitMinY) blitMinY=y;
	if (y>blitMaxY) blitMaxY=y;
}


void isowin::showIso()
{
	if (!fract) return;	// fractal class not yet initialized
		
	initBlitRange();
	maxiter=fract->getMaxIter();
	
	switch (dispMode)
	{
			case ISOMODE_LINES		: showIsoLines(false) ; break;
			case ISOMODE_COLORLINES	: showIsoLines(true)  ; break;
			case ISOMODE_AREAS		: showIsoAreas()	  ; break;
			case ISOMODE_BARS 		: showIsoBars()		  ; break;
	}
	QApplication::syncX();
}


void isowin::switchFullScreen()
{
	fullScreen = !fullScreen;
	if (fullScreen)
	{
		kisomandel->showFullScreen();
		// kisomandel->menuBar()->hide();
	}
	else
	{
		kisomandel->showNormal();
		// kisomandel->menuBar()->show();
	}
	
	updateMenusAndButtons();
}


short inline isowin::getFractDat (short x, short y)
{
	return fract->getDataSave (x+startX,y+startY);
}


short inline isowin::getIsoDat (short x, short y)
{
	return fract->getIsoDataSave (x+startX,y+startY);
}


void isowin::getIsoMinMax()
{
	short x,y,i;
	
	if (!autoHeight)
	{
		isomin=0;
		isomax=fract->getMaxIter();
	}
	else
	{
		isomin=512;
		isomax=0;
		for (y=0;y<resY;y++)
		{
			for (x=0;x<resX;x++)
			{
				i=getIsoDat (x,y);
				if (i<isomin) isomin=i;
				if (i>isomax) isomax=i;
			}
		}
	}
	isomax-=isomin;
	if (isomax<1) isomax=1;
}


float isowin::calcDrawHeight()
{
	float kw=sin(vert_rotate);

	getIsoMinMax();
	if (kw>=0) kw=(1-kw); else kw=-1-kw;
	return fabs(isoheight*zoom*heightFactor*kw/isomax);
}


void isowin::showIsoLines (bool colormode)
{
	short x,y,x1,y1;
	short i,oi=0,ox1=0,oy1=0;
	short endx=resX-1;
	short endy=resY-1;
	short ystart=drawYStart;
	
	clearIso();
	painter->setPen (QPen(linecolor,lineWidth));

	drawHeight=calcDrawHeight();
	
	if (showHorLines)
	{
		for (y=0;y<=endy;y++)
		{
			for (x=0;x<=endx;x++)
			{
				i=getIsoDat(x,y)-isomin;
				x1=drawX[y][x];
				y1=ystart+drawY[y][x] - short(i*drawHeight);
				
				if (x==0)
				{
					painter->moveTo (x1,y1); adjustBlitRange (x1,y1);
					oi=i;
				}
				else
				if (i!=oi || x==endx || colormode)
				{
					if (colormode)
					{
						QColor *col = colorpalette[getAreaColor(getFractDat(x,y))];
						painter->setPen(QPen(*col,lineWidth));
					}
					painter->lineTo (ox1,oy1); adjustBlitRange (ox1,oy1);
					if (x1!=ox1 || y1!=oy1) painter->lineTo (x1,y1); adjustBlitRange (x1,y1);
				}
				oi=i; ox1=x1; oy1=y1;
			
			}	// for x
		}	// for y
	}	// HORLINES

	if (showVerLines)
	{
		for (x=0;x<=endx;x++)
		{
			for (y=0;y<=endy;y++)
			{
				i=getIsoDat(x,y)-isomin;
				x1=drawX[y][x];
				y1=ystart+drawY[y][x]-(short)(i*drawHeight);

				if (y==0)
				{
					painter->moveTo (x1,y1); adjustBlitRange (x1,y1);
					oi=i;
				}
				else
				if (i!=oi || y==endy || colormode)
				{
					if (colormode)
					{
						QColor *col = colorpalette[getAreaColor(getFractDat(x,y))];
						painter->setPen(QPen(*col,lineWidth));
					}
					painter->lineTo (ox1,oy1); adjustBlitRange (ox1,oy1);
					if (x1!=ox1 || y1!=oy1) painter->lineTo (x1,y1); adjustBlitRange (x1,y1);
				}
				oi=i; ox1=x1; oy1=y1;
			}	// for y
		}	// for x
	}	// VERLINES
	blitIso();
}



short inline isowin::calcCoord (QPointArray &p, short i, short x, short y, short ax, short ay)
{
	short col=getFractDat(x+ax,y+ay);
	short h=getIsoDat(x+ax,y+ay);
	short areaX,areaY;
	
	areaX=drawX[y+ay][x+ax];
	areaY=drawY[y+ay][x+ax]-(short)((h-isomin)*drawHeight)+drawYStart;
	p.setPoint (i,areaX,areaY);
	adjustBlitRange (areaX,areaY);
	return col;
}


short isowin::getAreaColor (short isodat)
{
	if (isodat>=maxiter)
		return AREA_BUTTOM;
	else
		return isodat;
}	


short isowin::calcQuadrant()
{
	return ((short)(rotate*180/PI))/90;
}


void isowin::showIsoAreas()
{
	QPointArray xy(4);
	short x,y;
	short startx,endx,stepx,stepy,starty,endy;
	short yy=0;
	short iter,old_iter;
	
	clearIso();
	drawHeight=calcDrawHeight();

	switch (calcQuadrant())
	{
		case 0 : // 0 - 89 Grad
			starty=0; endy=resY-2;
			startx=resX-2; endx=0;
			break;
		case 1 :	// 90 - 179 Grad
			starty=resY-2; endy=0;
			startx=resX-2; endx=0;
			break;
		case 2 :	// 180 - 269 Grad
			starty=resY-2; endy=0;
			startx=0; endx=resX-2;
			break;
		case 3 :	// 270 - 359 Grad
		default:
			starty=0; endy=resY-2;
			startx=0; endx=resX-2;
			break;
	}
	
	stepx= (endx>startx) ? 1 : -1;
	stepy= (endy>starty) ? 1 : -1;
	
	outlinepen=(areaOutline) ? QPen(outlinecolor,0) : QPen(NoPen);
	painter->setPen (outlinepen);
	old_iter=0xffff;
	
	for (y=starty;y!=endy+stepy;y+=stepy)
	{
		yy++;
		for (x=startx;x!=endx+stepx;x+=stepx)
		{
			iter=calcCoord (xy,0,x,y,0,0);
			calcCoord (xy,1,x,y,1,0);
			calcCoord (xy,2,x,y,1,1);
			calcCoord (xy,3,x,y,0,1);

			if (iter != old_iter)
			{
				painter->setBrush (*colorpalette[getAreaColor(iter)]);
				old_iter=iter;
			}
			painter->drawPolygon (xy,false,0,4);
		}	// for x
	}	// for y

	
	blitIso();

}	/* ShowIsoAreas */



void inline isowin::fillBar (QPointArray &p, QColor *col)
{
	painter->setBrush (*col);
	painter->drawPolygon (p,false,0,4);
}


void inline isowin::calcBarCoord (QPointArray &p, short i, short x, short y, short ax, short ay, short barY)
{
	short areaX,areaY;
	
	areaX=drawX[y+ay][x+ax];
	areaY=drawY[y+ay][x+ax]+drawYStart - (short) (float)( (barY-isomin)*drawHeight);
	p.setPoint (i,areaX,areaY);
	adjustBlitRange (areaX,areaY);
}


void isowin::showIsoBars()
{
	QPointArray xy(4);
	short col,barCol;
	short quadrant=calcQuadrant();
	short x,y,startx,endx,stepx,starty,endy,stepy;
	short yy=0;
	bool view_from_top=vert_rotate<=PI;

	clearIso();
	
	outlinepen=(areaOutline) ? QPen(outlinecolor,0) : QPen(NoPen);
	painter->setPen (outlinepen);
	
	drawHeight=calcDrawHeight();

	stepy=bdi[quadrant].stepy;
	stepx=bdi[quadrant].stepx;

	switch (stepy)
	{
		case 1  : starty=0; endy=resY-2; break;
		case -1 :
		default : starty=resY-2; endy=0; break;
	}
	switch (stepx)
	{
		case 1  : startx=0; endx=resX-2; break;
		case -1 :
		default : startx=resX-2; endx=0; break;
	}
	
	for (y=starty;y!=endy+stepy;y+=stepy)
	{
		short next_barCol,next_barCol2;
		
		yy++;
		for (x=startx;x!=endx+stepx;x+=stepx)
		{
			barCol=getIsoDat(x,y);
			col=getAreaColor(getFractDat(x,y));
			
			if (x==endx || y==endy)
			{
				next_barCol=isomin;
				next_barCol2=isomin;
			}
			else
			{
				next_barCol=getIsoDat(x+stepx,y);
				next_barCol2=getIsoDat(x,y+stepy);
			}
			
			if (next_barCol < barCol || x==endx)
			{
				calcBarCoord (xy,0,x,y,bdi[quadrant].barx1,bdi[quadrant].bary1,next_barCol);
				calcBarCoord (xy,1,x,y,bdi[quadrant].barx1,bdi[quadrant].bary1,barCol);
				calcBarCoord (xy,2,x,y,bdi[quadrant].barx2,bdi[quadrant].bary2,barCol);
				calcBarCoord (xy,3,x,y,bdi[quadrant].barx2,bdi[quadrant].bary2,next_barCol);
				fillBar (xy,colorpalette_bright[col]);
			}
	
			if (next_barCol2 < barCol || y==endy)
			{
				calcBarCoord (xy,0,x,y,bdi[quadrant].barx2,bdi[quadrant].bary2,next_barCol2);
				calcBarCoord (xy,1,x,y,bdi[quadrant].barx2,bdi[quadrant].bary2,barCol);
				calcBarCoord (xy,2,x,y,bdi[quadrant].barx3,bdi[quadrant].bary3,barCol);
				calcBarCoord (xy,3,x,y,bdi[quadrant].barx3,bdi[quadrant].bary3,next_barCol2);
				fillBar (xy,colorpalette_dark[col]);
			}
			
			// Deckel
			short barY;
			
			if (view_from_top) barY=barCol; else barY=isomin;
			calcBarCoord (xy,0,x,y,bdi[quadrant].barx1,bdi[quadrant].bary1,barY);
			calcBarCoord (xy,1,x,y,bdi[quadrant].barx2,bdi[quadrant].bary2,barY);
			calcBarCoord (xy,2,x,y,bdi[quadrant].barx3,bdi[quadrant].bary3,barY);
			calcBarCoord (xy,3,x,y,bdi[quadrant].barx4,bdi[quadrant].bary4,barY);
			fillBar (xy,colorpalette[col]);
		
		}	// for x
	
	}	// for y
	
	blitIso();
}


void isowin::drawIsoRangeRect()
{
  	if (fract)
  	{
  		QPainter *p = fract->getPainter();
  		RasterOp saverop = p->rasterOp();
	  	p->setRasterOp (NotROP);
	  	p->drawRect (isoRangeX,isoRangeY,isoRangeWidth,isoRangeHeight);
	  	p->setRasterOp (saverop);

	  	
	  	if (fract->blitToWinEnabled())
    	{
    		fract->blitRectangle (isoRangeX,isoRangeY,isoRangeX+isoRangeWidth-1,isoRangeY+isoRangeHeight-1);
/*    	  	QPixmap *pm = fract->getBackPixmap();
    	  	// line (x1,y1,x2,y1)
    	  	bitBlt (fract,isoRangeX,isoRangeY,pm,isoRangeX,isoRangeY,isoRangeWidth,1);
    	  	// line (x1,y2,x2,y2)
    	  	bitBlt (fract,isoRangeX,isoRangeY+isoRangeHeight-1,pm,isoRangeX,isoRangeY+isoRangeHeight-1,isoRangeWidth,1);
    	  	// line (x1,y1,x1,y2)
    	  	bitBlt (fract,isoRangeX,isoRangeY,pm,isoRangeX,isoRangeY,1,isoRangeHeight);
    	  	// line (x2,y1,x2,y2)
    	  	bitBlt (fract,isoRangeX+isoRangeWidth-1,isoRangeY,pm,isoRangeX+isoRangeWidth-1,isoRangeY,1,isoRangeHeight);
*/    	
    	}
	 }
}



void isowin::hideIsoRange()
{
	if (isoRangeDrawn)
	{
		drawIsoRangeRect();
		isoRangeDrawn=false;
	}
}


void isowin::showIsoRange (bool checkhide)
{
	if (isoRangeDrawn && checkhide) hideIsoRange();
  	isoRangeX=startX;
  	isoRangeY=startY;
  	isoRangeWidth=resX;
  	isoRangeHeight=resY;
  	drawIsoRangeRect();
  	isoRangeDrawn=true;
}
