/***************************************************************************
                          settings.cpp  -  description
                             -------------------
    begin                : Mon Aug 14 2000
    copyright            : (C) 2000 by Helmut Steger
    email                : hel.steger@rolmail.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qtooltip.h>
#include "kisomandel.h"
#include "settings.h"



void setButtonColor (QPushButton *b, QColor *c)
{
	b->setBackgroundMode (QWidget::PaletteBase);
	b->setPalette (QPalette(*c));
}


void makeColorString (QColor *col, char *s)
{
	sprintf (s,"%02x%02x%02x",col->red(),col->green(),col->blue());
}


bool setColorFromString (QColor &col, char *s)
{
   	if (strlen(s)>=6)
   	{
   		int r,g,b;
   		if (sscanf (s,"%02x%02x%02x",&r,&g,&b) == 3)
   		{
   			col=QColor (r,g,b);
   			return true;
   		}
   	}
   	return false;
}
	


///////////////////// DoubleNumEdit /////////////////////

DoubleNumEdit::DoubleNumEdit (QWidget *parent = 0, const char* name = 0, WFlags fl = 0 ) : QLineEdit (parent,name)
{
	setValidator (new QDoubleValidator(this) );
}


long double DoubleNumEdit::getNumber()
{
	long double n;
	QString s = text();
	
	if (s.length()>0 && s != "-" && s != ".")
		sscanf ( (const char *)s,"%llf",&n);
	else
		n=0.0;
		
	return n;
}


void DoubleNumEdit::setNumber (long double num, unsigned short maxdecimals)
{
	QString s;
	fnum2str (num,s,maxdecimals);
	setText (s);
	setSelection (0,0);
	home (true);
}


void DoubleNumEdit::keyPressEvent (QKeyEvent *e)
{
	if (e->key()==Key_Return || e->key()==Key_Enter)
	{
		emit NumberChanged();	// send signal
		setEdited (false);
	}
	QLineEdit::keyPressEvent (e);
}


void DoubleNumEdit::focusOutEvent (QFocusEvent *e)
{
	QLineEdit::focusOutEvent (e);
	setSelection (0,0);
	home (true);
	if (edited())
	{
		emit NumberChanged();	// send signal
		setEdited (false);
	}
}

		
///////////////////// IsoPrefsForm /////////////////////

IsoPrefs::IsoPrefs (QWidget* parent = 0, const char* name = 0, WFlags fl = 0 ) : IsoPrefsForm (parent,name)
{
	settingUp=false;
	
	if (mainwin->prefsPosX == -1 || mainwin->prefsPosY == -1)
	{
		mainwin->prefsPosX=(desktopWidth()-width())/2;
		mainwin->prefsPosY=(desktopHeight()-height())/2;
	}
	setGeometry (
		mainwin->prefsPosX,
		mainwin->prefsPosY,
		mainwin->prefsSize.width(),
		mainwin->prefsSize.height()
		);
}

		
void IsoPrefs::updateValues()
{
	settingUp=true;
	Slider_IsoRes->setValue (mainwin->resX-1);

	Slider_Zoom->blockSignals (true);
	int z=(int)(mainwin->zoom*100.0);
	Slider_Zoom->setValue (z);
	SpinBox_Zoom->setValue (z);
	Slider_Zoom->blockSignals (false);
	
	Slider_IsoHeight->setValue ((int)(mainwin->heightFactor*100.0));
	Check_AutoHeight->setChecked (mainwin->autoHeight);
	
	setButtonColor (LineColorButton,&mainwin->linecolor);
	setButtonColor (OutlineColorButton,&mainwin->outlinecolor);
	setButtonColor (MaxiterColorButton,mainwin->getMaxiterColor());
	
	ComboBox_LineWidth->setCurrentItem (mainwin->lineWidth);
	
	Dial_RotateX->setValue (359-CalcDeg10(mainwin->rotate)/10);
	Dial_RotateY->setValue (359-CalcDeg10(mainwin->vert_rotate)/10);	
	settingUp=false;
}


void IsoPrefs::keyPressEvent (QKeyEvent *e)
{
	if (e->key() == Key_Escape)
		slotCloseClicked();
	else
		mainwin->handleKeyCommand ( e->key(),e->state() );
}


void IsoPrefs::slotIsoHeightChanged()
{
	short h=Slider_IsoHeight->value();

	if (!settingUp)
	{
		mainwin->setHeightFactor ( ((float)h) / 100.0);
	}
}


void IsoPrefs::slotAutoHeightChanged()
{
	if (settingUp) return;
	mainwin->setAutoHeight (Check_AutoHeight->isChecked());
}


void IsoPrefs::slotIsoResChanged()
{
	short i=Slider_IsoRes->value();
	
	QString s;
	s.sprintf ("Resolution (%d x %d)",i,i);
	GroupBox_Resolution->setTitle (s);
	
	if (!settingUp)
	{
		mainwin->changeIsoRes (i+1,i+1,true);
		mainwin->showIso();
	}
}


void IsoPrefs::slotIsoZoomChanged()
{
	short i=Slider_Zoom->value();
	float z=((float)i)/100.0;

	if (!settingUp);
	{
		mainwin->changeZoom (z);
		mainwin->showIso();
	}
}


void IsoPrefs::slotLineWidth()
{
	if (settingUp) return;
	mainwin->setLineWidth (ComboBox_LineWidth->currentItem());
}


void IsoPrefs::slotLineColor()
{
	if (settingUp) return;
	QColor col = QColorDialog::getColor (mainwin->linecolor);
	if (col.isValid())
	{
		setButtonColor (LineColorButton,&col);
		mainwin->setLineColor (&col);
	}
}


void IsoPrefs::slotOutlineColor()
{
	if (settingUp) return;
	QColor col = QColorDialog::getColor (mainwin->outlinecolor);
	if (col.isValid())
	{
		setButtonColor (OutlineColorButton,&col);
		mainwin->setOutlineColor (&col);
	}
}


void IsoPrefs::slotMaxiterColorClicked()
{
	if (settingUp) return;
	QColor col = QColorDialog::getColor (*mainwin->getMaxiterColor());
	if (col.isValid())
	{
		setButtonColor (MaxiterColorButton,&col);
		mainwin->setMaxiterColor (&col);
	}
}


void IsoPrefs::slotHorRotateChanged()
{
	if (settingUp) return;
	mainwin->setRotate ((float)(359-Dial_RotateX->value())*10);
	mainwin->showIso();
}


void IsoPrefs::slotVerRotateChanged()
{
	if (settingUp) return;
	mainwin->setVertRotate ((float)(359-Dial_RotateY->value())*10);
	mainwin->showIso();
}


void IsoPrefs::slotCloseClicked()
{
	mainwin->closePrefsWindow();
}


void IsoPrefs::closeEvent (QCloseEvent *e)
{
	slotCloseClicked();
}



///////////////////// FractPrefsForm /////////////////////


FractPrefs::FractPrefs (QWidget* parent = 0, const char* name = 0, WFlags fl = 0 ) : FractPrefsForm (parent,name)
{
	if (fract->prefsPosX == -1 || fract->prefsPosY == -1)
	{
		fract->prefsPosX=(desktopWidth()-width())/2;
		fract->prefsPosY=(desktopHeight()-height())/2;
	}
	setGeometry (
		fract->prefsPosX,
		fract->prefsPosY,
		fract->prefsSize.width(),
		fract->prefsSize.height());
		
	SpinBox_MaxIter->setRange (MIN_ITER,MAX_ITER);
   	fract->juliaConstInteractiveMode=false;
    QToolTip::add (PushButton_SetJuliaConstants, tr(IDS_JULIACONST_BUTTON_TOOLTIP) );

	//TextLabel_JuliaInfo->setText(IDS_JULIACONST_ACTIVATE_BUTTON);
	julia_rconst=fract->julia_rconst;
	julia_iconst=fract->julia_iconst;
	updateValues();
	checkValues();

	julia_preview = new JuliaPreview (PixmapLabel_JuliaPreview);
	plotJuliaPreview();
}


FractPrefs::~FractPrefs()
{
	if (julia_preview) delete julia_preview;
   	fract->juliaConstInteractiveMode=false;
}


void FractPrefs::updateValues()
{
	LineEdit_rmin->setNumber (fract->rmin);
	LineEdit_rmax->setNumber (fract->rmax);
	LineEdit_imin->setNumber (fract->imin);
	LineEdit_imax->setNumber (fract->imax);
	
	LineEdit_rwidth->setNumber (fract->rmax - fract->rmin);
	LineEdit_iheight->setNumber (fract->imax - fract->imin);
	
	LineEdit_JuliaRconst->setNumber (julia_rconst);
	LineEdit_JuliaIconst->setNumber (julia_iconst);
	
	SpinBox_MaxIter->setValue (fract->maxiter);
	ComboBox_ZoomInOut->setCurrentItem (fract->zoomInOutFactor - 2);
	SpinBox_ZoomInOutIterChange->setValue (fract->zoomInOutIterChange);
}


bool FractPrefs::checkValues()
{
	QString msg="Status: ok";
	bool valid=true;
	
	rmin=LineEdit_rmin->getNumber();
	rmax=LineEdit_rmax->getNumber();
	imin=LineEdit_imin->getNumber();
	imax=LineEdit_imax->getNumber();
	rwidth=LineEdit_rwidth->getNumber();
	iheight=LineEdit_iheight->getNumber();

	try
	{
		bool real_bad = (rmin>=rmax);
		bool imag_bad = (imin>=imax);
		
		if (real_bad && imag_bad) throw ("rmax/imax are less or equal than rmin/imin");
		if (real_bad) throw ("rmax is less or equal than rmin");
		if (imag_bad) throw ("imax is less or equal than imin");
		if (rwidth<=0.0) throw ("real width is less or equal 0.0");
		if (iheight<=0.0) throw ("imag. height is less or equal 0.0");
	}
	catch (const char *err)
	{
		msg=QString("Error: ")+err;
		valid=false;
	}
	
	TextLabel_Status->setText (msg);	
	PushButton_Apply->setEnabled(valid);
	return valid;
}


void FractPrefs::keyPressEvent (QKeyEvent *e)
{
	if (e->key() == Key_Escape)
		fract->closePrefsWindow();
	else
		mainwin->handleKeyCommand ( e->key(),e->state() );
}


void FractPrefs::closeEvent (QCloseEvent *e)
{
	fract->closePrefsWindow();
}


void FractPrefs::slotApplyClicked()
{
	if (checkValues())
	{
		fract->setCoords (rmin,rmax,imin,imax);
		fract->julia_rconst=julia_rconst;
		fract->julia_iconst=julia_iconst;
		int iter = SpinBox_MaxIter->value();
		
		if (iter != fract->getMaxIter()) fract->changeIterations (iter);
		plotJuliaPreview();
		fract->plotMandel();
	}
	else
	{
		QMessageBox::warning (this,"Fractal coordinates",
			"Bad coordinates. Please correct first!",
			QMessageBox::Ok,0);
	}
}


void FractPrefs::slotCloseClicked()
{
	fract->closePrefsWindow();
}


void FractPrefs::slotRminChanged()
{
	rmin=LineEdit_rmin->getNumber();
	rmax=LineEdit_rmax->getNumber();
	
	LineEdit_rwidth->setNumber(rmax-rmin);
	checkValues();
}


void FractPrefs::slotRmaxChanged()
{
	slotRminChanged();
}


void FractPrefs::slotIminChanged()
{
	imin=LineEdit_imin->getNumber();
	imax=LineEdit_imax->getNumber();
	
	LineEdit_iheight->setNumber(imax-imin);
	checkValues();
}


void FractPrefs::slotImaxChanged()
{
	slotIminChanged();
}


void FractPrefs::slotRWidthChanged()
{
	long double n=LineEdit_rwidth->getNumber();
	
	LineEdit_rmax->setNumber (LineEdit_rmin->getNumber()+n);	// rmax = rmin + rwidth
	checkValues();
}
	
	
void FractPrefs::slotIHeightChanged()
{
	long double n=LineEdit_iheight->getNumber();
	
	LineEdit_imax->setNumber (LineEdit_imin->getNumber()+n);	// imax = imin + iheight
	checkValues();
}


void FractPrefs::slotJuliaRconstChanged()
{
	julia_rconst=LineEdit_JuliaRconst->getNumber();
	plotJuliaPreview();
}		


void FractPrefs::slotJuliaIconstChanged()
{
	julia_iconst=LineEdit_JuliaIconst->getNumber();
	plotJuliaPreview();
}


void FractPrefs::slotJuliaConstantsToggle()
{
	// int w=width();
	// int h=height();
	
	fract->juliaConstInteractiveMode=PushButton_SetJuliaConstants->isOn();
	if (fract->juliaConstInteractiveMode)
	{
		//TextLabel_JuliaInfo->setText(IDS_JULIACONST_INTERACTIVE_MODE);
		fract->finishRangeMode();
	}
	else
	{
		//TextLabel_JuliaInfo->setText(IDS_JULIACONST_ACTIVATE_BUTTON);
	}

//	QSize s=sizeHint();
//	resize (min(w,s.width()),min(h,s.height()));
}


void FractPrefs::plotJuliaPreview()
{
	if (julia_preview)
	{
		julia_preview->setJuliaConst (julia_rconst,julia_iconst);
		julia_preview->plot();
	}
}


void FractPrefs::plotJuliaPreview (long double rconst, long double iconst)
{
	julia_rconst=rconst;
	julia_iconst=iconst;
	plotJuliaPreview();
}


void FractPrefs::slotZoomInOutFactorChanged()
{
	short z=ComboBox_ZoomInOut->currentItem()+2;
	
	fract->setZoomInOutFactor (z);
}


void FractPrefs::slotZoomInOutIterChanged()
{
	short i=SpinBox_ZoomInOutIterChange->value();
	
	fract->setZoomInOutIterChange (i);
}



///////////////////// GeneralPrefsForm /////////////////////

GeneralPrefs::GeneralPrefs (QWidget* parent = 0, const char* name = 0, WFlags fl = 0 )
{
	if (mainwin->generalPrefsPosX == -1 || mainwin->generalPrefsPosY == -1)
	{
		mainwin->generalPrefsPosX=(desktopWidth()-width())/2;
		mainwin->generalPrefsPosY=(desktopHeight()-height())/2;
	}
	setGeometry (
		mainwin->generalPrefsPosX,
		mainwin->generalPrefsPosY,
		mainwin->generalPrefsSize.width(),
		mainwin->generalPrefsSize.height()
		);
	
	initValues();

}


GeneralPrefs::~GeneralPrefs()
{
}


void GeneralPrefs::initValues()
{
	// Iso settings

	initIsoValues();

	// Other settings
		
	QDir d1 (kisomandel->globalPrefs.paletteDir);
	LineEdit_PaletteDir->setText (d1.path());
	
	LineEdit_PaletteAtStart->setText (kisomandel->globalPrefs.paletteFileAtStart);
	CheckBox_LoadPaletteAtStart->setChecked (kisomandel->globalPrefs.loadPaletteAtStart);
	
	QDir d2 (kisomandel->globalPrefs.fractPrefsDir);
	LineEdit_FractPrefsDir->setText (d2.path());
	
	short i=0;
	if (kisomandel->globalPrefs.fractalType==Fract_Julia) i=1;
	ComboBox_FractType->setCurrentItem (i);

	CheckBox_ShowStatusBar->setChecked (kisomandel->globalPrefs.showStatusBar);
	CheckBox_ShowToolBar->setChecked (kisomandel->globalPrefs.showToolBar);
}


void GeneralPrefs::initIsoValues()
{
	Slider_IsoRes->setValue (kisomandel->globalPrefs.isoResolution);
	Slider_IsoHeight->setValue (kisomandel->globalPrefs.isoHeight);
	CheckBox_AutoHeight->setChecked (kisomandel->globalPrefs.autoHeight);
	Slider_IsoZoom->setValue (kisomandel->globalPrefs.isoZoom);
	setButtonColor (PushButton_LineColor,&kisomandel->globalPrefs.lineColor);
	setButtonColor (PushButton_OutlineColor,&kisomandel->globalPrefs.outlineColor);
	setButtonColor (PushButton_MaxiterColor,&kisomandel->globalPrefs.maxiterColor);
	CheckBox_Outline->setChecked (kisomandel->globalPrefs.areaOutline);
	ComboBox_Smoothing->setCurrentItem (kisomandel->globalPrefs.isoSmoothing);
}


void GeneralPrefs::slotSelectLineColor()
{
	QColor col = QColorDialog::getColor (kisomandel->globalPrefs.lineColor);
	if (col.isValid())
	{
		kisomandel->globalPrefs.lineColor = col;
		setButtonColor (PushButton_LineColor,&col);
	}
}


void GeneralPrefs::slotSelectOutlineColor()
{
	QColor col = QColorDialog::getColor (kisomandel->globalPrefs.outlineColor);
	if (col.isValid())
	{
		kisomandel->globalPrefs.outlineColor = col;
		setButtonColor (PushButton_OutlineColor,&col);
	}
}


void GeneralPrefs::slotSelectMaxiterColor()
{
	QColor col = QColorDialog::getColor (kisomandel->globalPrefs.maxiterColor);
	if (col.isValid())
	{
		kisomandel->globalPrefs.maxiterColor = col;
		setButtonColor (PushButton_MaxiterColor,&col);
	}
}




void GeneralPrefs::slotSelectFractPrefsDir()
{
	QString s = QFileDialog::getExistingDirectory(
		LineEdit_FractPrefsDir->text(),
		this,
		NULL,
		"Select Fractal Prefs Directory",
		true);
	if (s)
	{
		LineEdit_FractPrefsDir->setText (s);
	}	
}


void GeneralPrefs::slotSelectPaletteDir()
{
	QString s = QFileDialog::getExistingDirectory(
		LineEdit_PaletteDir->text(),
		this,
		NULL,
		"Select Palette Directory",
		true);
	if (s)
	{
		LineEdit_PaletteDir->setText (s);
	}
}


void GeneralPrefs::slotSelectPaletteAtStart()
{
  QString s=LineEdit_PaletteAtStart->text();
  QFileInfo fi (s);
  if (!fi.isDir()) s=fi.dirPath();
		
  QString fileName = QFileDialog::getOpenFileName(s,0,this,0,"Select start palette");
  if (!fileName.isEmpty())
  {
	LineEdit_PaletteAtStart->setText (fileName);
  }
}


void GeneralPrefs::slotGetCurrentIsoSettings()
{
	kisomandel->globalPrefs.isoResolution=mainwin->getRes()-1;
	kisomandel->globalPrefs.isoHeight=(short)(mainwin->heightFactor*100.0);
	kisomandel->globalPrefs.autoHeight=mainwin->autoHeight;
	kisomandel->globalPrefs.isoZoom=(short)(mainwin->zoom*100.0);
	kisomandel->globalPrefs.isoSmoothing=fract->getSmoothLevel();
	kisomandel->globalPrefs.areaOutline=mainwin->areaOutline;
	kisomandel->globalPrefs.lineColor=QColor(mainwin->linecolor);
	kisomandel->globalPrefs.outlineColor=QColor(mainwin->outlinecolor);
	kisomandel->globalPrefs.maxiterColor=QColor(*mainwin->getMaxiterColor());

	initIsoValues();
}


void GeneralPrefs::slotSaveSettings()
{
	FILE *f;
	QString cur_path=QDir::currentDirPath();
	
	QDir d = QDir::home();
	QDir::setCurrent (d.path());	// write settings file into user's home directory
	f=fopen(".kisomandelrc","w");
	if (f)
	{
		QString s;
		
		fputs ("#Kisomandel general settings\n\n",f);
		fprintf (f,"PaletteDir          %s\n",(const char *)LineEdit_PaletteDir->text());
		fprintf (f,"FractPrefsDir       %s\n",(const char *)LineEdit_FractPrefsDir->text());
		fprintf (f,"LoadPaletteAtStart  %d\n",getChecked(CheckBox_LoadPaletteAtStart));
		fprintf (f,"PaletteAtStart      %s\n\n",(const char *)LineEdit_PaletteAtStart->text());
		
		fprintf (f,"IsoResolution       %d\n",Slider_IsoRes->value());
		fprintf (f,"IsoHeight           %d\n",Slider_IsoHeight->value());
		fprintf (f,"IsoAutoHeight       %d\n",getChecked(CheckBox_AutoHeight));
		fprintf (f,"IsoZoom             %d\n",Slider_IsoZoom->value());
		fprintf (f,"IsoSmoothing        %d\n\n",ComboBox_Smoothing->currentItem());

		char rgb[20];
		makeColorString (&kisomandel->globalPrefs.lineColor,rgb);
		fprintf (f,"LineColor           %s\n",rgb);
		makeColorString (&kisomandel->globalPrefs.outlineColor,rgb);
		fprintf (f,"OutlineColor        %s\n",rgb);
		makeColorString (&kisomandel->globalPrefs.maxiterColor,rgb);
		fprintf (f,"MaxiterColor        %s\n",rgb);
		fprintf (f,"AreaOutline         %d\n\n",getChecked(CheckBox_Outline));
				
		s="Mandelbrot";
		if (ComboBox_FractType->currentItem()==1) s="Julia";
		fprintf (f,"FractalType         %s\n",(const char *)s);
		
		fprintf (f,"ShowStatusBar       %d\n",getChecked(CheckBox_ShowStatusBar));
		fprintf (f,"ShowToolBar         %d\n",getChecked(CheckBox_ShowToolBar));
		fclose (f);
	}
	else
	{
		// Error
		QString s;
		s.sprintf (
			"Could not create settings file (.kisomandelrc)\n" \
			"in the user's home directory (%s)!",
			(const char *)d.path()
			);
		
		QMessageBox::critical (this,"Error",s);
	}

	QDir::setCurrent (cur_path);
	slotClose();
}


void GeneralPrefs::slotClose()
{
	// Remember actual settings in globalPrefs
	// (lineColor, outlineColor, maxiterColor are saved immidiately after selection)
	
	kisomandel->globalPrefs.paletteDir=LineEdit_PaletteDir->text();
   	kisomandel->globalPrefs.paletteFileAtStart=LineEdit_PaletteAtStart->text();
   	kisomandel->globalPrefs.loadPaletteAtStart=CheckBox_LoadPaletteAtStart->isChecked();
	kisomandel->globalPrefs.fractPrefsDir=LineEdit_FractPrefsDir->text();

	kisomandel->globalPrefs.isoResolution=Slider_IsoRes->value();
	kisomandel->globalPrefs.isoHeight=Slider_IsoHeight->value();
	kisomandel->globalPrefs.isoZoom=Slider_IsoZoom->value();
	kisomandel->globalPrefs.autoHeight=CheckBox_AutoHeight->isChecked();
	kisomandel->globalPrefs.isoSmoothing=ComboBox_Smoothing->currentItem();
	kisomandel->globalPrefs.areaOutline=CheckBox_Outline->isChecked();
	
	kisomandel->globalPrefs.showStatusBar=CheckBox_ShowStatusBar->isChecked();
	kisomandel->globalPrefs.showToolBar=CheckBox_ShowToolBar->isChecked();
	
	short b=ComboBox_FractType->currentItem();
	kisomandel->globalPrefs.fractalType= (b==0) ? Fract_Mandel : Fract_Julia ;

	mainwin->closeGeneralSettingsWindow();
}


void GeneralPrefs::keyPressEvent (QKeyEvent *e)
{
	if (e->key() == Key_Escape)
		slotClose();
	else
		mainwin->handleKeyCommand ( e->key(),e->state() );
}


void GeneralPrefs::closeEvent (QCloseEvent *e)
{
	slotClose();
}
