from pykludge3d import *

def cone( originX, originY, originZ, height, radius, sides, endcap ):
	'''creates a cone
	origin X,Y,Z specifies the location for the center of the cone's base
	height specifies the height
	radius specifies the radius of the cone's base
	sides specifies how many sides the cone has
	endcap determines if the cone should have an endcap (0 = no, 1 = yes)'''
	if sides < 3:
		return []
	topPoint = vert_insert( 0, 0, height )
	verts = []
	angle = 360.0 / sides
	for i in range( sides ):
		verts.append( vert_insert( radius, 0, 0 ) )
		rotate_verts( verts, 0, 0, -1.0 * angle )
	
	for i in range( len( verts ) ):
		poly_insert( [topPoint, verts[i], verts[(i+1)%len( verts )]] )
	
	if endcap:
		poly_insert( verts )
	
	verts.append( topPoint )
	translate_verts( verts, originX, originY, originZ )
	return verts
# end of cone


def cylinder( originX, originY, originZ, height, topRadius, bottomRadius, sides, endcaps ):
	'''creates a cylinder
	origin X,Y,Z specifies the location for the center of the cylinder's base
	height specifies the height
	topRadius specifies the radius of the cylinder's top end
	bottomRadius specifies the radius of the cylinder's bottom end
	sides specifies how many sides the cylinder has
	endcap determines if the cylinder should have endcaps (0 = no, 1 = yes)'''
	if sides < 3:
		return []
	topVerts = []
	bottomVerts = []
	angle = 360.0 / sides
	for i in range( sides ):
		topVerts.append( vert_insert( topRadius, 0, height ) )
		rotate_verts( topVerts, 0, 0, -1.0 * angle )
		bottomVerts.append( vert_insert( bottomRadius, 0, 0 ) )
		rotate_verts( bottomVerts, 0, 0, -1.0 * angle )
	
	for i in range( len( topVerts ) ):
		poly_insert( [bottomVerts[(i+1)%len( bottomVerts )], bottomVerts[i], topVerts[i], topVerts[(i+1)%len( topVerts )]] )
	
	topVerts.reverse()
	if endcaps:
		poly_insert( topVerts )
		poly_insert( bottomVerts )

	verts = topVerts + bottomVerts
	translate_verts( verts, originX, originY, originZ )
	return verts
# end of cylinder


def cube( minX, minY, minZ, maxX, maxY, maxZ ):
	'''creates a cuboid
	min X,Y,Z specifies the coordinate for the cube's corner closest to 
	the origin (0,0,0)
	max X,Y,Z specifies the coordinate for the cube's corner furthest from 
	the origin (0,0,0)'''
	# cubes don't lend themselves to iterative construction, as cones and 
	# cylinders do.
	# 
	# Arbitrary vertex naming:
	# v8 ------ v7  v4 ------ v3
	# |   Top   |   | Bottom  |
	# |         |   |		  |
	# v5 ------ v6  v1 ------ v2
	# 
	# v1 will be at <minX, minY, minZ>, v7 will be at <maxX, maxY, maxZ>
	# 
	v1 = vert_insert( minX, minY, minZ )
	v2 = vert_insert( maxX, minY, minZ )
	v3 = vert_insert( maxX, maxY, minZ )
	v4 = vert_insert( minX, maxY, minZ )
	v5 = vert_insert( minX, minY, maxZ )
	v6 = vert_insert( maxX, minY, maxZ )
	v7 = vert_insert( maxX, maxY, maxZ )
	v8 = vert_insert( minX, maxY, maxZ )
	
	poly_insert( [ v4, v3, v2, v1 ] ) # bottom
	poly_insert( [ v5, v6, v7, v8 ] ) # top
	poly_insert( [ v1, v2, v6, v5 ] ) # sides...
	poly_insert( [ v2, v3, v7, v6 ] )
	poly_insert( [ v3, v4, v8, v7 ] )
	poly_insert( [ v4, v1, v5, v8 ] )
	
	return [ v1, v2, v3, v4, v5, v6, v7, v8 ]
# end of cube


def sphere( originX, originY, originZ, radius, numSlices, numWedges ):
	'''creates a sphere
	origin X,Y,Z specifies the center of the sphere
	radius specifies the radius
	numSlices determines how many 'layers' will be used to make the sphere 
	(think 'wedding cake', or the latitude lines on a globe)
	numWedges determines how many 'wedges' will be used to make the sphere
	(think of an orange, or the longitude lines on a globe)'''
	if numSlices < 5 or numWedges < 5:
		return []
	vert = vert_insert( 0, 0, radius )
	slices = [[vert]]
	verts = [vert]
	sliceAngle = 180.0 / numSlices
	wedgeAngle = 360.0 / numWedges
	currSliceAngle = -90.0 + sliceAngle
	while currSliceAngle < 89.9:
		currSlice = []
		for j in range( numWedges ):
			vert = vert_insert( radius, 0, 0 )
			verts.append( vert )
			currSlice.append( vert )
			# first, rotate the vert to its slice's position
			rotate_verts( [vert], 0, currSliceAngle, 0 )
			# then, rotate the slice
			rotate_verts( currSlice, 0, 0, -1.0 * wedgeAngle )
		slices.append( currSlice )
		currSliceAngle += sliceAngle
	vert = vert_insert( 0, 0, -1.0 * radius )
	slices.append( [vert] )
	verts.append( vert )
	
	for j in range( len( slices ) - 1 ):
		currSlice = slices[j]
		nextSlice = slices[j+1]
		for i in range( max( len(currSlice), len(nextSlice) ) ):
			# it's ok to include the same vert twice (as would occur when 
			# building the top and bottom slices) in the list below, because 
			# poly_add_vertex will check to make sure that the same vertex 
			# does not get added to a poly more than once
			poly_insert( [currSlice[(i+1)%len( currSlice )], \
						  currSlice[i%len( currSlice )], \
						  nextSlice[i%len( nextSlice )], \
						  nextSlice[(i+1)%len( nextSlice )]] )
	
	translate_verts( verts, originX, originY, originZ )
	return verts
# end of sphere

register_function( __name__, cone )
register_function( __name__, cylinder )
register_function( __name__, cube )
register_function( __name__, sphere )
