/*
 * File:        doc_man.c
 * 
 * Description:	handles document switching, the 'Documents' menu, dialogs 
 *				 for loading/saving models, etc
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


#include <string.h>

#include "doc_man.h"
#include "undo.h"		/* for action_clear_stack(), etc */
#include "bottombar.h"
#include "model_load_save.h"
#include "documents.h"
#include "gui.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


/* STRUCTS ****************************************************************/

/* load/save dialog internals */
struct fmt_dialog_data {
	GtkFileSelection *selector;
	gint fmt;
	gint active : 1;
	gint cancelled : 1;
	Model *save_me;
};


/* PROTOTYPES *************************************************************/

gint fmt_overwrite_dlg( void );

void fmt_file_open_set_fmt(GtkWidget *widget, gint fmt_idx);
void fmt_file_open_ok(GtkWidget *widget, struct fmt_dialog_data *fdd);
void fmt_file_open_cancel(GtkWidget *widget, struct fmt_dialog_data *fdd);
void fmt_file_open_cancel_2(GtkWidget *widget, GdkEvent *event,
					struct fmt_dialog_data *fdd) {
	fmt_file_open_cancel(widget, fdd);
}

void fmt_file_save_set_fmt(GtkWidget *widget, gint fmt_idx);
void fmt_file_save_ok(GtkWidget *widget, struct fmt_dialog_data *fdd);
void fmt_file_save_cancel(GtkWidget *widget, struct fmt_dialog_data *fdd);
void fmt_file_save_cancel_2(GtkWidget *widget, GdkEvent *event,
					struct fmt_dialog_data *fdd) {
	fmt_file_save_cancel(widget, fdd);
}

gint fmt_load_dialog_available( void );
gint fmt_save_dialog_available( void );

void doc_man_add_model( Model *m ) ;
void doc_man_remove_model( Model *m ) ;
int doc_man_add_model_cb( gpointer no, Model *m, gpointer data ) ;
int doc_man_remove_model_cb( gpointer no, Model *m, gpointer data ) ;
int doc_man_switched_model_cb( gpointer no, Model *m, gpointer data ) ;
void doc_man_menutoggle_cb( 
		GtkCheckMenuItem *checkmenuitem, gpointer data );


/* FILE-SCOPE VARS ********************************************************/

struct fmt_dialog_data open_fdd[] = { { NULL, -1, FALSE, FALSE, NULL } };
struct fmt_dialog_data save_fdd[] = { { NULL, -1, FALSE, FALSE, NULL } };

GtkWidget *docmenu = NULL;
GSList *docmenu_radiobtn_group = NULL;
GHashTable *docmenu_hashtable = NULL;

/* FUNCS ******************************************************************/

/* LOAD DIALOG STUFF ****************************************************/

gint fmt_show_load_dialog(gchar *box_title)
{
	const gchar *filename;
	int result = FALSE;
	Model *new_model = NULL;

	if(open_fdd->selector == NULL)
	{
		int i = 0;
		GtkWidget *widget, *hbox, *menu, *mi;

		open_fdd->selector = (GtkFileSelection*) gtk_file_selection_new(NULL);

		g_signal_connect(G_OBJECT(open_fdd->selector), "delete_event",
				 (GtkSignalFunc)fmt_file_open_cancel_2, open_fdd);
		g_signal_connect(G_OBJECT(open_fdd->selector->ok_button),
			"clicked", (GtkSignalFunc)fmt_file_open_ok, open_fdd);
		g_signal_connect(G_OBJECT(open_fdd->selector->cancel_button),
			 "clicked", (GtkSignalFunc)fmt_file_open_cancel, open_fdd);

		/* now build the menu */
		menu = gtk_menu_new();
		mi = gtk_menu_item_new_with_label("Use Extension");
		g_signal_connect(G_OBJECT(mi), "activate",
					 G_CALLBACK(fmt_file_open_set_fmt), (gpointer) -1);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
		gtk_widget_show(mi);

		mi = gtk_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
		gtk_widget_set_sensitive(mi, FALSE);
		gtk_widget_show(mi);

		for(i = 0; fmt_list[i] != NULL; i++)
		{
			if(fmt_list[i]->load_file == NULL)
				continue;
			mi = gtk_menu_item_new_with_label(fmt_list[i]->descr);
			g_signal_connect(G_OBJECT(mi), "activate",
						G_CALLBACK(fmt_file_open_set_fmt), (gpointer) i);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
			gtk_widget_show(mi);
		}

		/* and the selector box infrastructure */
		hbox = gtk_hbox_new(FALSE, 0);
		widget = gtk_label_new("File Type:");
		gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, FALSE, 0);
		gtk_widget_show(widget);
		widget = gtk_option_menu_new();
		gtk_option_menu_set_menu(GTK_OPTION_MENU(widget), menu);
		gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, FALSE, 0);
		gtk_widget_show(widget);

		gtk_box_pack_end(GTK_BOX(GTK_FILE_SELECTION(open_fdd->selector)->main_vbox),
				 hbox, FALSE, FALSE, 0);
		gtk_widget_show(hbox);
	}

	if(open_fdd->active)
		return FALSE;

	open_fdd->active = TRUE;

	if(box_title == NULL)
		gtk_window_set_title(GTK_WINDOW(open_fdd->selector), "Open File");
	else
		gtk_window_set_title(GTK_WINDOW(open_fdd->selector), box_title);

	gtk_widget_show(GTK_WIDGET(open_fdd->selector));

	/* file open dialog box loop */
	while(1) {
		gtk_main();

		/* examine the structure */
		if(open_fdd->cancelled)
			break;

		filename = gtk_file_selection_get_filename(open_fdd->selector);
		gtk_widget_set_sensitive(GTK_WIDGET(open_fdd->selector), FALSE);

		new_model = model_load(filename, open_fdd->fmt);

		if( new_model == NULL ) {
			bb_push_message_f( 3.0, "Failed to load file %s", filename );
		} else {
			docs_add_model( new_model );
			docs_switch_model( new_model );
			result = TRUE;

			bb_push_message_f( 2.5, "Loaded file %s", filename );
			break;
		}

		gtk_widget_set_sensitive(GTK_WIDGET(open_fdd->selector), TRUE);
	}

	gtk_widget_hide(GTK_WIDGET(open_fdd->selector));
	gtk_widget_set_sensitive(GTK_WIDGET(open_fdd->selector), TRUE);
	open_fdd->active = FALSE;

//fixme - remove these, not needed
	g_signal_emit_by_name( notificationObj, 
		"notify::model-structure-changed", NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-texturelist-changed", NULL );
	
	return result;
}

void fmt_file_open_ok(GtkWidget *widget, struct fmt_dialog_data *fdd)
{
	fdd->cancelled = FALSE;
	gtk_main_quit();
}

void fmt_file_open_cancel(GtkWidget *widget, struct fmt_dialog_data *fdd)
{
	fdd->cancelled = TRUE;
	gtk_main_quit();
}

void fmt_file_open_set_fmt(GtkWidget *widget, gint fmt_idx)
{
	open_fdd->fmt = fmt_idx;
}

gint fmt_load_dialog_available( void )
{
	if(open_fdd->active == FALSE)
		return TRUE;

	return FALSE;
}


/* SAVE DIALOG STUFF ****************************************************/

gint fmt_show_save_dialog( Model *doc, gchar *box_title ) {
	int file_exists;
	if( doc->fname )
		if( model_save( doc, doc->fname, TRUE, &file_exists ) == FALSE ) {
			bb_push_message_f( 3.0, "Failed to save to file %s", doc->fname );
		} else {
			bb_push_message_f( 2.5, "Saved to file %s", doc->fname );
		}
	else
		fmt_show_saveAs_dialog( doc, box_title );
	return TRUE;
}


gint fmt_show_saveAs_dialog(Model *doc, gchar *box_title)
{
	const gchar *filename;
	gint ret = FALSE;

	if(save_fdd->selector == NULL)
	{
		int i = 0;
		GtkWidget *widget, *hbox, *menu, *mi;

		save_fdd->selector = (GtkFileSelection*) gtk_file_selection_new(NULL);

		g_signal_connect(G_OBJECT(save_fdd->selector), "delete_event",
							(GtkSignalFunc)fmt_file_save_cancel_2, save_fdd);
		g_signal_connect(G_OBJECT(save_fdd->selector->ok_button),
			"clicked", (GtkSignalFunc)fmt_file_save_ok, save_fdd);
		g_signal_connect(G_OBJECT(save_fdd->selector->cancel_button),
			"clicked", (GtkSignalFunc)fmt_file_save_cancel, save_fdd);

		/* now build the menu */
		menu = gtk_menu_new();
		mi = gtk_menu_item_new_with_label("Use Extension");
		g_signal_connect(G_OBJECT(mi), "activate",
					 G_CALLBACK(fmt_file_save_set_fmt), (gpointer) -1);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
		gtk_widget_show(mi);

		mi = gtk_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
		gtk_widget_set_sensitive(mi, FALSE);
		gtk_widget_show(mi);

		for(i = 0; fmt_list[i] != NULL; i++)
		{
			if(fmt_list[i]->save_file == NULL)
				continue;
			mi = gtk_menu_item_new_with_label(fmt_list[i]->descr);
			g_signal_connect(G_OBJECT(mi), "activate",
							G_CALLBACK(fmt_file_save_set_fmt), (gpointer) i);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
			gtk_widget_show(mi);
		}

		/* and the selector box infrastructure */
		hbox = gtk_hbox_new(FALSE, 0);
		widget = gtk_label_new("File Type:");
		gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, FALSE, 0);
		gtk_widget_show(widget);
		widget = gtk_option_menu_new();
		gtk_option_menu_set_menu(GTK_OPTION_MENU(widget), menu);
		gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, FALSE, 0);
		gtk_widget_show(widget);

		gtk_box_pack_end(GTK_BOX(GTK_FILE_SELECTION(save_fdd->selector)->main_vbox),
				 hbox, FALSE, FALSE, 0);
		gtk_widget_show(hbox);
	}

	if(save_fdd->active)
		return FALSE;

	save_fdd->active = TRUE;
	save_fdd->save_me = doc;

	if(box_title == NULL)
		gtk_window_set_title(GTK_WINDOW(save_fdd->selector), "Save File");
	else
		gtk_window_set_title(GTK_WINDOW(save_fdd->selector), box_title);

//	view_set_sensitive_docwide(save_fdd->save_me, FALSE);
	gtk_widget_show(GTK_WIDGET(save_fdd->selector));

	while(1) {
		int file_exists = FALSE;
		gtk_main();

		if(save_fdd->cancelled)
			break;

		filename = gtk_file_selection_get_filename(save_fdd->selector);

		/* adapt to a potential change in save format */
		/* the idea is that if this function gets called, the user wants */
		/* to set the misc data over again anyway */
		model_ls_clean_miscdata(save_fdd->save_me);
		save_fdd->save_me->fmt_idx = save_fdd->fmt;

		if(model_save(save_fdd->save_me, filename, FALSE, &file_exists) == FALSE) {
			/* if the file exists, ask the user if it is ok to overwrite, 
			then try to save again */
			if( file_exists ) {
				if( fmt_overwrite_dlg() == FALSE ) {
					bb_push_message_f( 3.0, "Canceled save: file already exists" );
				} else {
					if( model_save( save_fdd->save_me, filename, TRUE, 
									&file_exists) == FALSE) 
					{
						/* save still didn't succeed */
						bb_push_message_f( 3.0, "Failed to save to file %s", filename );
					}
				}
			} else {
				bb_push_message_f( 3.0, "Failed to save to file %s", filename );
			}
		} else {
			ret = TRUE;
			bb_push_message_f( 2.5, "Saved to file %s", filename );
			break;
		}

		gtk_widget_set_sensitive(GTK_WIDGET(save_fdd->selector), TRUE);
	}

	gtk_widget_hide(GTK_WIDGET(save_fdd->selector));
	gtk_widget_set_sensitive(GTK_WIDGET(save_fdd->selector), TRUE);
//	view_set_sensitive_docwide(save_fdd->save_me, TRUE);
	save_fdd->active = FALSE;

	return ret;
}

void fmt_file_save_ok(GtkWidget *widget, struct fmt_dialog_data *fdd)
{
	fdd->cancelled = FALSE;
	gtk_main_quit();
}

void fmt_file_save_cancel(GtkWidget *widget, struct fmt_dialog_data *fdd)
{
	fdd->cancelled = TRUE;
	gtk_main_quit();
}

void fmt_file_save_set_fmt(GtkWidget *widget, gint fmt_idx)
{
	save_fdd->fmt = fmt_idx;
}

gint fmt_save_dialog_available( void )
{
	if(save_fdd->active == FALSE)
		return TRUE;

	return FALSE;
}


/* OVERWRITE DIALOG STUFF *************************************************/

gint fmt_overwrite_dlg( void )
{
	gint decision = FALSE;
	GtkWidget *dialog;
	GtkWidget *label;
	gint dlg_response;

	dialog = gtk_dialog_new_with_buttons ("Confirm",
										  GTK_WINDOW( TopLevelWindow ),
										  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
										  GTK_STOCK_YES,
										  GTK_RESPONSE_OK,
										  GTK_STOCK_NO,
										  GTK_RESPONSE_CANCEL,
										  NULL);
	
	label = gtk_label_new("File Exists.  Overwrite?");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
	
	/* set up the OK button as the default widget */
	gtk_dialog_set_default_response( GTK_DIALOG(dialog), GTK_RESPONSE_OK );
	
	/* shows the dialog, and since it's modal, prevents rest of 
	   kludge3d from doing stuff */
	gtk_widget_show_all (dialog);
	
	/* runs a gtkmain until the user clicks on something */
	dlg_response = gtk_dialog_run( GTK_DIALOG (dialog) );
	
	switch( dlg_response ) {
	case GTK_RESPONSE_OK:
		decision = TRUE;
		break;
	default:
		decision = FALSE;
		break;
	}
	
	gtk_widget_destroy( dialog );

	return decision;
}


/* DOCUMENTS MENU *********************************************************/

void doc_man_create_menu( GtkWidget *parent ) {
	GtkWidget *menuitem;
	GSList *l;
	
	docmenu = parent;
	
	menuitem = gtk_tearoff_menu_item_new();
	gtk_menu_shell_append( GTK_MENU_SHELL( docmenu ), menuitem );
	gtk_widget_show( menuitem );
	
	for( l = models; l; l = l->next ) {
		doc_man_add_model( (Model*)l->data );
	}
	
	/* listen for added-model signal */
	g_signal_connect( notificationObj, "added-model", 
					G_CALLBACK(doc_man_add_model_cb), NULL );
	/* listen for removed-model signal */
	g_signal_connect( notificationObj, "removed-model", 
					G_CALLBACK(doc_man_remove_model_cb), NULL );
	/* listen for switched-model signal */
	g_signal_connect( notificationObj, "switched-model", 
					G_CALLBACK(doc_man_switched_model_cb), NULL );
}


void doc_man_add_model( Model *m ) {
	GtkWidget *menuitem;
	
	menuitem = gtk_radio_menu_item_new_with_label( 
					docmenu_radiobtn_group, m->name );
	docmenu_radiobtn_group = 
		gtk_radio_menu_item_get_group( GTK_RADIO_MENU_ITEM( menuitem ) );
	gtk_menu_shell_append( GTK_MENU_SHELL( docmenu ), menuitem );
	gtk_widget_show( menuitem );

	g_signal_connect( menuitem, "activate", 
					G_CALLBACK(doc_man_menutoggle_cb), m );
	
	if( docmenu_hashtable == NULL )
		docmenu_hashtable = g_hash_table_new( NULL, NULL );
	g_hash_table_insert( docmenu_hashtable, m, menuitem );
}


void doc_man_remove_model( Model *m ) {
	GtkWidget *menuitem;
	
	menuitem = g_hash_table_lookup( docmenu_hashtable, m );
	if( menuitem )
		gtk_widget_destroy( menuitem );
	
	/* update the docmenu_radiobtn_group list */
	if( models == NULL ) {
		docmenu_radiobtn_group = NULL;
	} else {
		/* use the first model to look up one of the menu items 
		(doesn't matter which) */
		menuitem = g_hash_table_lookup( docmenu_hashtable, 
										(Model*)models->data );
		docmenu_radiobtn_group = 
			gtk_radio_menu_item_get_group( GTK_RADIO_MENU_ITEM( menuitem ) );
	}
	
	/* update the hash table */
	g_hash_table_remove( docmenu_hashtable, m );
}


int doc_man_add_model_cb( gpointer no, Model *m, gpointer data ) {
	
	doc_man_add_model( m );
	
	return FALSE;
}

int doc_man_remove_model_cb( gpointer no, Model *m, gpointer data ) {
	
	doc_man_remove_model( m );
	
	return FALSE;
}

int doc_man_switched_model_cb( gpointer no, Model *m, gpointer data ) {
	GtkWidget *menuitem;
	
	/* first, look up m in the table */
	menuitem = g_hash_table_lookup( docmenu_hashtable, m );
	if( !menuitem )
		return FALSE;
	
	/* now make this radiobutton the selected one */
	if( !gtk_check_menu_item_get_active( GTK_CHECK_MENU_ITEM(menuitem) ) ) {
		gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(menuitem), TRUE );
	}
	
	return FALSE;
}

void doc_man_menutoggle_cb( 
		GtkCheckMenuItem *checkmenuitem, gpointer data )
{

	/* bail if this isn't the newly-activated item */
	if( ! checkmenuitem->active )
		return;

	docs_switch_model( (Model *)data );
}


