/*
 * ME3D 3-D Modeler Program
 * Copyright (C) 1998 Sam Revitch
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* file utility functions */

/* (this program was written with basic portability in mind) */

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <stdarg.h>

#include "file_utils.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


gint fu_pointers_equal(gconstpointer p1, gconstpointer p2);

gint fu_pointers_equal(gconstpointer p1, gconstpointer p2)
{ if(p1 == p2) return TRUE; return FALSE; }


struct fu_file *fu_open(const gchar *name, gint create)
{
  struct fu_file *file;
  int fd;

  g_return_val_if_fail(name != NULL, NULL);

  if(create)
    fd = open(name, O_RDWR | O_CREAT, 0644);
  else
    fd = open(name, O_RDONLY);

  if(fd == -1)
    return NULL;

  file = (struct fu_file*) malloc(sizeof(struct fu_file));
  file->fd = fd;


  if(create)
    file->fp = fdopen( fd, "a+" );
  else
    file->fp = fdopen( fd, "r" );

  if( file->fp == NULL ) {
      printf( "%s: fdopen failed.  errno is %i\n", __FUNCTION__, errno );
      if( errno == EINVAL )
          printf( "\terrno is EINVAL - invalid 'mode' arg\n" );
  }


  file->len_ht = g_hash_table_new(g_direct_hash, fu_pointers_equal);
  return file;
}

void fu_close(struct fu_file *file)
{
  g_return_if_fail(file != NULL);

  fflush( file->fp );
  close(file->fd);

  g_hash_table_destroy(file->len_ht);
  free(file);
}

guint fu_file_size(struct fu_file *file)
{
  struct stat fu_stat;

  g_return_val_if_fail(file != NULL, 0);
  fstat(file->fd, &fu_stat);
  return fu_stat.st_size;
}

guchar *fu_map_block(struct fu_file *file, guint start, guint len)
{
  guchar *r;

  g_return_val_if_fail(file != NULL, NULL);
  r = (guchar*) mmap(NULL, (size_t)len, PROT_READ,
		     MAP_PRIVATE, file->fd, (off_t)start);
  g_return_val_if_fail(r != (guchar*) -1, NULL);

  g_hash_table_insert(file->len_ht, r, (gpointer)len);
  return r;
}

void fu_unmap_block(struct fu_file *file, guchar *block)
{
  gpointer tmp;
  guint len;

  g_return_if_fail(file != NULL);

  /* just doing some casting, to silence compiler warnings */
  tmp = g_hash_table_lookup(file->len_ht, block);
  len = (guint)tmp;
  g_return_if_fail(len != 0);

  g_hash_table_remove(file->len_ht, block);
  munmap(block, len);
}

void fu_write_block(struct fu_file *file, guchar *block, guint len)
{
  g_return_if_fail(file != NULL);

  write(file->fd, block, (size_t)len);
}

void fu_truncate(struct fu_file *file, guint size)
{
  g_return_if_fail(file != NULL);

  ftruncate(file->fd, size);
}

void fu_seek(struct fu_file *file, gint pos)
{
  g_return_if_fail(file != NULL);

  lseek(file->fd, (off_t) pos, SEEK_SET);
}

gint fu_gets(struct fu_file *file, gchar *buf, guint len)
{
  guint pos = 0;
  off_t offset;

  g_return_val_if_fail(file != NULL, 0);
  g_return_val_if_fail(buf != NULL, 0);
  g_return_val_if_fail(len > 0, 0);

  offset = lseek(file->fd, 0, SEEK_CUR);

  /* this is a cool yet inefficient one */
  pos = read(file->fd, buf, len);
  if(pos <= 0)
    return -1;

  if(pos < len)
    len = pos;

  pos = 0;
  while(pos < len)
  {
    if(buf[pos] == '\n') {
      buf[pos] = '\0';
      lseek(file->fd, (off_t) (offset + pos + 1), SEEK_SET);
      return pos;
    }
    pos++;
  }

  buf[len - 1] = '\0';
  return (len - 1);
}

gint fu_puts(struct fu_file *file, gchar *src)
{
  gint len;

  g_return_val_if_fail(file != NULL, 0);

  len = strlen(src);
  return (gint) write(file->fd, src, len);
}

gint fu_printf(struct fu_file *file, gchar *fmt, ...)
{
  va_list ap;
  gchar buf[1024];		/* bigger is better */
  gint length;

  g_return_val_if_fail(file != NULL, 0);

  va_start(ap, fmt);
  length = vsnprintf(buf, 1024, fmt, ap);

  return (gint) write(file->fd, buf, length);
}
